//______________________________________________________________________________

//	The Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader.gui;

import org.demo.webwader.WebSite;
import org.demo.webwader.SiteScanner;
import org.demo.webwader.WebNode;
import org.demo.webwader.Resources;
import org.ariane.tools.ToolBox;
import org.ariane.gui.TextField;

import java.io.*;
import java.util.StringTokenizer;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

/**
 * The window used to scan a web site.
 * <Br>
 * It displays controls which can be used to scan a site.
 *
 * @see SiteScanner
 * @version $Id: ScanPane.java,v 3.7 2001/01/28 11:33:01 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */
//______________________________________________________________________________

public	class ScanPane extends JPanel implements Runnable {
  /**
   * @serial The unique instance.
   */
private static ScanPane  scan_pane = null;
  /**
   * @serial The name field.
   */
private  JComboBox name_field;
  /**
   * @serial The index of the selected site.
   */
private  int site_index;
  /**
   * @serial The url field.
   */
private  TextField url_field;
    /**
     * @serial the button Stop.
     */
private JButton stop_button;
    /**
     * @serial the button Start.
     */
private JButton start_button;
    /**
     * @serial The button stop command.
     */
private	static final String STOP   = "Stop";
    /**
     * @serial The button start command.
     */
private	static final String START  = "Start";
  /**
   * @serial The current scanning order.
   */
private int scan_order = WebNode.BREADTH_FIRST;
    /**
     * @serial The current site being scanned.
     */
private	WebSite site = null;
    /**
     * @serial The list of tokens to decide if an URL should be ignored.
     */
private	String ignored;

//______________________________________________________________________________
/**
 * Gets an unique instance of the ScanPane.
 * <Br>
 * It is a singleton.
 * @return the unique instance.
 */
protected static ScanPane instance()
  {
    if(scan_pane == null) scan_pane = new ScanPane();

    return scan_pane;
  }
//______________________________________________________________________________
/**
 * Creates the Panel allowing the user to navigate thru a web site.
 */
private ScanPane()
  {
      super();

      GridBagLayout grid     = new GridBagLayout();
      GuiResources resources = GuiResources.instance();
      Color fg               = resources.getPaneForeground();

      int width  = resources.get("ScanPane.Width",  550);
      int height = resources.get("ScanPane.Height", 300);
      setMaximumSize(new Dimension(width, height));

      setLayout(grid);
      setBackground(resources.getPaneBackground());
      setForeground(fg);
      setFont(resources.getHeaderFont());

      GridBagConstraints cons = new GridBagConstraints();

      cons.weightx       = 0.0;
      cons.weighty       = 0.0;
      cons.insets.left   = resources.getSpace();
      cons.insets.right  = cons.insets.left;
      cons.insets.top    = 2 * cons.insets.left;
      cons.insets.bottom = cons.insets.left;

      /**
       * Updates fields when a new item is selected.
       */
      class SelectionListener implements ItemListener {
	  final public void itemStateChanged(ItemEvent ev) {
	      if(ev.getStateChange() == ItemEvent.SELECTED)
		  selectSite();
	  }
      }
      /**
       * Reacts to an action in the name field.
       */
      class ReturnListener implements ActionListener {
	  final public void actionPerformed(ActionEvent ev) {
	      updateSitesList();
	  }
      }

      JLabel item;

      item  =  new JLabel("Site Name : ");
      item.setFont(resources.getLabelFont());
      item.setForeground(fg);
      org.ariane.gui.ToolBox.fixComponentSize(item);

      cons.gridwidth    = 1;
      cons.anchor       = GridBagConstraints.WEST;
      cons.insets.right = 0;
      grid.setConstraints(item, cons);
      add(item);

      int ncol    = resources.get("NameField.Width", 48);
      name_field  = new JComboBox();
      name_field.setFont(resources.getTextFont());
      name_field.setBackground(resources.getBoardBackground());
      name_field.setForeground(resources.getBoardForeground());
      name_field.setEditable(true);
      name_field.setBorder(resources.getInBorder());
      name_field.setToolTipText("Press Return to store the new name");

      name_field.addItemListener(new SelectionListener());
      name_field.addActionListener(new ReturnListener());

      cons.gridwidth = 3;
      cons.fill	     = GridBagConstraints.HORIZONTAL;
      grid.setConstraints(name_field, cons);
      add(name_field);


      item  =  new JLabel("Site URL : ");
      item.setFont(resources.getLabelFont());
      item.setForeground(fg);
      org.ariane.gui.ToolBox.fixComponentSize(item);

      cons.anchor       = GridBagConstraints.WEST;
      cons.fill	        = GridBagConstraints.NONE;
      cons.gridwidth    = 1;
      cons.insets.right = 0;
      cons.gridy        = 1;
      grid.setConstraints(item, cons);
      add(item);

      /**
       * Reacts to an action in the url field.
       */
      class UrlListener implements ActionListener {
	  final public void actionPerformed(ActionEvent ev) {
	      updateSiteURL();
	  }
      }

      ncol       = resources.get("URLField.Width", 48);
      url_field  =  new TextField(ncol);
      url_field.setFont(resources.getTextFont());
      url_field.setBackground(resources.getBoardBackground());
      url_field.setForeground(resources.getBoardForeground());
      url_field.setEditable(true);
      url_field.setText(null);
      url_field.addActionListener(new UrlListener());
      url_field.setToolTipText("Press Return to store the new URL");

      cons.gridwidth    = 3;
      grid.setConstraints(url_field, cons);
      add(url_field);

      Dimension bdim  = new Dimension(31, 31);
      Insets margin   = new Insets(0,0, 0,0);
      JButton button;
      /**
       * Process a Button pressed.
       */
      class ScanListener implements ActionListener {
	  final public void actionPerformed(ActionEvent ev) {
	      processAction(ev.getActionCommand());
	  }
      }
      ScanListener scan_listener = new ScanListener();

      button  =  new JButton(resources.getIcon("stop.gif"));
      button.setToolTipText("Stop the current scan");
      button.setMargin(margin);
      button.addActionListener(scan_listener);
      button.setActionCommand(STOP);
      button.setPreferredSize(bdim);

      cons.gridx        = 1;
      cons.gridy        = 2;
      cons.gridwidth    = 1;
      grid.setConstraints(button, cons);
      add(button);
      stop_button = button;

      button  =  new JButton(resources.getIcon("start.gif"));
      button.setToolTipText("Start a new scan");
      button.setMargin(margin);
      button.addActionListener(scan_listener);
      button.setActionCommand(START);
      button.setPreferredSize(bdim);

      cons.gridx         = 2;
      cons.insets.right  = 4 * cons.insets.bottom;
      grid.setConstraints(button, cons);
      add(button);
      start_button = button;

      setSitesList();
      setIgnoredList(Resources.instance().get(Resources.PREFIX + "Ignored"));
      updateButtonStatus(STOP);
  }
//______________________________________________________________________________
/**
 * Processes the command launched by a button.
 * <br>
 * The site object is created, then it is scanned in a thread.
 * @param cmd the command issued by the button.
 * @see #createSite
 * @see #run
 */
final private void processAction(String cmd)
{
    updateButtonStatus(cmd);

    if(cmd.equals(START)) {

	createSite();
	Thread thread = new Thread(this);
	thread.start();
    }
    else { // STOP
	killScan();
    }
}
//______________________________________________________________________________
/**
 * Starts the scanning of the site.
 * <Br>
 * The console is cleared, the site is scanned. The result is saved.
 * @see SiteScanner#scan
 */
public void run()
{
    try {
	String url = url_field.getText();
	if(url == null || url.length() < 2) {
	    ErrorWindow.instance().display("Invalid null URL !");
	}
	else {
	    ScanWindow.instance().clearConsole();
	    WebNode top =
	    SiteScanner.instance().scan(site.getName(), url, scan_order);
	    site.setTopNode(top);
	}
    }
    catch(Exception ex) {
	updateButtonStatus(STOP);
	if(ToolBox.debug) {
	    ToolBox.warn("Can't scan site");
	    ex.printStackTrace();
	}
	site = null;
	ErrorWindow.instance().display("Can't scan site :\n" + ex.getMessage());
	return;
    }

    backupWebSite();
    updateButtonStatus(STOP);
}
//______________________________________________________________________________
/**
 * Kills the running scan if it exists.
 * @see SiteScanner#giveUpScan
 */
final protected void killScan()
{
    if(site != null) SiteScanner.instance().giveUpScan();
    updateButtonStatus(STOP);
}
//______________________________________________________________________________
/**
 * Saves the description of the site in a temporary file.
 * <P>
 * The temporary file should be "sitename.site~" in the current directory.
 * @see WebSite#write()
 */
final private void backupWebSite()
{
    if(site == null) return;
    try {
	site.write();
    }
    catch(Exception ex) {
	ToolBox.warn("Can't backup site", ex);
	if(ToolBox.debug) ex.printStackTrace();
	ErrorWindow.instance().display("Can't backup site !");
    }
}
//______________________________________________________________________________
/**
 * Creates a new site and set it current.
 * @see WebSite
 */
final private void createSite()
{
    site = new WebSite(getNameField());

    SiteScanner.instance().setPrinter(
	        ScanWindow.instance().getConsolePrinter());
}
//______________________________________________________________________________
/**
 * Updates the status of the buttons.
 * <Br>
 * @param cmd the command issued by the button.
 */
final private void updateButtonStatus(String cmd)
{
    /**
     * Update the status of the buttons.
     * When Start is pressed, only Stop is kept sensitive.
     */
    if(cmd.equals(START)) {
	start_button.setEnabled(false);
	stop_button.setEnabled(true);
    }
    else { // STOP
	start_button.setEnabled(true);
	stop_button.setEnabled(false);
    }
}
//______________________________________________________________________________
/**
 * Gets the content of the name field.
 * @return the content of the field.
 */
final private String getNameField()
{
    return name_field.getSelectedItem().toString();
}
//______________________________________________________________________________
/**
 * Gets the current site.
 * @return the current site.
 */
final protected WebSite getWebSite()
{
    return site;
}
//______________________________________________________________________________
/**
 * Gets the scanning order.
 * @return the current order.
 */
final protected int getScanningOrder()
{
    return scan_order;
}
//______________________________________________________________________________
/**
 * Sets the scanning order.
 * @param the new order.
 */
final protected void setScanningOrder(int scan_order)
{
    this.scan_order = scan_order;
}
//______________________________________________________________________________
/**
 * Gets the list of ignored tokens.
 * <Br>
 * @return the current list.
 */
final protected String getIgnoredList()
{
    return ignored;
}
//______________________________________________________________________________
/**
 * Sets the list of ignored tokens.
 * <Br>
 * @param the new list.
 */
final protected void setIgnoredList(String ignored)
{
    this.ignored = ignored;
}
//______________________________________________________________________________
/**
 * Initializes the list of sites managed by the combo box.
 * <P>
 * The list is read in the resources : it consists of couples
 * name, url. The list is parsed, and the couples are put in
 * the combo box.
 */
final private void setSitesList()
{
    String list = GuiResources.instance().get("SitesList");
    if(list == null) return;

    StringTokenizer stk = new StringTokenizer(list);

    while(stk.hasMoreTokens()) {

	String name = stk.nextToken();
	if(! stk.hasMoreTokens()) {
	    ToolBox.warn("Malformed list of sites for " + name);
	    return;
	}

	SiteRef ref = new SiteRef(name, stk.nextToken());
	name_field.addItem(ref);
    }

    if(name_field.getItemCount() > 0) name_field.setSelectedIndex(0);
}
//______________________________________________________________________________
/**
 * Gets the list of sites known by the combo box.
 * <br>
 * The number of couples is limited to 16.
 * @return the list of couple (name, url).
 * @see OptionPane#getUserProperties
 */
final protected String getSitesList()
{
    int n = name_field.getItemCount();
    if(n > 16) n = 16;

    StringBuffer buf = new StringBuffer();

    for(int i = 0; i < n; i++) {
	SiteRef ref = (SiteRef)name_field.getItemAt(i);
	buf.append(ref.getReference()).append(' ');
    }

    return buf.toString();
}
//______________________________________________________________________________
/**
 * Does what is needed when a new name is selected.
 * <P>
 * Updates the url field with the url of the selected site.
 * Keeps track of the selected index.
 * @see SiteRef
 */
final private void selectSite()
{
    Object obj = name_field.getSelectedItem();
    if(! (obj instanceof SiteRef)) {
	return;
    }

    site_index = name_field.getSelectedIndex();
    url_field.setText(((SiteRef)(name_field.getItemAt(site_index))).getURL());  
}
//______________________________________________________________________________
/**
 * Updates the url of the selected site from the content of the field.
 * @see SiteRef
 */
final private void updateSiteURL()
{
    ((SiteRef)(name_field.getItemAt(site_index))).setURL(url_field.getText());  
}
//______________________________________________________________________________
/**
 * Updates the url of the selected site from the content of the field.
 * <P>
 * If a carriage return was entered the selected item is a plain string.
 * If it is not the case, the method returns immediatly.
 * If the length of the string is 0, the reference which was at this
 * position is removed. If the string has not changed nothing happens.
 * Otherwise a new reference is created and inserted in the list.
 * @see SiteRef
 */
final private void updateSitesList()
{
    Object obj = name_field.getSelectedItem();
    if(! (obj instanceof String)) {
	return;
    }
    
    name_field.removeItem(obj);

    String str = (String)obj;
    str.trim(); // removes white spaces

    if(str.length() < 1) {
	if(name_field.getItemCount() > 0)
	    name_field.removeItemAt(site_index);
	url_field.setText(null);
	return;
    }
    else if(name_field.getItemCount() > 0 &&
	    str.equals(name_field.getItemAt(site_index).toString())) {
	return;
    }

    SiteRef ref = new SiteRef(str, url_field.getText());
    name_field.addItem(ref);
    name_field.setSelectedItem(ref);
}
//______________________________________________________________________________
}
