//______________________________________________________________________________

//	The Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader.gui;

import org.demo.webwader.WebNode;
import org.demo.webwader.Resources;
import org.ariane.tools.ToolBox;

import java.util.Properties;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.io.*;

/**
 * The customized OptionPane.
 * <Br>
 * It presents a tabbed pane displaying the various settings and
 * allowing changes.
 *
 * @version $Id: OptionPane.java,v 3.10 2001/01/13 18:33:56 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */
//______________________________________________________________________________

class OptionPane extends Dialog {
  /**
   * @serial The unique instance.
   */
private  static OptionPane option_pane = null;
  /**
   * @serial The associated Navigation Pane.
   */
private  NavePane nave_pane;
  /**
   * @serial The associated Scanning Pane.
   */
private  ScanPane scan_pane;
  /**
   * @serial The temporary local flag.
   */
private boolean verbose = true;
  /**
   * @serial The temporary local flag.
   */
private boolean debug = false;
  /**
   * @serial A mode button.
   */
private JRadioButton quiet_button;
  /**
   * @serial A mode button.
   */
private JRadioButton verbose_button;
  /**
   * @serial A mode button.
   */
private JRadioButton debug_button;
  /**
   * @serial The navigation order combo box.
   */
private JComboBox nave_combo;
  /**
   * @serial The temporary local navigation order.
   */
private int nave_order;
  /**
   * @serial The displayer combo box.
   */
private JComboBox displayer_combo;
  /**
   * @serial The temporary local displayer.
   */
private int displayer = 0;
  /**
   * @serial The display time field.
   */
private  JTextField pause_field;
  /**
   * @serial The list of suffix to display field.
   */
private  JTextField suffix_field;
  /**
   * @serial The scanning order combo box.
   */
private JComboBox scan_combo;
  /**
   * @serial The temporary local scanning order.
   */
private int scan_order;
  /**
   * @serial The list of ignored files field.
   */
private  JTextField ignored_field;
  /**
   * @serial The flag giving the status of the backup.
   */
private boolean backup_done = false;

//______________________________________________________________________________
/**
 * Gets an unique instance of the option_pane.
 * <br>
 * It is a singleton.
 * @return the unique instance.
 */
protected static OptionPane instance()
  {
    if(option_pane == null) option_pane = new OptionPane();

    return option_pane;
  }
//______________________________________________________________________________
/**
 * Creates the OptionPane.
 */
private OptionPane()
  {
      super();
  }
//______________________________________________________________________________
/**
 * Builds the main component.
 * @return the tabbed pane.
 */
protected JComponent createComponent()
{
      scan_pane = ScanPane.instance();
      nave_pane = NavePane.instance();

      JTabbedPane pane = new JTabbedPane();

      pane.setBackground(resources.getPaneBackground());
      pane.setForeground(resources.getPaneForeground());

      pane.add("Navigation", createNaveOptionsPane());
      pane.add("Scanning",   createScanOptionsPane());
      pane.add("Options",    createMiscOptionsPane());

      Color bg = resources.getPaneBackground().darker();
      pane.setBackgroundAt(0, bg);
      pane.setBackgroundAt(1, bg);

      setInitialSize("Options", 300, 200);

      return pane;
}
//______________________________________________________________________________
/**
 * Builds the pane showing the options.
 * <Br>
 * The verbosity mode can be selected.
 * @return the pane.
 */
private JComponent createMiscOptionsPane()
{
      JPanel pane     = new JPanel();
      Color  fg       = resources.getPaneForeground();
      Component space = Box.createHorizontalStrut(resources.getSpace());

      pane.setLayout(new BoxLayout(pane, BoxLayout.X_AXIS));
      pane.setOpaque(true);
      pane.setBackground(resources.getPaneBackground());
      pane.setForeground(fg);
      pane.setBorder(resources.getMarginBorder());

      JLabel header = new JLabel("Verbosity : ");
      header.setAlignmentX(Component.LEFT_ALIGNMENT);
      header.setFont(resources.getLabelFont());
      header.setHorizontalAlignment(JLabel.LEFT);
      header.setForeground(fg);
      header.setToolTipText("To select one of quiet, verbose, debug");
      pane.add(header);
      pane.add(space);

      ButtonGroup group  = new ButtonGroup();

      /**
       * Define what to change when a button is selected.
       * The mode is set to one of quiet, verbose or debug.
       * Actually, only local variables are changed. The
       * global ones are updated if the OK button is pressed.
       * Note that there is only one global OK.
       */
      class VerbosityListener implements ActionListener {
	  final public void actionPerformed(ActionEvent ev) {
	      int c = ev.getActionCommand().charAt(0);
	      switch(c) {
	      case 'Q' :	// Quiet
		 verbose = false; 
		 debug   = false; 
		 break;

	      case 'D' :	// Debug
		 verbose = true; 
		 debug   = true; 
		 break;

	      default :		// Verbose
		 verbose = true; 
		 debug   = false; 
		 break;
	      }
	  }
      }
      VerbosityListener listener = new VerbosityListener();

      quiet_button   = new JRadioButton("Quiet");
      if(! (ToolBox.verbose || ToolBox.debug)) quiet_button.setSelected(true);
      quiet_button.addActionListener(listener);
      quiet_button.setFont(resources.getTextFont());
      quiet_button.setOpaque(false);
      quiet_button.setForeground(fg);
      group.add(quiet_button);
      pane.add(quiet_button);

      verbose_button = new JRadioButton("Verbose");
      if(ToolBox.verbose && ! ToolBox.debug) verbose_button.setSelected(true);
      verbose_button.addActionListener(listener);
      verbose_button.setFont(resources.getTextFont());
      verbose_button.setOpaque(false);
      verbose_button.setForeground(fg);
      group.add(verbose_button);
      pane.add(verbose_button);

      debug_button   = new JRadioButton("Debug");
      if(ToolBox.debug) debug_button.setSelected(true);
      debug_button.addActionListener(listener);
      debug_button.setFont(resources.getTextFont());
      debug_button.setOpaque(false);
      debug_button.setForeground(fg);
      group.add(debug_button);
      pane.add(debug_button);

      return pane;
}
//______________________________________________________________________________
/**
 * Builds the pane showing the navigation parameters.
 * @return the pane.
 */
private JComponent createNaveOptionsPane()
{
      JComponent pane = buildNaveOptionsPane();

      /**
       * Defines what to do when a new navigation is selected in the combo box.
       */
      class NaveOrderListener implements ActionListener {
	  final public void actionPerformed(ActionEvent ev) {
	      updateNavigationOrder(nave_combo.getSelectedIndex());
	  }
      }
      nave_combo.addActionListener(new NaveOrderListener());

      /**
       * Defines what to do when a new displayer is selected in the combo box.
       */
      class DisplayerListener implements ActionListener {
	  final public void actionPerformed(ActionEvent ev) {
	      updateDisplayer(displayer_combo.getSelectedIndex());
	  }
      }
      displayer_combo.addActionListener(new DisplayerListener());

      return pane;
}
//______________________________________________________________________________
/**
 * Builds the pane showing the scanning parameters.
 * @return the pane.
 */
private JComponent createScanOptionsPane()
{
      JComponent pane = buildScanOptionsPane();

      /**
       * Defines what to do when a new navigation is selected in the combo box.
       */
      class ScanOrderListener implements ActionListener {
	  final public void actionPerformed(ActionEvent ev) {
	      updateScanningOrder(scan_combo.getSelectedIndex());
	  }
      }
      scan_combo.addActionListener(new ScanOrderListener());

      return pane;
}
//______________________________________________________________________________
/**
 * Builds the pane showing navigation parameters.
 *
 * @return the pane.
 */
private JComponent buildNaveOptionsPane()
{
      JPanel pane        = new JPanel();
      GridBagLayout grid = new GridBagLayout();
      Color fg           = resources.getPaneForeground();
      pane.setLayout(grid);
      pane.setOpaque(true);
      pane.setBackground(resources.getPaneBackground());
      pane.setForeground(fg);

      GridBagConstraints cons = new GridBagConstraints();
      cons.weightx       = 0.0;
      cons.weighty       = 0.0;
      cons.anchor        = GridBagConstraints.CENTER;
      cons.insets.left   = resources.getSpace();
      cons.insets.right  = cons.insets.left;
      cons.insets.top    = 2 * cons.insets.left;
      cons.insets.bottom = cons.insets.left;


      JLabel header = new JLabel("Navigation");
      header.setFont(resources.getLabelFont());
      header.setHorizontalAlignment(JLabel.CENTER);
      header.setToolTipText("To change navigation settings");
      header.setForeground(fg);
      cons.gridwidth = GridBagConstraints.REMAINDER;
      grid.setConstraints(header, cons);
      pane.add(header);

      JLabel item  =  new JLabel("Displayer : ");
      item.setFont(resources.getLabelFont());
      item.setToolTipText("Tool used to display HTML pages");
      item.setForeground(fg);
      cons.gridwidth  = 1;
      cons.anchor     = GridBagConstraints.WEST;
      cons.insets.top = cons.insets.left;
      grid.setConstraints(item, cons);
      pane.add(item);

      displayer_combo = new JComboBox(DisplayersBag.instance().getNamesList());
      displayer_combo.setForeground(fg);
      displayer_combo.setOpaque(false);
      cons.gridwidth = GridBagConstraints.REMAINDER;
      grid.setConstraints(displayer_combo, cons);
      pane.add(displayer_combo);

      item  =  new JLabel("Order : ");
      item.setFont(resources.getLabelFont());
      item.setToolTipText("Navigation order");
      item.setForeground(fg);
      cons.gridwidth  = 1;
      cons.anchor     = GridBagConstraints.WEST;
      cons.insets.top = cons.insets.left;
      grid.setConstraints(item, cons);
      pane.add(item);

      nave_combo = new JComboBox(WebNode.ENUMERATION_MODE);
      nave_combo.setForeground(fg);
      nave_combo.setOpaque(false);
      cons.gridwidth = GridBagConstraints.REMAINDER;
      grid.setConstraints(nave_combo, cons);
      pane.add(nave_combo);

      item  =  new JLabel("Pause (s.) : ");
      item.setFont(resources.getLabelFont());
      item.setToolTipText("Display time in seconds");
      item.setForeground(fg);

      cons.gridwidth  = 1;
      cons.anchor     = GridBagConstraints.WEST;
      cons.insets.top = cons.insets.left;
      grid.setConstraints(item, cons);
      pane.add(item);

      int ncol = 8;
      pause_field  = new JTextField(ncol);
      pause_field.setFont(resources.getTextFont());
      pause_field.setBackground(resources.getBoardBackground());
      pause_field.setForeground(resources.getBoardForeground());
      pause_field.setBorder(resources.getInBorder());
      pause_field.setEditable(true);
      pause_field.setHorizontalAlignment(JTextField.RIGHT);
      pause_field.setText(String.valueOf(10));

      /**
       * Checks content of the pause field.
       * If it is not correct reset the original value;
       */
      class PauseListener implements ActionListener {
	  final public void actionPerformed(ActionEvent ev) {
	      try {
		  Integer.parseInt(pause_field.getText());
	      }
	      catch(NumberFormatException ex) {
		 pause_field.setText(String.valueOf(nave_pane.getPauseTime())); 
	      }
	  }
      }
      pause_field.addActionListener(new PauseListener());

      cons.gridwidth = GridBagConstraints.REMAINDER;
      cons.anchor    = GridBagConstraints.WEST;
      grid.setConstraints(pause_field, cons);
      pane.add(pause_field);


      item  =  new JLabel("To display : ");
      item.setFont(resources.getLabelFont());
      item.setToolTipText("List of types to display");
      item.setForeground(fg);

      cons.gridwidth  = 1;
      cons.anchor     = GridBagConstraints.WEST;
      cons.insets.top = cons.insets.left;
      grid.setConstraints(item, cons);
      pane.add(item);

      ncol = 24;
      suffix_field  = new JTextField(ncol);
      suffix_field.setFont(resources.getTextFont());
      suffix_field.setBackground(resources.getBoardBackground());
      suffix_field.setForeground(resources.getBoardForeground());
      suffix_field.setBorder(resources.getInBorder());
      suffix_field.setEditable(true);
      suffix_field.setText("htm html");

      cons.gridwidth = GridBagConstraints.REMAINDER;
      cons.anchor    = GridBagConstraints.EAST;
      grid.setConstraints(suffix_field, cons);
      pane.add(suffix_field);

      return pane;
}
//______________________________________________________________________________
/**
 * Builds the pane showing scanning parameters.
 *
 * @return the pane.
 */
private JComponent buildScanOptionsPane()
{
      JPanel pane        = new JPanel();
      GridBagLayout grid = new GridBagLayout();
      Color fg           = resources.getPaneForeground();
      pane.setLayout(grid);
      pane.setOpaque(true);
      pane.setBackground(resources.getPaneBackground());
      pane.setForeground(fg);

      GridBagConstraints cons = new GridBagConstraints();
      cons.weightx       = 0.0;
      cons.weighty       = 0.0;
      cons.anchor        = GridBagConstraints.CENTER;
      cons.insets.left   = resources.getSpace();
      cons.insets.right  = cons.insets.left;
      cons.insets.top    = 2 * cons.insets.left;
      cons.insets.bottom = cons.insets.left;

      JLabel header = new JLabel("Scanning");
      header.setFont(resources.getLabelFont());
      header.setHorizontalAlignment(JLabel.CENTER);
      header.setToolTipText("To change scanning settings");
      header.setForeground(fg);
      cons.gridwidth = GridBagConstraints.REMAINDER;
      grid.setConstraints(header, cons);
      pane.add(header);

      JLabel item  =  new JLabel("Order : ");
      item.setFont(resources.getLabelFont());
      item.setToolTipText("Scanning order");
      item.setForeground(fg);
      cons.gridwidth  = 1;
      cons.anchor     = GridBagConstraints.WEST;
      cons.insets.top = cons.insets.left;
      grid.setConstraints(item, cons);
      pane.add(item);

      scan_combo = new JComboBox(WebNode.ENUMERATION_MODE);
      scan_combo.setForeground(fg);
      scan_combo.setOpaque(false);
      cons.gridwidth = GridBagConstraints.REMAINDER;
      grid.setConstraints(scan_combo, cons);
      pane.add(scan_combo);

      item  =  new JLabel("Ignored : ");
      item.setFont(resources.getLabelFont());
      item.setToolTipText("Ignore URL containing these tokens");
      item.setForeground(fg);

      cons.gridwidth  = 1;
      cons.anchor     = GridBagConstraints.WEST;
      cons.insets.top = cons.insets.left;
      grid.setConstraints(item, cons);
      pane.add(item);

      final int ncol = 24;
      ignored_field  = new JTextField(ncol);
      ignored_field.setFont(resources.getTextFont());
      ignored_field.setBackground(resources.getBoardBackground());
      ignored_field.setForeground(resources.getBoardForeground());
      ignored_field.setBorder(resources.getInBorder());
      ignored_field.setEditable(true);
      ignored_field.setHorizontalAlignment(JTextField.LEFT);
      ignored_field.setText(null);

      cons.gridwidth = GridBagConstraints.REMAINDER;
      cons.anchor    = GridBagConstraints.EAST;
      grid.setConstraints(ignored_field, cons);
      pane.add(ignored_field);

      return pane;
}
//______________________________________________________________________________
/**
 * Carries out the operation when a button item is pressed.
 * <Br>
 * It overrides the default method in the super Class.
 * It OK is pressed, verbosity, scanning and navigation are updated.
 *
 * @param ev the event generated.
 */
public void actionPerformed(ActionEvent ev)
{
    super.actionPerformed(ev);
    if(! confirmed) return;

    ToolBox.verbose = verbose;
    ToolBox.debug   = debug;

    try {
	int time = Integer.parseInt(pause_field.getText());
	nave_pane.setPauseTime(time);
    }
    catch(NumberFormatException ex) {
	ErrorWindow.instance().display("Invalid pause time value");
    }

    nave_pane.setNavigationOrder(nave_order);
    nave_pane.setDisplayer(displayer);
    SwingDisplayer.instance().setListOfSuffix(suffix_field.getText());

    scan_pane.setScanningOrder(scan_order);
    scan_pane.setIgnoredList(ignored_field.getText());
}
//______________________________________________________________________________
/**
 * Displays the Pane and waits for a button pressed.
 * <Br>
 * @return true if the answer is OK, false otherwise.
 * @see #actionPerformed
 */
public boolean display()
{
    if(! (ToolBox.verbose || ToolBox.debug)) {
	verbose = false;
	debug   = false;
	quiet_button.setSelected(true);
    }
    else if(ToolBox.debug) {
	verbose = true;
	debug   = true;
	debug_button.setSelected(true);
    }
    else {
	verbose = true;
	debug   = false;
	verbose_button.setSelected(true);
    }

    scan_combo.setSelectedIndex(scan_pane.getScanningOrder());
    ignored_field.setText(scan_pane.getIgnoredList());

    DisplayersBag bag = DisplayersBag.instance();
    int index         = bag.getDisplayerIndex();
    if(index >= 0) displayer_combo.setSelectedIndex(index);
    nave_combo.setSelectedIndex(nave_pane.getNavigationOrder());
    pause_field.setText(String.valueOf(nave_pane.getPauseTime()));
    suffix_field.setText(SwingDisplayer.instance().getListOfSuffix());

    setVisible(true);

    return confirmed;
}
//______________________________________________________________________________
/**
 * Updates stuff depending on the navigation order.
 * <Br>
 * @param order the selected order.
 */
final private void updateNavigationOrder(int order)
{
    nave_order = order;
}
//______________________________________________________________________________
/**
 * Saves the current setting in <code>$HOME/.jvs.properties</code>
 * <br>
 * The originale file is saved in a temp file (<code>.jvs.properties~</code>)
 * @see #getUserProperties
 */
final protected void save()
{
    if(ToolBox.debug) ToolBox.warn("Saving current settings");

    try {
	Properties prop = getUserProperties();

	Resources rsrc = (org.demo.webwader.Resources)Resources.instance();
	String jvsprop = rsrc.getUserHomeDirectory() + ".jvs.properties";

	String path    = backup_done ? jvsprop : ToolBox.markAsTemp(jvsprop);
	FileOutputStream file = new FileOutputStream(path);

	prop.store(file, " JVS Resources");
	file.close();
	if(! backup_done) ToolBox.renameFile(jvsprop);
	backup_done = true;
    }
    catch(Exception ex) {
	ErrorWindow.instance().display(
		    "Can't backup settings :\n" + ex.getMessage());
    }
}
//______________________________________________________________________________
/**
 * Gets the properties defined by the user himself.
 * <Br>
 * They are taken in <code>$HOME/.jvs.properties</code> and in the current
 * settings.
 * @return the user's resources.
 * @throw IOException if the user's properties cannot be read.
 * @see #save
 */
final private Properties getUserProperties() throws  IOException
{
    Properties prop = new Properties();
    Resources rsrc  = (org.demo.webwader.Resources)Resources.instance();
    String filename = rsrc.getUserHomeDirectory() + ".jvs.properties";
    File   file     = new File(filename);

    if(file.exists()) {
	FileInputStream input = new FileInputStream(file);
	prop.load(input);
	input.close();
    }

    String value = ToolBox.verbose ? "verbose" : "quiet";
    if(ToolBox.debug) value = "debug";
    prop.put("Mode", value);

    value = String.valueOf(nave_pane.getPauseTime());
    prop.put("WebWader.Navigation.Pause", value);

    value = nave_pane.getMode() ? "Manual" : "Automatic";
    prop.put("WebWader.Navigation.Mode", value);

    value = WebNode.navigationOrder(nave_pane.getNavigationOrder());
    prop.put("WebWader.Navigation.Order", value);

    prop.put("WebWader.Displayer", DisplayersBag.instance().getDisplayerName());

    value = SwingDisplayer.instance().getListOfSuffix();
    if(value.length() > 0) prop.put("WebWader.TypesToDisplay", value);

    value = String.valueOf(nave_pane.getDoDisplay());
    prop.put("WebWader.DoDisplay", value);

    value = String.valueOf(MainFrame.instance().getShowMiniBoardState());
    prop.put("WebWader.ShowMiniBoard", value);

    value = WebNode.navigationOrder(scan_pane.getScanningOrder());
    prop.put("WebWader.Scanning.Order", value);

    value = scan_pane.getIgnoredList();
    if(value.length() > 0) prop.put("WebWader.Ignored", value);

    value = scan_pane.getSitesList();
    if(value.length() > 0) prop.put("WebWader.SitesList", value);

    return prop;
}
//______________________________________________________________________________
/**
 * Changes the current displayer.
 * <Br>
 * @param displayer the selected displayer by the combo.
 */
final private void updateDisplayer(int displayer)
{
    this.displayer = displayer;
}
//______________________________________________________________________________
/**
 * Updates stuff depending on the scanning order.
 * <Br>
 * @param order the selected order.
 */
final private void updateScanningOrder(int order)
{
    scan_order = order;
}
//______________________________________________________________________________
}



