//______________________________________________________________________________

//	The Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader.gui;

import org.demo.webwader.WebNode;
import org.demo.webwader.WebSite;
import org.ariane.tools.ToolBox;
import org.ariane.gui.TextField;
import org.ariane.exec.ExecutionException;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.util.ListIterator;
import java.util.NoSuchElementException;
import java.net.MalformedURLException;
import java.net.URL;

/**
 * The window used to navigate through the web site.
 * <Br>
 * It displays controls which can be used to navigate through the tree
 * of web pages. Information about the selected node is presented.
 *
 * @version $Id: NavePane.java,v 3.21 2001/02/14 20:55:03 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */
//______________________________________________________________________________

class NavePane extends JPanel implements ActionListener {
  /**
   * @serial The unique instance.
   */
private static NavePane  nave_pane = null;
  /**
   * @serial The associated TreePane.
   */
private TreePane  tree_pane = null;
  /**
   * @serial The bag of tools used to display pages.
   */
private DisplayersBag bag = null;
  /**
   * @serial If true do show the pages.
   */
private boolean do_display = false;
  /**
   * @serial The page path field.
   */
private  TextField path_field;
  /**
   * @serial The page title field.
   */
private  TextField title_field;
    /**
     * @serial the button Top.
     */
private JButton top_button;
    /**
     * @serial the button Previous.
     */
private JButton prev_button;
    /**
     * @serial the button Stop.
     */
private JButton stop_button;
    /**
     * @serial the button Next.
     */
private JButton next_button;
    /**
     * @serial the button Start.
     */
private JButton start_button;
  /**
   * @serial A mode button.
   */
private JRadioButton auto_button;
  /**
   * @serial A mode button.
   */
private JRadioButton manual_button;
  /**
   * @serial The navigation listener.
   */
private ActionListener nave_listener;
  /**
   * @serial The change mode listener.
   */
private ActionListener mode_listener;
  /**
   * @serial The previous button icon.
   */
private Icon prev_icon;
  /**
   * @serial The next button icon.
   */
private Icon next_icon;
  /**
   * @serial The start button icon.
   */
private Icon start_icon;
  /**
   * @serial The stop button icon.
   */
private Icon stop_icon;
  /**
   * @serial The mode either manual or automatic.
   */
private boolean manual = true;
  /**
   * @serial One of the navigation order.
   */
private int nave_order = WebNode.BREADTH_FIRST;
  /**
   * @serial The time (in seconds) during which a page
   * is displayed in automatic mode.
   */
private int pause_time = 1;
    /**
     * @serial The button top command.
     */
private	static final String TOP   = "TOP";
    /**
     * @serial The button previous command.
     */
protected static final String PREV   = "PREV";
    /**
     * @serial The button stop command.
     */
protected static final String STOP   = PREV;
    /**
     * @serial The button previous command.
     */
protected static final String NEXT   = "NEXT";
    /**
     * @serial The button start command.
     */
protected static final String START  = NEXT;
    /**
     * @serial The current Iterator.
     */
private	ListIterator iterator = null;
    /**
     * @serial The current displayed URL.
     */
private	URL current_url = null;
    /**
     * @serial The direction of navigation, either forward or backward.
     */
private	boolean forward = true;
    /**
     * @serial The previous selected node.
     */
private	WebNode former = null;
  /**
   * @serial The timer controlling the display of the pages.
   */
private Timer timer;
  /**
   * @serial The initial time of a navigation.
   */
private long ti;
//______________________________________________________________________________
/**
 * Gets an unique instance of the NavePane.
 * <Br>
 * It is a singleton.
 * @return the unique instance.
 */
protected static NavePane instance()
  {
    if(nave_pane == null) nave_pane = new NavePane();

    return nave_pane;
  }
//______________________________________________________________________________
/**
 * Creates the Panel allowing the user to navigate a web site.
 */
private NavePane()
  {
      super();

      GridBagLayout grid     = new GridBagLayout();
      GuiResources resources = GuiResources.instance();
      Color fg               = resources.getPaneForeground();

      int width  = resources.get("NavePane.Width",  450);
      int height = resources.get("NavePane.Height", 300);
      setMaximumSize(new Dimension(width, height));

      setLayout(grid);
      setBackground(resources.getPaneBackground());
      setForeground(fg);
      setFont(resources.getHeaderFont());

      prev_icon  = resources.getIcon("previous.gif");
      next_icon  = resources.getIcon("next.gif");
      start_icon = resources.getIcon("start.gif");
      stop_icon  = resources.getIcon("stop.gif");

      GridBagConstraints cons = new GridBagConstraints();

      cons.weightx       = 0.0;
      cons.weighty       = 0.0;
      cons.anchor        = GridBagConstraints.CENTER;
      cons.insets.left   = resources.getSpace();
      cons.insets.right  = cons.insets.left;
      cons.insets.top    = 2 * cons.insets.left;
      cons.insets.bottom = cons.insets.left;

      JLabel item;

      item  =  new JLabel("Page path : ");
      item.setFont(resources.getLabelFont());
      item.setForeground(fg);
      org.ariane.gui.ToolBox.fixComponentSize(item);
      cons.weightx      = 0.0;
      cons.gridwidth    = 1;
      cons.anchor       = GridBagConstraints.WEST;
      cons.insets.right = 0;
      grid.setConstraints(item, cons);
      add(item);

      int ncol    = resources.get("PathField.Width", 48);
      path_field  = new TextField(ncol);
      path_field.setFont(resources.getTextFont());
      path_field.setBackground(resources.getBoardBackground());
      path_field.setForeground(resources.getBoardForeground());
      path_field.setEditable(false);
      path_field.setText(null);
      cons.gridwidth    = 4;
      grid.setConstraints(path_field, cons);
      add(path_field);

      item  =  new JLabel("Page title : ");
      item.setFont(resources.getLabelFont());
      item.setForeground(fg);
      org.ariane.gui.ToolBox.fixComponentSize(item);
      cons.gridy        = 1;
      cons.weightx      = 0.0;
      cons.gridwidth    = 1;
      cons.anchor       = GridBagConstraints.WEST;
      cons.insets.right = 0;
      grid.setConstraints(item, cons);
      add(item);

      title_field  =  new TextField(ncol);
      title_field.setFont(resources.getTextFont());
      title_field.setBackground(resources.getBoardBackground());
      title_field.setForeground(resources.getBoardForeground());
      title_field.setEditable(false);
      title_field.setText(null);
      cons.gridwidth    = 4;
      grid.setConstraints(title_field, cons);
      add(title_field);

      /**
       * Process a Button pressed.
       */
      class NaveListener implements ActionListener {
	  final public void actionPerformed(ActionEvent ev) {
	      processAction(ev.getActionCommand());
	  }
      }
      nave_listener = new NaveListener();

      Dimension bdim  = new Dimension(31, 31);
      Icon icon       = resources.getIcon("home.gif");
      Insets margin   = new Insets(0,0, 0,0);
      JButton button  = new JButton(icon);
      button.setToolTipText("Return to Top");
      button.setMargin(margin);
      button.addActionListener(nave_listener);
      button.setActionCommand(TOP);
      button.setPreferredSize(bdim);

      cons.gridy        = 2;
      cons.gridwidth    = 1;
      cons.anchor       = GridBagConstraints.WEST;
      grid.setConstraints(button, cons);
      add(button);
      top_button = button;

      button  =  new JButton(prev_icon);
      button.setToolTipText("Go to previous page");
      button.setMargin(margin);
      button.addActionListener(nave_listener);
      button.setActionCommand(PREV);
      button.setPreferredSize(bdim);

      grid.setConstraints(button, cons);
      add(button);
      prev_button = button;
      stop_button = button; // It is the same button, the name is an alias.

      button  =  new JButton(next_icon);
      button.setToolTipText("Go to next page");
      button.setMargin(margin);
      button.addActionListener(nave_listener);
      button.setActionCommand(NEXT);
      button.setPreferredSize(bdim);

      cons.insets.right  = 4 * cons.insets.bottom;
      grid.setConstraints(button, cons);
      add(button);
      next_button  = button;
      start_button = button; // It is the same button

      ButtonGroup group  = new ButtonGroup();

      /**
       * Updates the navigation buttons when the mode is changed.
       * This Pane and the MiniBoard are concerned.
       */
      class ModeListener implements ActionListener {
	  final public void actionPerformed(ActionEvent ev) {
	      setMode(ev.getActionCommand().charAt(0) == 'M');
	      NaveBoard.instance().setMode(manual);
	  }
      }
      mode_listener = new ModeListener();

      manual_button  = new JRadioButton("Manual");
      manual_button.setToolTipText("Set manual mode");
      manual_button.setFont(resources.getTextFont());
      manual_button.setOpaque(false);
      manual_button.setForeground(fg);
      manual_button.addActionListener(mode_listener);

      cons.insets.right =     cons.insets.bottom;
      cons.insets.left  = 4 * cons.insets.bottom;
      grid.setConstraints(manual_button, cons);
      add(manual_button);
      group.add(manual_button);

      auto_button   = new JRadioButton("Auto");
      auto_button.setToolTipText("Set automatic mode");
      auto_button.setFont(resources.getTextFont());
      auto_button.setOpaque(false);
      auto_button.setForeground(fg);
      auto_button.addActionListener(mode_listener);

      cons.insets.left  = cons.insets.bottom;
      grid.setConstraints(auto_button, cons);
      add(auto_button);
      group.add(auto_button);

      String mode = resources.get("Navigation.Mode", "Manual");
      setMode(mode.equalsIgnoreCase("Manual"));

      timer = new Timer(getSleepTime(), this);
      timer.setRepeats(false);

      setPauseTime(resources.get("Navigation.Pause", 10));

      mode  = resources.get("Navigation.Order", "BreadthFirst");
      setNavigationOrder(WebNode.navigationOrder(mode));

      bag = DisplayersBag.instance();
  }
//______________________________________________________________________________
/**
 * Gets the navigation mode : manual or automatic.
 *
 * @return the mode either manual (true) or automatic (false).
 * @see NaveBoard#setMode
 */
final protected boolean getMode()
{
    return manual;
}
//______________________________________________________________________________
/**
 * Sets the navigation mode : manual or automatic.
 * <Br>
 * The navigation buttons are updated. The same buttons are used in both modes.
 * <B>Previous</B> is <B>Stop</B> and <B>Next</B> is <B>Start</B>.
 *
 * @param manual, changes to manual if true, to automatic otherwise.
 * @see NaveBoard#setMode
 */
final private void setMode(boolean manual)
{
    this.manual = manual;

    if(manual) {
	manual_button.setSelected(true);
	prev_button.setIcon(prev_icon);
	prev_button.setToolTipText("Go to previous page");

	next_button.setIcon(next_icon);
	next_button.setToolTipText("Go to next page");

	top_button.setEnabled(true);
	prev_button.setEnabled(true);
	next_button.setEnabled(true);
    }
    else {
	auto_button.setSelected(true);
	stop_button.setIcon(stop_icon);
	stop_button.setToolTipText("Stop navigation in progress");

	start_button.setIcon(start_icon);
	start_button.setToolTipText("Start new navigation");

	top_button.setEnabled(true);
	start_button.setEnabled(true);
	stop_button.setEnabled(false);
    }
}
//______________________________________________________________________________
/**
 * Updates fields when a new node is selected.
 * <Br>
 * @param node the new web node.
 */
final protected void updateNodeInfo(WebNode node)
{
    if(node == null) {
	path_field.setText(null);
	title_field.setText(null);
	return;
    }

    path_field.setText(node.getPagePath());
    title_field.setText(node.getTitle());
}
//______________________________________________________________________________
/**
 * Gets the navigation listener.
 *
 * @return the listener.
 * @see NaveBoard
 */
final protected ActionListener getNaveListener()
{
    return nave_listener;
}
//______________________________________________________________________________
/**
 * Gets the change mode listener.
 *
 * @return the listener.
 * @see NaveBoard
 */
final protected ActionListener getModeListener()
{
    return mode_listener;
}
//______________________________________________________________________________
/**
 * Processes the command launched by a button.
 * <Br>
 * @param cmd the command issued by the button.
 */
final private void processAction(String cmd)
{
    if(tree_pane == null) {
	tree_pane  = TreePane.instance();
    }

    if(manual) 
	processManualAction(cmd);
    else
	processAutoAction(cmd); 
}
//______________________________________________________________________________
/**
 * Processes the command launched by the button in manual mode.
 * <Br>
 * @param cmd the command issued by the button.
 */
final private void processManualAction(String cmd)
{
    if(cmd.equals(TOP)) {
	former  = null;
	forward = true;
	resetIterator();
	tree_pane.setSelectedNode(null);
	return;
    }

    getIterator();
    if(iterator == null) return;

    WebNode node = former;

    try {
	if(cmd.equals(NEXT)) {
	    if(! forward) {
		/**
		 * The previous action was PREV.
		 */
		if(former != null) node  = (WebNode)iterator.next();
		forward = true;
	    }

	    if(! iterator.hasNext()) {
		/**
		 * Loop again.
		 */
		resetIterator();
		getIterator();
	    }

	    node = (WebNode)iterator.next();
	}
	else if(cmd.equals(PREV)) {
	    if(forward) {
		/**
		 * The previous action was NEXT.
		 */
		if(former != null) node = (WebNode)iterator.previous();
		forward = false;
	    }

	    if(! iterator.hasPrevious()) {
		/**
		 * Loop again.
		 */
		positionIterator(null);
	    }
	    node = (WebNode)iterator.previous();
	}
    }
    catch(NoSuchElementException ex) {
	ToolBox.warn("Invalid manual action " + cmd, ex);
    }

    former = node;
    tree_pane.setSelectedNode(node);
    displayPage(node);
}
//______________________________________________________________________________
/**
 * Processes the command launched by the button in automatic mode.
 * <Br>
 * @param cmd the command issued by the button.
 * @see #actionPerformed
 */
final private void processAutoAction(String cmd)
{
    updateButtonStatus(cmd);
    forward = true;

    if(cmd.equals(START)) {
	ti = System.currentTimeMillis();
	actionPerformed(null);
    }
    else if(cmd.equals(TOP)) {
	former  = null;
	resetIterator();
	tree_pane.setSelectedNode(null);
	MainFrame.instance().stopTimer();
    }
    else { // STOP
	timer.stop();
    }
}
//______________________________________________________________________________
/**
 * Processes the command launched by the button in automatic mode.
 * <p>
 * It is triggered by the timer. When a page is actually displayed
 * a pause takes place to give user a chance to see how the document is
 * rendered.
 */
public void actionPerformed(ActionEvent evt)
{
    getIterator();
    if(iterator == null) {
	updateButtonStatus(STOP);
	timer.stop();
	return;
    }
    else if(! iterator.hasNext()) {
	actionStopped();
	return;
    }

    WebNode node = (WebNode)iterator.next();

    tree_pane.setSelectedNode(node);

    boolean rc = displayPage(node);

    if(rc) { // Pause if actually displayed.
	timer.start();
	MainFrame.instance().startTimer(getSleepTime());
    }
    else {
	timer.setInitialDelay(10); // Just a tiny delay instead of 0 ms.
	timer.start();
	timer.setInitialDelay(getSleepTime()); // Reset the original one.
    }

    former = node;
}
//______________________________________________________________________________
/**
 * Processes end of a navigation. The timer is stopped.
 */
private void actionStopped()
{
    timer.stop();
    resetIterator();
    updateButtonStatus(STOP);

    long dt = (System.currentTimeMillis() - ti) / 1000;
    MainFrame.inform("Navigation done in " + dt + " s.");
}
//______________________________________________________________________________
/**
 * Displays the node selected in the tree with the mouse.
 * <Br>
 * The iterator is positionned and the page is displayed
 * @param node the Web node.
 * @see TreePane#processNodeMouseSelection
 * @see #positionIterator
 */
final protected void displaySelectedNode(WebNode node)
{
    if(node == null) {
	ToolBox.warn("Can't display null selected node");
	return;
    }

    positionIterator(node);
    displayPage(node);
}
//______________________________________________________________________________
/**
 * Updates the status of the button.
 * <Br>
 * @param cmd the command issued by the button.
 * @see NaveBoard#updateButtonStatus
 */
final private void updateButtonStatus(String cmd)
{
    /**
     * Update the status of the buttons.
     * When Start is pressed, only Stop is kept sensitive.
     */
    if(cmd.equals(START)) {
	top_button.setEnabled(false); 
	start_button.setEnabled(false);
	stop_button.setEnabled(true);

	manual_button.setEnabled(false);
	auto_button.setEnabled(false);
    }
    else if(cmd.equals(TOP)) {
	top_button.setEnabled(true);
	start_button.setEnabled(true);
	stop_button.setEnabled(false);
    }
    else { // STOP
	top_button.setEnabled(true);
	start_button.setEnabled(true);
	stop_button.setEnabled(false);

	manual_button.setEnabled(true);
	auto_button.setEnabled(true);
    }

    NaveBoard.instance().updateButtonStatus(cmd);
}
//______________________________________________________________________________
/**
 * Displays the specified page if there is a displayer and if
 * display is actually requested.
 * <br>
 * Keeps track of the current URL.
 * @param node the node containing the page.
 * @return true is the page has been displayed.
 * @see Displayer#displayPage
 */
final private boolean displayPage(WebNode node)
{
    Displayer displayer = bag.getDisplayer();

    if(displayer == null || ! do_display)
	return false;

    URL url    = getNodeURL(node);
    boolean rc = displayer.displayPage(url);
    if(rc) current_url = url;

    return rc;
}
//______________________________________________________________________________
/**
 * Gets the URL stored in the node.
 * <Br>
 * It is a concatenation of <Code>site-root/node-path</Code>.
 * It the node is null, returns null.
 * @param node the Web node.
 * @return the string giving the URL.
 * @see WebNode#getURL()
 */
final private URL getNodeURL(WebNode node)
{
    if(node == null) return null;
    try {
	return node.getURL();
    }
    catch(MalformedURLException ex) {
	ToolBox.warn("Invalid url of " + node.toString(), ex);
	return null;
    }
    catch(Exception ex) {
	if(ToolBox.debug)
	    ToolBox.warn("Can't get url of " + node.toString(), ex);
	return null;
    }
}
//______________________________________________________________________________
/**
 * Gets the navigation order.
 * <Br>
 * @return the current order.
 */
final protected int getNavigationOrder()
{
    return nave_order;
}
//______________________________________________________________________________
/**
 * Sets the navigation order.
 * <Br>
 * The current iterator is reset to null;
 * @param the new order.
 */
final protected void setNavigationOrder(int nave_order)
{
    if(this.nave_order == nave_order) return;
    this.nave_order = nave_order;
    resetIterator();
}
//______________________________________________________________________________
/**
 * Changes the condition of display.
 * <br>
 * It do_display is false, pages are not displayed by a browser.
 * Useful for debugging.
 * @param do_display the new value of the flag.
 * @see #getDoDisplay
 */
final protected void setDoDisplay(boolean do_display)
{
    this.do_display = do_display;
}
//______________________________________________________________________________
/**
 * Gets the condition of display.
 * <Br>
 * @return the flag.
 * @see #setDoDisplay
 */
final protected boolean getDoDisplay()
{
    return do_display;
}
//______________________________________________________________________________
/**
 * Sets the current tool used to display pages.
 * <Br>
 * @param index the index of the new displayer.
 * @see DisplayersBag#setWindowsDisplayer
 * @see DisplayersBag#setUnixDisplayer
 */
final protected void setDisplayer(int index)
{
    int previous = bag.getDisplayerIndex();
    /**
     * No change, simply return
     */
    if(previous == index) return;

    boolean done = bag.setDisplayer(index);
    if(! done) {
	MainFrame.inform("Displayer not changed !");
	return;
    }

    Displayer displayer = bag.getDisplayer();

    if(displayer == null) {
	 bag.setDisplayer(previous);
	 return;
    }

    try {
	displayer.activate();
	if(current_url != null) displayer.displayPage(current_url);
    }
    catch(ExecutionException ex) {
	ErrorWindow.instance().display(
        "Can't activate displayer " + displayer.getName() + " !");
	bag.setDisplayer(previous);
    }
}
//______________________________________________________________________________
/**
 * Sets the iterator at a specific position.
 * <Br>
 * If the target node is null, positions at end.
 * @param node the Web node.
 * @see #getIterator()
 */
final private void positionIterator(WebNode node)
{
    resetIterator();
    getIterator();

    former = null;

    if(iterator == null) return;

    while(iterator.hasNext()) {
	WebNode n = (WebNode)iterator.next();
	if(n == node) {
	    return;
	}
        former = n;
    }
}
//______________________________________________________________________________
/**
 * Resets to null the current iterator.
 * @see #getIterator()
 */
final protected void resetIterator()
{
    iterator = null;
}
//______________________________________________________________________________
/**
 * Initializes the iterator if needed.
 * @see #resetIterator()
 * @see WebSite#getIterator
 */
final private void getIterator()
{
    if(iterator != null) return;

    WebSite site = Controller.instance().getWebSite();
    if(site == null) return;

    iterator = site.getIterator(nave_order);
}
//______________________________________________________________________________
/**
 * Gets the display time in milliseconds.
 * <Br>
 * @return the current time.
 */
final private int getSleepTime()
{
    return pause_time * 1000;
}
//______________________________________________________________________________
/**
 * Gets the display time in seconds.
 *
 * @return the current time.
 */
final protected int getPauseTime()
{
    return pause_time;
}
//______________________________________________________________________________
/**
 * Sets the display time in seconds.
 * <Br>
 * If the argument is negative, time is set to 1.
 * @param the new time.
 */
final protected void setPauseTime(int time)
{
    pause_time = (time < 1) ? 1 : time;
    timer.setDelay(getSleepTime());
    timer.setInitialDelay(getSleepTime());
    if(false)
	ToolBox.warn("Timer set to : " + timer.getDelay() + " ms.");
}
//______________________________________________________________________________
}
