//______________________________________________________________________________

//	The Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader.gui;

import java.io.File;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.border.Border;
import java.net.URL;

import org.ariane.applet.Messenger;
import org.ariane.gui.TimerBar;
import org.ariane.tools.ToolBox;

/**
 * Manages the main frame and the menu bar of the Traverser.
 * <Br>
 * The main frame is divided in four parts :
 * <Ul>
 * <Li>the top pane which displays a couple of nice icons.
 *
 * <Li>the nave pane allowing control of the navigation.

 * <Li>the main pane showing the map of a web site.
 *
 * <Li>the bottom pane used to give some feeback to the user.
 * </Ul>
 * @version $Id: MainFrame.java,v 3.11 2001/01/28 11:33:01 lefevre Exp $
 * @author Jean-Paul Le Fvre
 * @see Traverser
 */
//______________________________________________________________________________

public	class MainFrame extends JFrame implements ActionListener {
    /**
     * The button Open label.
     */
private	static final String OPEN   = "Open";
    /**
     * The button Save label.
     */
private	static final String SAVE   = "Save";
    /**
     * The button Set Options label.
     */
private	static final String SET_OPTIONS = "SetOptions";
    /**
     * The button Save Options label.
     */
private	static final String SAVE_OPTIONS = "SaveOptions";
    /**
     * The button Display label.
     */
private	static final String DISPLAY = "Display";
    /**
     * The button Display label.
     */
private	static final String SHOW_MINI = "ShowMiniBoard";
    /**
     * The button Scan label.
     */
private	static final String SCAN = "Scan";
    /**
     * The button Quit label.
     */
private	static final String QUIT   = "Quit";
    /**
     * The button About label.
     */
private	static final String ABOUT  = "About WebWader ...";
    /**
     * The button Manual label.
     */
private	static final String MANUAL = "WebWader short help ...";
    /**
     * The button Manual label.
     */
private	static final String WADERSITE = "WebWader home page ...";
    /**
     * @serial The main container for the widgets.
     */
private	Container window;
    /**
     * @serial The object used to display transient messages.
     */
static private Messenger messenger;
    /**
     * @serial The object used to show work in progress.
     */
private TimerBar timerbar;
    /**
     * @serial The set of properties.
     */
private GuiResources resources;
  /**
   * @serial The unique instance.
   */
private  static MainFrame main_frame = null;
    /**
     * @serial The general controller.
     */
private Controller controller;
    /**
     * @serial The button used to open or close the mini board.
     */
private JCheckBoxMenuItem showmini;

//______________________________________________________________________________
/**
 * Gets an unique instance of the main frame.
 * <Br>
 * It is a singleton.
 * @return the unique instance.
 */
protected static MainFrame instance()
  {
    if(main_frame == null) main_frame = new MainFrame("WebWader");
    return main_frame;
  }
//______________________________________________________________________________
  /**
   * Displays a transient message at the bottom.
   * @parameter text the message to display.
   * @see Messenger#display
   */
static final protected void inform(String text)
{
    messenger.display(text);
}
//______________________________________________________________________________
  /**
   * Clears the message displayed at the bottom.
   * @see Messenger#erase
   */
static final protected void clearMessage()
{
    messenger.erase();
}
//______________________________________________________________________________
/**
 * Creates the MainFrame of the application.
 * @param name the name of the application.
 */
private MainFrame(String name)
  {
      super(name);
      ToolBox.dot();

      controller = Controller.instance();
      resources  = GuiResources.instance();

      ToolBox.dot();
      int width  = resources.get("Width",  800);
      int height = resources.get("Height", 400);
      setSize(width, height);

      JMenuBar menubar = buildMenuBar();
      setJMenuBar(menubar);

      window = getContentPane();
      window.setLayout(new BoxLayout(window, BoxLayout.Y_AXIS));

      window.setBackground(resources.getMainBackground());
      window.setForeground(resources.getMainForeground());
      window.setFont(resources.getHeaderFont());

      JComponent toppane = buildTopPane();
      window.add(toppane);
      ToolBox.dot();

      JComponent nave_pane = buildNavePane();
      window.add(nave_pane);
      ToolBox.dot();

      window.add(Box.createVerticalStrut(2*resources.getSpace()));

      JComponent tree_pane = buildTreePane();
      window.add(tree_pane);
      ToolBox.dot();

      JComponent bottompane = buildBottomPane();
      window.add(bottompane);
      ToolBox.dot();

      setVisible(true);
      ToolBox.dot();

      System.out.println();

      inform("Welcome to this new session of Traverser !");
  }
//______________________________________________________________________________
/**
 * Builds the top pane.
 * <Br>
 * It is a strip at the top of the viewer with icons and title.
 * @return the built component.
 */
private JComponent buildTopPane()
  {
      JPanel pane    = new JPanel();
      pane.setLayout(new BoxLayout(pane, BoxLayout.X_AXIS));
      pane.add(Box.createRigidArea(new Dimension(resources.getSpace(), 50)));
      pane.setOpaque(false);

      Icon icon      = resources.getIcon("jvs.gif");
      JLabel label   = new JLabel(icon);
      label.setAlignmentX(Component.LEFT_ALIGNMENT);
      label.setBorder(resources.getLineBorder());
      pane.add(label);

      pane.add(Box.createHorizontalGlue());

      icon    = resources.getIcon(resources.get("Title", "webwader.gif"));
      label   = new JLabel(icon);
      label.setOpaque(false);
      label.setAlignmentX(Component.CENTER_ALIGNMENT);
      pane.add(label);
      pane.add(Box.createRigidArea(new Dimension(40*resources.getSpace(), 0)));

      return pane;
  }
//______________________________________________________________________________
/**
 * Builds the navigation pane.
 * <Br>
 * @return the built component.
 * @see NavePane
 */
private JComponent buildNavePane()
  {
      JPanel pane    = new JPanel();

      pane.setBackground(resources.getPaneBackground());
      pane.setLayout(new BoxLayout(pane, BoxLayout.X_AXIS));
      pane.setOpaque(true);
      Border border = BorderFactory.createCompoundBorder(
      resources.getMatteBorder(), resources.getLineBorder());
      pane.setBorder(border);

      JComponent nave = NavePane.instance();
      nave.setAlignmentX(Component.LEFT_ALIGNMENT);
      pane.add(nave);

      pane.add(Box.createHorizontalGlue());

      return pane;
  }
//______________________________________________________________________________
/**
 * Builds the tree pane.
 * <Br>
 * @return the built component.
 * @see TreePane
 */
private JComponent buildTreePane()
  {
      JPanel pane    = new JPanel();
      pane.setLayout(new BoxLayout(pane, BoxLayout.Y_AXIS));
      pane.setAlignmentX(Component.CENTER_ALIGNMENT);
      pane.setOpaque(false);
      Border border = BorderFactory.createCompoundBorder(
      resources.getMatteBorder(), resources.getLineBorder());
      pane.setBorder(border);

      ToolBox.dot();

      JComponent tree = TreePane.instance();
      ToolBox.dot();

      JScrollPane scroll = new JScrollPane(tree);
      pane.add(scroll);

      return pane;
  }
//______________________________________________________________________________
/**
 * Builds the bottom pane.
 * <Br>
 * It is a strip at the bottom of the viewer. It displays a progress
 * bar and a message area.
 *
 * @return the built component.
 */
private JComponent buildBottomPane()
  {
      JPanel pane = new JPanel();
      pane.setLayout(new FlowLayout(FlowLayout.LEFT));
      pane.setOpaque(false);

      Component bar = buildTimerBar();
      pane.add(bar);

      Component area = buildMessageArea();
      pane.add(area);

      return pane;
  }
//______________________________________________________________________________
/**
 * Builds the progress bar.
 * @return the built component.
 */
private JComponent buildTimerBar()
  {
      TimerBar pane = new TimerBar();
      pane.setPeriod(resources.get("TimerPeriod", 100));
      pane.setBackground(resources.getPaneBackground());
      pane.setForeground(resources.get("TimerColor", Color.red));
      pane.setBorder(resources.getInBorder());
      timerbar = pane;

      return pane;
  }
//______________________________________________________________________________
/**
 * Starts a count down bar.
 * @return the component.
 */
final protected void startTimer(int duration)
  {
      timerbar.start(duration);
  }
//______________________________________________________________________________
/**
 * Stops the timer. It is forced.
 */
final protected void stopTimer()
  {
      timerbar.stop();
  }
//______________________________________________________________________________
/**
 * Builds the message area.
 * @return the built component.
 * @see Messenger
 */
private JComponent buildMessageArea()
  {
      JLabel pane = new JLabel();

      pane.setBackground(resources.getMainBackground());
      pane.setForeground(resources.getMainForeground());
      pane.setFont(resources.getTextFont());
      ToolBox.dot();

      messenger = new Messenger(pane);
      messenger.setDisplayTime(
      resources.get("DisplayTime", 5));

      return pane;
  }
//______________________________________________________________________________
/**
 * Builds the menu bar.
 * @return the built component.
 */
private JMenuBar buildMenuBar()
  {
      JMenuBar menubar  = new JMenuBar();
      Color fg          = resources.getMainForeground();
      Color bg          = resources.getMainBackground();
      menubar.setBackground(bg);
      menubar.setForeground(fg);

      JMenu menu;
      JMenuItem item;

      menu   = new JMenu("File");
      menu.setToolTipText("To execute main operations");
      menu.setOpaque(false);
      menu.setForeground(fg);
      menubar.add(menu);

      item = new JMenuItem("Open file ...");
      item.setActionCommand(OPEN);
      setMenuItem(item);
      menu.add(item);

      item = new JMenuItem("Scan site ...");
      item.setActionCommand(SCAN);
      setMenuItem(item);
      menu.add(item);

      item = new JMenuItem("Save as ...");
      item.setActionCommand(SAVE);
      setMenuItem(item);
      menu.add(item);

      item = new JMenuItem("Quit");
      item.setActionCommand(QUIT);
      setMenuItem(item);
      menu.addSeparator();
      menu.add(item);


      // The settings menu.

      menu = new JMenu("Settings");
      menu.setOpaque(false);
      menu.setForeground(fg);
      menu.setToolTipText("To change current configuration");
      menubar.add(menu);

      item = new JMenuItem("Set options ...");
      item.setActionCommand(SET_OPTIONS);
      setMenuItem(item);
      item.setToolTipText("To set various parameters");
      menu.add(item);

      item = new JMenuItem("Save options ...");
      item.setActionCommand(SAVE_OPTIONS);
      setMenuItem(item);
      item.setToolTipText("To save the current setting");
      menu.add(item);

      JCheckBoxMenuItem 
      check = new JCheckBoxMenuItem("Display Pages");
      check.setActionCommand(DISPLAY);
      check.setState(resources.get("DoDisplay", true));
      setMenuItem(check);
      check.setToolTipText("To actually display web pages");
      menu.add(check);
      processDoDisplay(check.getState());

      showmini = new JCheckBoxMenuItem("Show MiniBoard");
      showmini.setActionCommand(SHOW_MINI);
      showmini.setState(resources.get("ShowMiniBoard", false));
      setMenuItem(showmini);
      showmini.setToolTipText("To open the mini navigation board");
      menu.add(showmini);

      // The help menu.

      menu   = new JMenu("Help");
      menu.setOpaque(false);
      menu.setForeground(fg);
      menubar.add(menu); //setHelpMenu
      menu.setToolTipText("To get information about WebWader");

      item = new JMenuItem(MANUAL);
      item.setToolTipText("Where to find documentation");
      menu.add(item);
      setMenuItem(item);

      item = new JMenuItem(WADERSITE);
      item.setToolTipText("Connect to WebWader home page");
      menu.add(item);
      setMenuItem(item);

      menu.addSeparator();

      item = new JMenuItem(ABOUT);
      item.setToolTipText("To find out current version");
      menu.add(item);
      setMenuItem(item);

      return menubar;
  }
//______________________________________________________________________________
/**
 * Sets attributes to a menu item.
 * @param item the button to define.
 */
final private void setMenuItem(JMenuItem item)
  {
      item.addActionListener(this);
      item.setBackground(resources.getMainBackground());
      item.setForeground(resources.getMainForeground());
  }
//______________________________________________________________________________
/**
 * Returns the state of the ShowMiniBoard option.
 * @return the current state.
 * @see JCheckBoxMenuItem#getState
 */
final protected boolean getShowMiniBoardState()
{
    return showmini.getState();
}
//______________________________________________________________________________
/**
 * Changes the state of the ShowMiniBoard option.
 * @param status the new state.
 * @see JCheckBoxMenuItem#setState
 */
final protected void setShowMiniBoardState(boolean status)
{
    showmini.setState(status);
}
//______________________________________________________________________________
/**
 * Converts MainFrame to String.
 * @return str	the string.
 */
public String toString()
  {
    return "MainFrame " + getName();
  }
//______________________________________________________________________________
/**
 * Carries out an operation when a menu item is selected.
 * <Br>
 * Depending on the command name stored in the event, a specific method
 * is called.
 *
 * @parameter ev the event generated.
 */
public void actionPerformed(ActionEvent ev)
{
    String cmd = ev.getActionCommand();

    if(cmd.equals(QUIT)) {
	processQuit();
    }
    else if(cmd.equals(OPEN)) {
	processOpen();
    }
    else if(cmd.equals(SCAN)) {
	processScan();
    }
    else if(cmd.equals(SAVE)) {
	processSave();
    }
    else if(cmd.equals(SET_OPTIONS)) {
	processSetOptions();
    }
    else if(cmd.equals(SAVE_OPTIONS)) {
	processSaveOptions();
    }
    else if(cmd.equals(DISPLAY)) {
	JCheckBoxMenuItem check = (JCheckBoxMenuItem)ev.getSource();
	processDoDisplay(check.getState());
    }
    else if(cmd.equals(SHOW_MINI)) {
	processShowMiniBoard(showmini.getState());
    }
    else if(cmd.equals(ABOUT)) {
	processAbout();
    }
    else if(cmd.equals(MANUAL)) {
	processManual();
    }
    else if(cmd.equals(WADERSITE)) {
	processWaderSite();
    }
    else {
	ToolBox.warn("Invalid command " + cmd);
    }
}
//______________________________________________________________________________
/**
 * Processes the Quit option.
 */
final private void processQuit()
{
    if(Controller.instance().isSiteNotSaved()) {
	  if(Confirmer.instance().ask("Save current site before ?")) {
	      processSave();
	  }
    }
    if(Confirmer.instance().ask("Really want to quit ?")) {
	ToolBox.warn("See you later");
	System.exit(0);
    }
}
//______________________________________________________________________________
/**
 * Processes the Open option.
 * <Br>
 * It displays the file chooser allowing the user to select a file
 * containing a description of a web site. These files are supposed
 * to have the extension ".site".
 * @see FileChooser
 * @see SiteFileFilter
 */
final private void processOpen()
{
    FileChooser fc = FileChooser.instance();
    fc.setDialogTitle("Open file");
    fc.setFileFilter(SiteFileFilter.instance());
    fc.setMultiSelectionEnabled(false);

    int rc = FileChooser.instance().display("Open");
    if(rc != JFileChooser.APPROVE_OPTION) return;

    controller.start(Controller.OPEN);
}
//______________________________________________________________________________
/**
 * Processes the Save option.
 * <Br>
 * It displays the file chooser allowing the user to select a file
 * use to store the description of a web site. These files are supposed
 * to have the extension ".site".
 * @see FileChooser
 * @see SiteFileFilter
 */
final protected void processSave()
{
    FileChooser fc = FileChooser.instance();
    fc.setDialogTitle("Save file");
    fc.setFileFilter(SiteFileFilter.instance());
    fc.setMultiSelectionEnabled(false);

    int rc = FileChooser.instance().display("Save");
    if(rc != JFileChooser.APPROVE_OPTION) return;

    /**
     * Finally write data.
     */
    File file = fc.getSelectedFile();
    if(file.exists()) {
	if(! Confirmer.instance().ask("Overwrite " + file.getName() + " ?")) {
	    return;
	}
    }

    controller.start(Controller.SAVE);
}
//______________________________________________________________________________
/**
 * Processes the SetOptions option.
 * @see OptionPane#display
 */
final private void processSetOptions()
{
    OptionPane.instance().display();
}
//______________________________________________________________________________
/**
 * Processes the SaveOptions option.
 * @see OptionPane#save
 */
final private void processSaveOptions()
{
    OptionPane.instance().save();
}
//______________________________________________________________________________
/**
 * Processes the DoDisplay option.
 * @param status the new condition.
 */
final private void processDoDisplay(boolean status)
{
    NavePane.instance().setDoDisplay(status);
}
//______________________________________________________________________________
/**
 * Processes the ShowMiniBoard option.
 * @param status the new condition.
 */
final private void processShowMiniBoard(boolean status)
{
    NaveBoard.instance().setVisible(status);
}
//______________________________________________________________________________
/**
 * Processes the Scan option.
 */
final private void processScan()
{
    ScanWindow.instance().display();
}
//______________________________________________________________________________
/**
 * Processes the About option.
 */
final private void processAbout()
{
    VersionWindow.instance().display();
}
//______________________________________________________________________________
/**
 * Processes the Manual option.
 */
final private void processManual()
{
    HelpWindow.instance().display();
}
//______________________________________________________________________________
/**
 * Processes the Wader Site option.
 * <br>
 * The WebWader home page is displayed by the current displayer.
 * @see NavePane#getDisplayer
 */
final private void processWaderSite()
{
    Displayer displayer = DisplayersBag.instance().getDisplayer();
    if(displayer == null) displayer = SwingDisplayer.instance();
    
    try {
	final URL home =
        new URL("http://jeanpaul.lefevre.free.fr/java/webwader/index.html");
	displayer.displayPage(home);
    }
    catch(Exception ex) {
	ErrorWindow.instance().display("Can't show WebWader home page !");
    }
}
//______________________________________________________________________________
}

