//______________________________________________________________________________

//	The Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader.gui;

import org.ariane.tools.ToolBox;
/**
 * The list of available displayers for the WebWader.
 * <br>
 * It manages the collection of possible displayers and the current displayer.
 * <p>
 * To add a new displayer :
 * <ul>
 * <li>Implement the code, update Makefile.am.
 * <li>Add an index definition.
 * <li>Append the name in DISPLAYERS.
 * <li>Add a case in <code>setDisplayer()</code>, Unix or Window.
 * <li>Add a choice in <code>webwader.properties<code>.
 * </ul>
 *
 * @see org.demo.webwader.rbi.DisplayersBag
 * @version $Id: DisplayersBag.java,v 3.14 2001/06/19 19:21:12 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */
//______________________________________________________________________________

class DisplayersBag extends AbstractDisplayersBag {
  /**
   * @serial The unique instance.
   */
private static DisplayersBag  bag = null;
    /**
     * @serial The index of the displayer local SwingDisplayer.
     */
private static final int LOCAL_SWING = 0;
    /**
     * @serial The index of the displayer Netscape.
     */
private static final int NETSCAPE = LOCAL_SWING + 1;
    /**
     * @serial The index of the displayer IExplorer.
     */
private static final int IEXPLORER = NETSCAPE + 1;
    /**
     * @serial The index of the displayer Mozilla.
     */
private static final int MOZILLA = IEXPLORER + 1;
    /**
     * @serial The index of the displayer EmacsW3.
     */
private static final int EMACS = MOZILLA + 1;
    /**
     * @serial The index of the displayer PrintURL.
     */
private static final int PRINT_URL = EMACS + 1;
    /**
     * @serial The index of the displayer remote Swing displayer.
     */
private static final int RBI_SWING = PRINT_URL + 1;
    /**
     * @serial The index of the displayer IceBorwser.
     */
private static final int ICE = RBI_SWING + 1;
    /**
     * @serial The index of the displayer NetClue.
     */
private static final int CLUE = ICE + 1;
    /**
     * @serial The index of the displayer Host.
     */
private static final int HORST = CLUE + 1;
    /**
     * @serial The index of the displayer based on Xml RPC.
     */
private static final int XMLRPC = HORST + 1;
    /**
     * @serial The index of the displayer Opera.
     * Last OS independant component.
     */
private static final int OPERA = XMLRPC + 1;

    /**
     * @serial The index of the displayer NeoPlanet.
     * Last Win32 component.
     */
private static final int NEOPLANET = OPERA + 1;

    /**
     * @serial The index of the displayer Konqueror.
     */
private static final int KONQUEROR = OPERA + 1;
    /**
     * @serial The index of the displayer Galeon.
     */
private static final int GALEON = KONQUEROR + 1  ;
    /**
     * @serial The index of the displayer SkipStone.
     */
private static final int SKIPSTONE = GALEON + 1;
    /**
     * @serial The index of the displayer Socket.
     * Last Unix component.
     */
private static final int SOCKET = SKIPSTONE + 1;

//______________________________________________________________________________
/**
 * Creates the bag.
 * <br>
 * The array of names is initialized.
 */
private DisplayersBag()
  {
    if(ToolBox.isUnderUnix()) {
	 DISPLAYERS              = new String[SOCKET + 1];
	 DISPLAYERS[KONQUEROR]   = Konqueror.NAME;
	 DISPLAYERS[GALEON]      = Galeon.NAME;
	 DISPLAYERS[SKIPSTONE]   = SkipStone.NAME;
	 DISPLAYERS[SOCKET]      = SocketDisplayer.NAME;
    }
    else if(ToolBox.isUnderWindows()) {
	 DISPLAYERS              = new String[NEOPLANET + 1];
	 DISPLAYERS[NEOPLANET]   = NeoPlanet.NAME;
    }
    else {
	DISPLAYERS               = new String[1];
   }

    DISPLAYERS[LOCAL_SWING]  = SwingDisplayer.NAME;
    DISPLAYERS[NETSCAPE]     = NetscapeWindows.NAME;
    DISPLAYERS[MOZILLA]      = MozillaUnix.NAME;
    DISPLAYERS[OPERA]        = OperaUnix.NAME;
    DISPLAYERS[EMACS]        = EmacsW3.NAME;
    DISPLAYERS[PRINT_URL]    = PrintURL.NAME;
    DISPLAYERS[IEXPLORER]    = IExplorerWindows.NAME;
    DISPLAYERS[RBI_SWING]    = RemoteSwingDisplayer.NAME;
    DISPLAYERS[XMLRPC]       = XmlDisplayer.NAME;
    DISPLAYERS[ICE]          = 
	               org.demo.webwader.rbi.DisplayersBag.ICE_DISPLAYER_NAME;
    DISPLAYERS[CLUE]         = 
	               org.demo.webwader.rbi.DisplayersBag.CLUE_DISPLAYER_NAME;
    DISPLAYERS[HORST]        =
	               org.demo.webwader.rbi.DisplayersBag.HORST_DISPLAYER_NAME;

  }
//______________________________________________________________________________
/**
 * Gets an unique instance of the DisplayersBag.
 * <br>
 * It is a singleton.
 * @return the unique instance.
 */
protected static DisplayersBag instance()
  {
    if(bag == null) bag = new DisplayersBag();

    return bag;
  }
//______________________________________________________________________________
/**
 * Sets the current tool used to display pages.
 * <br>
 * There are two different lists possible depending on the OS.
 *
 * @param index the index of the new displayer.
 * @return true if the displayer is actually changed.
 * @see #setWindowsDisplayer
 * @see #setUnixDisplayer
 */
final protected boolean setDisplayer(int index)
{
    if(index != LOCAL_SWING)
	WebWindow.instance().setVisible(false);

    boolean done = false;

    if(ToolBox.isUnderUnix()) {
	done = setUnixDisplayer(index);
    }
    else if(ToolBox.isUnderWindows()) {
	done = setWindowsDisplayer(index);
    }

    if(ToolBox.debug) {
	if(displayer == null) {
	    ToolBox.warn("Displayer undefined");
	}
	else {
	    ToolBox.warn("Displayer : " + displayer.getName());
	}
    }

    return done;
}
//______________________________________________________________________________
/**
 * Sets the current tool used to display pages.
 * <br>
 * It is used under Windows.
 * @param index the index of the new displayer.
 * @return true if the displayer is actually changed.
 * @see BrowserWindows
 */
final private boolean setWindowsDisplayer(int index)
{
    switch(index) {
    case LOCAL_SWING :
	displayer = SwingDisplayer.instance();
	break;

    case RBI_SWING :
	displayer = RemoteSwingDisplayer.instance();
	break;

    case NETSCAPE :
	displayer = NetscapeWindows.instance();
	break;

    case MOZILLA :
	displayer = MozillaWindows.instance();
	break;

    case EMACS :
	displayer = EmacsW3.instance();
	break;

    case PRINT_URL :
	displayer = PrintURL.instance();
	break;

    case IEXPLORER :
	displayer = IExplorerWindows.instance();
	break;

    case OPERA :
	displayer = OperaWindows.instance();
	break;

    case NEOPLANET :
	displayer = NeoPlanet.instance();
	break;

    case ICE :
	displayer = IceDisplayer.instance();
	break;

    case CLUE :
	displayer = ClueDisplayer.instance();
	break;

    case HORST :
	displayer = HorstDisplayer.instance();
	break;

    case XMLRPC :
	displayer = XmlDisplayer.instance();
	break;

    default :
	ToolBox.warn("Displayer unknown : " + index);
	return false;
    }

    return true;
}
//______________________________________________________________________________
/**
 * Sets the current tool used to display pages.
 * <br>
 * It is used under Unix.
 * @param index the index of the new displayer.
 * @return true if the displayer is actually changed.
 * @see Displayer
 */
final private boolean setUnixDisplayer(int index)
{
    switch(index) {
    case LOCAL_SWING :
	displayer = SwingDisplayer.instance();
	break;

    case RBI_SWING :
	displayer = RemoteSwingDisplayer.instance();
	break;

    case PRINT_URL :
	displayer = PrintURL.instance();
	break;

    case NETSCAPE :
	displayer = NetscapeUnix.instance();
	break;

    case MOZILLA :
	displayer = MozillaUnix.instance();
	break;

    case OPERA :
	displayer = OperaUnix.instance();
	break;

    case IEXPLORER :
	displayer = IExplorerUnix.instance();
	break;

    case EMACS :
	displayer = EmacsW3.instance();
	break;

    case KONQUEROR :
	displayer = Konqueror.instance();
	break;

    case GALEON :
	displayer = Galeon.instance();
	break;

    case SOCKET :
	displayer = SocketDisplayer.instance();
	break;

    case SKIPSTONE :
	displayer = SkipStone.instance();
	break;

    case ICE :
	displayer = IceDisplayer.instance();
	break;

    case CLUE :
	displayer = ClueDisplayer.instance();
	break;

    case HORST :
	displayer = HorstDisplayer.instance();
	break;

    case XMLRPC :
	displayer = XmlDisplayer.instance();
	break;

    default :
	ToolBox.warn("Displayer unknown : " + index);
	return false;
    }

    return true;
}
//______________________________________________________________________________
}








