//______________________________________________________________________________

//	The Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader.gui;

import org.demo.webwader.*;

import org.ariane.exec.RunnableTask;
import org.ariane.tools.ToolBox;
import org.demo.webwader.Resources;
import org.ariane.tools.InvalidDataException;
import javax.swing.SwingUtilities;
import java.io.*;

/**
 * The Controller manages the connection between the main application
 * and the GUI.
 * <Br>
 * It is a singleton which is runnable.

 * @version $Id: Controller.java,v 3.6 2000/11/12 10:18:49 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */
//______________________________________________________________________________

public class Controller extends RunnableTask {
  /**
   * @serial The unique instance.
   */
private static Controller  controller = null;
  /**
   * @serial The command NoOp.
   */
protected final static int NOOP = -1;
  /**
   * @serial The command Open.
   */
protected final static int OPEN = 1;
  /**
   * @serial The command Save.
   */
protected final static int SAVE = 2;
  /**
   * @serial The command Close.
   */
protected final static int CLOSE = 3;
  /**
   * @serial The current command.
   */
private  Integer command = null;
  /**
   * @serial The current web site.
   */
private  WebSite site = null;
  /**
   * @serial The flag indicated that the site is not written.
   */
private  boolean notsaved = false;

//______________________________________________________________________________
/**
 * Creates the Controller.
 */
private Controller()
  {
      super("Controller");
      this.command = new Integer(NOOP);
  }
//______________________________________________________________________________
/**
 * Gets an unique instance of the Controller.
 * <Br>
 * It is a singleton.
 * @return the unique instance.
 */
protected static Controller instance()
  {
    if(controller == null) controller = new Controller();

    return controller;
  }
//______________________________________________________________________________
/**
 * Gets the current web site.
 * @return the web site.
 */
final protected WebSite getWebSite()
  {
    return site;
  }
//______________________________________________________________________________
/**
 * Sets the current web site to a fresh new one.
 * <P>
 * The tree pane is updated. The iterator of the nave pane is reset.
 * @param the web site.
 * @see #setWebSite
 */
final protected void setNewWebSite(WebSite site)
  {
      setWebSite(site);
      notsaved  = true;
      NaveBoard.instance().display(true);
  }
//______________________________________________________________________________
/**
 * Sets the current web site.
 * <P>
 * The tree pane is updated. The iterator of the nave pane is reset.
 * The former site is saved if necessary.
 * @param the web site.
 * @see #setNewWebSite
 */
final protected void setWebSite(WebSite site)
  {
      if(this.site != null && this.site != site && notsaved) {
	  if(Confirmer.instance().ask("Save current site before ?")) {
	      MainFrame.instance().processSave();
	  }
      }

      this.site = site;
      notsaved  = false;
      NaveBoard.instance().display(true);

      TreePane.instance().displaySiteMap(site);
      NavePane.instance().resetIterator();
  }
//______________________________________________________________________________
/**
 * Saves data to the selected file.
 * @see WebSite#write
 * @throw IOException if the file cannot be written.
 */
final private void saveFile() throws IOException
  {
      if(site == null) throw new InvalidDataException("Can't save null site");

      File file   = FileChooser.instance().getSelectedFile();
      String path = file.getPath();
      String base = path;

      int index   = path.lastIndexOf(WebSite.SUFFIX);
      if(index < 0) index = path.lastIndexOf(".xml");
      if(index > 0) base = path.substring(0, index);

      site.write(base + WebSite.SUFFIX);
      notsaved  = false;

      if(Resources.instance().get(Resources.PREFIX + "XML.Save",
				                      ToolBox.hasXML())) {
	  String xmlname = base + ".xml";
	  site.writeXML(xmlname);
      }
  }
//______________________________________________________________________________
/**
 * Opens the selected file.
 * @throw IOException if the file cannot be read.
 * @see WebSite#read
 * @see #openFile(File)
 * @throw IOException if the file cannot be read.
 */
final private void openFile() throws IOException
  {
      openFile(FileChooser.instance().getSelectedFile());
  }
//______________________________________________________________________________
/**
 * Opens the selected file.
 * @return true if the current site was not saved.
 */
final protected boolean isSiteNotSaved()
  {
      return notsaved;
  }
//______________________________________________________________________________
/**
 * Opens the specified file.
 * @param file the file to open.
 * @see WebSite#read
 * @see XmlLoader#loadSite
 * @see #setWebSite
 * @throw IOException if the file cannot be read.
 */
    final protected void openFile(File file) throws IOException
  {
      String path = file.getPath();

      String name = file.getName();
      int index   = name.lastIndexOf(WebSite.SUFFIX);
      WebSite s   = null;

      if(index > 0) {
	  name = name.substring(0, index);
	  s    = new WebSite(name);
	  s.read(path);
      }
      else if((index = name.lastIndexOf(".xml")) > 0 && ToolBox.hasXML()) {
	  s = XmlLoader.instance().loadSite(path);
      }
      else {
	  throw new IOException("Unknow file type " + name);
      }

      setWebSite(s);
  }
//______________________________________________________________________________
/**
 * Creates then launches the thread executing the specified command.
 * @param command the command to execute.
 * @see #start()
 */
final protected void start(int command)
  {
      synchronized (this.command) {
	  this.command = new Integer(command);
	  done         = false;
      }
      start();
  }
//______________________________________________________________________________
 /**
   * Launches the tasks.
   * <Br>
   * Depending upon the value of the current command a specific action
   * is performed.
   */
public void run()
{
    if(false) ToolBox.warn("Controller " + command);

    try {Thread.sleep(200);} // Give GUI events a chance to refresh the screen.
    catch(Exception ex) {}

    try {
	switch(command.intValue()) {
	case NOOP :
	    ToolBox.warn("No Op command : " + command);
	    break;

	case OPEN :
	    MainFrame.inform("Opening file ...");
	    openFile();
	    break;

	case SAVE :
	    MainFrame.inform("Saving to file ...");
	    saveFile();
	    break;

	default :
	    ToolBox.warn("Unknown command : " + command);
	    break;
	}
    }
    catch(IOException ex) {
	ErrorWindow.instance().display("Can't process file !", ex);
    }
    catch(BadConfigurationException ex) {
	ErrorWindow.instance().display("Bad configuration !", ex);
    }
    catch(Exception ex) {
	ErrorWindow.instance().display(ex.getClass().getName() + " !", ex);
    }

    done   = true;
    MainFrame.clearMessage();

    if(false) ToolBox.warn("Controller done");
}
//______________________________________________________________________________
}


