//______________________________________________________________________________

//	The Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader.gui;

import org.ariane.tools.ToolBox;
import org.demo.webwader.Resources;
import org.ariane.tools.MiniSet;
import java.net.URL;
import java.util.StringTokenizer;

/**
 * A object used to factorize some common methods for the concrete displayers.
 * <p>
 * It manages the list of suffix indicating which documents should be
 * displayed.
 * @version $Id: AbstractDisplayer.java,v 1.3 2000/10/25 16:15:05 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */
//______________________________________________________________________________

abstract class AbstractDisplayer implements Displayer {
    /**
     * @serial The set of suffix to display.
     */
static private MiniSet suffix_set = null;

//______________________________________________________________________________
/**
 * Creates the displayer.
 * <br>
 * It initializes the list of suffix to handle.
 */
protected AbstractDisplayer()
  {
      if(suffix_set == null) loadSuffixSet();
  }
//______________________________________________________________________________
/**
 * Decides whether the document should be displayed or not.
 * <br>
 * The decision is based on the suffix of the URL.
 * @return true is the page has been displayed.
 */
final protected boolean isToShow(URL url)
{
    if(url == null) return false;

    String suffix = getSuffix(url);

    if(suffix == null || indexOf(suffix) < 0)
	return false;

    return true;
}
//______________________________________________________________________________
/**
 * Gets the suffix at this end of this URL.
 * <br>
 * The suffix is in lower case without the dot.
 * For 'http://host/path/Image.Gif' it returns 'gif'
 * If no suffix is found, it returns the string "void".
 * The internal representation of the string is returned.
 * @return the suffix.
 * @see String#intern
 */
private String getSuffix(URL url)
{
    String file = url.getFile();

    int k = file.lastIndexOf('?');    // The query separator
    if(k < 0) k = file.length() - 1;  // Discard arguments

    int i = file.lastIndexOf('/', k); // The URL separator
    int j = file.lastIndexOf('.', k); // The suffix separator

    if(j <= i || j < 0) return "void";

    return file.substring(j + 1).toLowerCase().intern();
}
//______________________________________________________________________________
  /**
   * Returns the position of an element in the set.
   * @param suffix the element to find.
   * @return the position.
   */
private int indexOf(String suffix)
  {
      int size = suffix_set.size();

      for(int i = 0; i < size; i++) {
	  if(suffix_set.get(i) == suffix) // It should work with interned str.
	      return i;
      }

      return -1;
  }
//______________________________________________________________________________
/**
 * Returns the list of suffix defining the types shown by the displayer.
 * <br>
 * The list is built by concatenation of the suffix separated by white space.
 * @return the list of suffix.
 */
final protected String getListOfSuffix()
{
      int size         = suffix_set.size();
      StringBuffer buf = new StringBuffer();

      for(int i = 0; i < size; i++) {
	  buf.append(suffix_set.get(i)).append(' ');
      }

      return buf.toString();
}
//______________________________________________________________________________
/**
 * Changes the list of suffix.
 * <br>
 * The suffix are changed in lower case, any leading dot is discarded.
 * @param list the new list of suffix.
 */
final protected void setListOfSuffix(String list)
{
    suffix_set = new MiniSet(20);

    if(list == null || list.length() < 1) {
	return;
    }

    StringTokenizer stk = new StringTokenizer(list);

    while(stk.hasMoreTokens()) {

	String suffix = stk.nextToken();
	int i  = suffix.lastIndexOf('.');

	if(i >= (suffix.length() - 1)) // Case 'suf.'
	    continue;
	else if(i >= 0)                // Case 'junk.suf'
	    suffix = suffix.substring(i + 1);

	suffix = suffix.toLowerCase().intern();

	if(indexOf(suffix) < 0)        // Add it if not already there
	    suffix_set.add(suffix);
    }
}
//______________________________________________________________________________
/**
 * Initializes the list of suffix.
 * <br>
 * The list of suffix is taken into the properties.
 * The name of the property is "TypesToDisplay".
 * Suffix "html", "htm" are automatically appended.
 * @see #setListOfSuffix
 */
private void loadSuffixSet()
{
    String list = Resources.instance().get(Resources.PREFIX + "TypesToDisplay");

    setListOfSuffix(list);
    suffix_set.add("html");
    suffix_set.add("htm");
}
//______________________________________________________________________________
}



