//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader;

import org.xml.sax.*;
import org.xml.sax.helpers.ParserFactory;
import java.io.*;
import java.util.Properties;
import org.ariane.tools.ToolBox;
import org.ariane.tools.Nobject;

/**
 * XmlLoader : a object used to load sites from an XML file.
 *
 * @see WebSite
 * @see XmlHandler
 * @version $Id: XmlLoader.java,v 1.6 2000/11/12 10:23:06 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */

public class XmlLoader extends Nobject {
  /**
   * @serial The unique instance.
   */
private static XmlLoader loader = null;
  /**
   * @serial The file describing the web site.
   */
private String filename;

//______________________________________________________________________________
  /**
   * Starts the program.
   * @see #printUsage
   */
public static void main (String args[])
  {
      XmlLoader prog = instance();

      if (! ToolBox.hasXML()) {
	  ToolBox.warn("Can't find XML classes");
	  return;
      }
      else if (! prog.parseArg(args)) {
	  ToolBox.warn("Can't parse command line");
	  return;
      }
      else if (! prog.run()) {
	  ToolBox.warn("Can't run " + prog.getName());
	  return;
      }
  }
//______________________________________________________________________________
/**
 * Creates the handler.
 */
private	XmlLoader()
  {
      super("XmlLoader");
  }
//______________________________________________________________________________
/**
 * Gets an unique instance of the loader.
 * <Br>
 * It is a singleton.
 * @return the unique instance.
 */
public static XmlLoader instance()
  {
    if(loader == null) loader = new XmlLoader();

    return loader;
  }
//______________________________________________________________________________
/**
 * Actually starts the program.
 * <Br>
 * The description of the site is read in an XML file.
 * The updated content is saved in a file.
 */
public	boolean run()
  {
      try {
	  if(ToolBox.verbose) ToolBox.warn("Reading from " + filename);
	  WebSite site = loadSite(filename);

	  String output = ToolBox.markAsTemp(filename);
	  site.writeXML(output);

	  output = site.getName() + ".site~";
	  site.write(output);
      }
      catch (java.io.IOException ex) { 
	  if(ToolBox.debug) ex.printStackTrace();
	  ToolBox.warn("Can't read or write site", ex);
	  return false;
      }
      catch (Exception ex) {
	  if(ToolBox.debug) ex.printStackTrace();
	  ToolBox.warn("Can't handle site", ex);
	  return false;
      }

      return true;
  }
//______________________________________________________________________________
/**
 * Reads the site described by the XML file.
 * <br>
 * The name of the parser is given in the property :
 * <code>org.xml.sax.parser</code>.
 * @param filename the file to read.
 * @return the built web site.
 * @throw IOException if the file cannot be read.
 * @see ParserFactory#makeParser
 * @see Parser#parse
 */
final public WebSite loadSite(String filename) throws IOException
  {
      Parser parser = null;
      WebSite site  = new WebSite();
      /**
       * Copy the property from the jvs to the system table.
       */
      String key = "org.xml.sax.parser";
      String val = Resources.instance().getProperties().getProperty(key);
      if(val == null)throw new BadConfigurationException("No value for " + key);
      System.getProperties().put(key, val);
  
      try {
	  parser = ParserFactory.makeParser();
      }
      catch (Exception ex) {
	  ex.printStackTrace();
	  throw new RuntimeException(ex.getMessage());
      }

      XmlHandler handler = new XmlHandler(site);
      parser.setDocumentHandler(handler); 
      parser.setErrorHandler(handler);

      InputSource src =  new InputSource(new FileReader(filename));
      try {
	  parser.parse(src);
      }
      catch(SAXException ex) {
	  throw new InvalidDataException(ex.getMessage());
      }

      return site;
  }
//______________________________________________________________________________
/**
 * Prints how to use the program.
 */
final private void printUsage()
  {
      System.out.println(
      "Usage   : java org.demo.webwader.XmlLoader [options] filename");
      System.out.println(
      "Options : [-v|-d|-q][-V][-h]");
  }
//______________________________________________________________________________
/**
 * Parses the command line
 * Usage   : [-q|v|d][-V][-h] filename
 * @param args the command line.
 */
public	boolean parseArg(String args[])
  {
    int i     = 0;
    int error = 0;

    while(i < args.length && args[i].startsWith("-")) {

      String arg = args[i++];
      int j;

      for (j = 1; j < arg.length(); j++) {

	char flag = arg.charAt(j);

	switch (flag) {
	case 'V':
	  VersionInfo.print();
	  break;

	case 'q':
	  ToolBox.verbose = false;
	  ToolBox.debug   = false;
	  break;

	case 'v':
	  ToolBox.verbose = true;
	  ToolBox.debug   = false;
	  break;

	case 'd':
	  ToolBox.verbose = true;
	  ToolBox.debug   = true;
	  break;

	default	:
	  error++;
	  break;
	}
      }
    }

    if(i < args.length) {
	filename = args[i];
    }
    else {
	printUsage();
	error++;
    }

    return (error == 0);
  }
//______________________________________________________________________________
}





