//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader;

import org.xml.sax.*;
import java.io.*;
import org.ariane.tools.ToolBox;

/**
 * XmlHandler : a convenient object used to read xml descriptions.
 * @see WebSite
 * @version $Id: XmlHandler.java,v 1.3 2000/11/10 20:38:11 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */

public class XmlHandler extends HandlerBase {
    /**
     * @serial The possible values for the mark.
     * It indicates which tag has just been detected in the file.
     */
final static int UNDEF   = -2;
final static int WEBSITE =  0;
final static int WEBPAGE =  1;
final static int TITLE   =  2;
final static int URL     =  3;
    /**
     * @serial the current mark detected.
     */
private int mark;
    /**
     * @serial the site being read.
     */
private WebSite site;
    /**
     * @serial the directory url of the top node.
     */
private String dir_url = null;
    /**
     * @serial the current node.
     */
private WebNode node;

//______________________________________________________________________________
/**
 * Creates the handler.
 * @param site the site to be loaded.
 */
public	XmlHandler(WebSite site)
  {
      this.site = site;
      mark      = UNDEF;
      node      = null;
  }
//______________________________________________________________________________
/**
 * Processes an element.
 *
 * @param tag the name of the element.
 * @param attrs the list of attributes.
 * @throw SAXException if the input file is not correct.
 */
public void startElement(String tag, AttributeList attrs) throws SAXException
  {
      if(tag.equalsIgnoreCase("WebSite")) {
	  mark = WEBSITE;
	  site.setName(attrs.getValue("name"));
      }
      else if(tag.equalsIgnoreCase("WebPage")) {
	  mark = WEBPAGE;

	  try {
	      Locator l = Locator.makeLocator(dir_url, attrs.getValue("path"));
	      WebPage page = new WebPage(l);
	      page.setName(attrs.getValue("name"));

	      WebNode n = new WebNode(page);

	      if(site.getTopNode() == null) {
		  site.setTopNode(n);
		  node = n;
		  return;
	      }
	      /**
	       * The newly created node is added to the current parent node.
	       * Then it is considered as the parent of the future childre.
	       */
	      node.add(n);
	      node = n;
	  }
	  catch(InvalidLocatorException ex) {
	      throw new SAXException(ex);
	  }
      }
      else if(tag.equalsIgnoreCase("URL")) {
	  mark = URL;
      }
      else if(tag.equalsIgnoreCase("Title")) {
	  mark = TITLE;
      }
  }
//______________________________________________________________________________
/**
 * Processes content of an element.
 * @param buf the array of characters read.
 * @param offset the start position in the character array
 * @param len the number of characters to use from the character array.
 * @throw SAXException if the input file is not correct.
 */
public void characters (char buf[], int offset, int len) throws SAXException
  {
      String str = new String(buf, offset, len);

      try {
	  switch(mark) {
	      /**
	       * The title of a page can be stored and retrieved in the xml.
	       */
	  case TITLE   : node.setTitle(str); break;
	      /**
	       * The url of the site. The rest of the pages are
	       * referenced from this location.
	       */
	  case URL     : dir_url = str.trim(); break;
	  }
      }
      catch(InvalidDataException ex) {
	  if(ToolBox.debug) {
	      String line = new String(buf, offset, len);
	      System.out.println("Invalid data : " + line);
	  }
	  throw new SAXException(ex);
      }
  }
//______________________________________________________________________________
/**
 * Processes a closing tag.
 *
 * @param tag the name of the element.
 * @throw SAXException if the input file is not correct.
 */
public void endElement(String tag) throws SAXException
  {
      if(tag.equalsIgnoreCase("WebPage")) {
	  /**
	   * Processing of the current node is over.
	   * Its parent becomes the new current node.
	   */
	  node = (WebNode)node.getParent();
      }
  }
//______________________________________________________________________________
/**
 * Converts this object to String.
 * @return the string.
 */
public String toString()
  {
      return "XmlHandler";
  }
//______________________________________________________________________________
/**
 * Processes fatal errors.
 * @param sex the exception to handle.
 */
public void fatalError(SAXParseException sex)
{
    error(sex);
}
//______________________________________________________________________________
/**
 * Processes errors.
 * @param sex the exception to handle.
 */
public void error(SAXParseException sex)
{
    explain(sex);
    Exception ex = sex.getException();
    if(ex == null)
	ToolBox.warn("XML parse   error", sex);
    else {
	ToolBox.warn("XML parse   error", sex);
	ToolBox.warn("XML handler error", ex);
    }
}
//______________________________________________________________________________
/**
 * Processes warnings.
 * @param sex the exception to handle.
 */
public void warning(SAXParseException sex)
{
    explain(sex);
    Exception ex = sex.getException();
    if(ex == null)
	ToolBox.warn("XML handler warning", sex);
    else
	ToolBox.warn("XML handler warning", ex);
}
//______________________________________________________________________________
/**
 * Prints details about the errot.
 * @param sex the exception to handle.
 */
final private void explain(SAXParseException sex)
{
    ToolBox.warn("Parse error line : " + sex.getLineNumber() +
                 " (column " + sex.getColumnNumber() + ")");
}
//______________________________________________________________________________
}
