//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader;

import org.ariane.tools.*;
import java.io.*;
import java.util.*;

/**
 * WebSite : a collection of connected html pages.
 * <P>
 * A site is defined by the topmost node of a tree of pages.
 * This object proposes methods to read, write, etc.
 * a site.
 *
 * @see SiteScanner
 * @see CheckSite
 * @see WebPage
 * @version $Id: WebSite.java,v 3.9 2001/01/15 16:36:59 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */

public	class WebSite extends Nobject {
  /**
   * @serial The top of the hierarchy of web pages.
   */
private WebNode top;
  /**
   * @serial The url of the top node directory.
   */
private String top_dir_url = null;
    /**
     * @serial The extension used for files containing site map.
     * @see org.demo.webwader.gui.SiteFileFilter
     */
static final public String SUFFIX = ".site";
    /**
     * @serial The white space character.
     */
static final public String SP = " ";
    /**
     * @serial The indentation used in the XML output.
     */
static private String indent = null;

//______________________________________________________________________________
/**
 * Creates the object.
 * <P>
 * The site is given the default name.
 */
public	WebSite()
  {
      this("WebSite");
  }
//______________________________________________________________________________
/**
 * Creates the object.
 * @param name the name of the site.
 * @see SiteScanner#loadIgnoredList
 */
public	WebSite(String name)
  {
      super(name);
      top  = null;
  }
//______________________________________________________________________________
/**
 * Sets the topmost node of this site.
 * @param the node.
 */
final public void setTopNode(WebNode top)
    {
	this.top = top;
    }
//______________________________________________________________________________
/**
 * Returns the topmost node of this site.
 * @return the node.
 */
final public WebNode getTopNode()
    {
	return top;
    }
//______________________________________________________________________________
/**
 * Returns the url of the directory of the top node.
 * @return the url.
 */
final protected String getTopDirURL()
    {
	if(top_dir_url != null)
	    return top_dir_url;
	else if(top == null)
	    return null;
	else {
	    Locator l   = top.getLocator();
	    top_dir_url = l.getSchemeServerSpec() + l.getParentPath();
	}

	return top_dir_url;
    }
//______________________________________________________________________________
/**
 * Creates an enumeration of the web pages.
 * <Br>
 * The content of the site is stored in a plain list. A bidirectionnal
 * iterator on this collection is made available.
 *
 * @param order the order used to create the list of pages.
 * @see PlainList
 * @see #getEnumeration
 */
final public ListIterator getIterator(int order)
  {
      if(top == null) return null;

      PlainList list = new PlainList(getEnumeration(order));

      return list.listIterator();
  }
//______________________________________________________________________________
/**
 * Creates an enumeration of the web pages.
 * <Br>
 * The enumeration can be built in 3 different ways : preorder,
 * postorder and breadthFirst.
 * If order is unknown it returns a enumeration to a breadthFirst list.
 *
 * @param order the order used to create the list of pages.
 * @return the enumeration;
 */
final private Enumeration getEnumeration(int order)
  {
      switch(order) {
      case WebNode.PREORDER    :
	  return top.preorderEnumeration();
      case WebNode.POSTORDER   :
	  return top.postorderEnumeration();
      default :
	  return top.breadthFirstEnumeration();
      }
  }
//______________________________________________________________________________
/**
 * Reads the content of this site from a plain ASCII file.
 * <P>
 * Comments can be inserted in the file. They follow a '#' character.
 * Empty lines are ignored.
 * <Br>
 * The first line must give the name of the site.
 * The second one the url pointing to the site.
 * The next lines must give the description of the tree.
 * A line must have a parent node at the left hand side of a ':' character,
 * and a list of children at the right hand side.
 * Elements of the list are separated by space characters.
 * Elements with space embedde must be quoted.
 * A node is specified by its relative pathname from the root of the site.
 * Example :
 * <P><Code>
 * name : my_site<Br>
 * url  : http://www.site.org/<Br>
 * index.html : unix.html java/index.html<Br>
 * java/index.html : java/about.html java/links.hmtl<Br>
 * ....<Br>
 * <Br></Code>
 * 
 * @param filename the path to the file to read.
 * @throw IOException if the file cannot be read or contains invalid data.
 * @see InputDataset
 * @see #write
 */
public void read(String filename) throws IOException
  {
      if(filename == null)
	  throw new IOException("Invalid null filename");

      final String SEP = ":";
      InputDataset input  = new InputDataset(filename);
      StreamReader reader = input.createReader();

      /**
       * Decode the first line which must be
       * Name : name_of_this_site
       */
      String line = reader.nextLine();
      int index   = line.indexOf(SEP);
      if(index < 1) {
	  throw new InvalidDataException("No separator found on line : "+ line);
      }
      else if(! line.substring(0, index).trim().equalsIgnoreCase("Name")) {
	  throw new InvalidDataException("No name tag found on line : " + line);
      }

      setName(line.substring(index + 1).trim());

      /**
       * Decode the second line which must be
       * URL : url_of_this_site
       */
      line  = reader.nextLine();
      index = line.indexOf(SEP);
      if(index < 1) {
	  throw new InvalidDataException("No separator found on line : " +line);
      }
      else if(! line.substring(0, index).trim().equalsIgnoreCase("URL")) {
	  throw new InvalidDataException("No url tag found on line : " + line);
      }

      String dir_url = line.substring(index + 1).trim();   

      while((line = reader.nextLine()) != null) {

	  /**
	   * On the LHS of ':' is the parent page.
	   * On the RHS are the children.
	   * The reference are relative to the root directory.
	   */
	  StringTokenizer stk = new LineTokenizer(line);
	  String ref;
	  WebNode node;

	  try {
	      ref = stk.nextToken().trim();

	      /**
	       * The first node is the top of the tree.
	       * Its directory gives the root directory.
	       */
	      if(top == null) {
		  node = new WebNode(Locator.makeLocator(dir_url, ref));
		  top  = node;
	      }
	      /**
	       * The next nodes are built relatively to the top.
	       * The parent page must have been stored. It must have
	       * at least one child.
	       */
	      else {
		  node = top.findNode(dir_url + ref);
		  if(node == null) {
		      if(ToolBox.debug) {
			  ToolBox.warn("Dir URL " + dir_url);
			  System.out.println(top.getVolume());
		      }
		      throw new InvalidDataException(
				"Page without parent " + ref);
		  }
	      }

	      stk.nextToken(SP); 		// swallow ":"
	      ref  = stk.nextToken();
	      addChild(node, dir_url, ref);
	  }
	  catch (InvalidLocatorException ex) {
	      throw new InvalidDataException("Invalid path on " + line);
	  }
	  catch (NoSuchElementException ex) {
	      throw new InvalidDataException("Invalid page on " + line);
	  }

	  /**
	   * Add other children if any.
	   */
	  while(stk.hasMoreTokens()) {
	      ref = stk.nextToken();
	      addChild(node, dir_url, ref);
	  }
      }

      input.close();
  }
//______________________________________________________________________________
/**
 * Adds a new child to a node.
 *
 * @param node the parent node.
 * @param url the server spec.
 * @param path the path of the new child.
 * @throw InvalidDataException if the page is already stored.
 * @see WebNode#WebNode
 */
final private void addChild(WebNode node, String url, String path)
                            throws InvalidDataException
{
    path.trim(); // discard any white spaces.
    WebNode child = top.findNode(path);
    if(child != null)
	throw new InvalidDataException("Page already defined " + path);

    try {
	child = new WebNode(Locator.makeLocator(url, path));
    }
    catch(InvalidLocatorException ex) {
	if(ToolBox.debug) ToolBox.warn("Url, path : " + url + ", " + path);
	throw new InvalidDataException("Invalid location " + path); 
    }

    node.add(child);
}
//______________________________________________________________________________
/**
 * Writes down the content of this site to an ASCII file.
 * <Br>
 * The name of the output file is built from the name to this site.
 * It is <Code>sitename.site~</Code>.
 * <br>
 * It will be deleted when the VM exits.
 * @throw IOException if the file cannot be written.
 * @see OutputDataset
 * @see #read
 * @see ToolBox#markAsTemp
 * @see File#deleteOnExit
 */
public void write() throws IOException
    {
	File tmp = new File(ToolBox.markAsTemp(getName() + SUFFIX));
	write(tmp.getPath());
	tmp.deleteOnExit();
    }
//______________________________________________________________________________
/**
 * Writes down the content of this site to an ASCII file.
 * <Br>
 * A header, the specification of the root, then the list of pages
 * are written in the file.
 * <Br>
 * The order used to traverse the tree is the breadth first order.
 *
 * @param filename the path to the file to write.
 * @throw IOException if the file cannot be written.
 * @see OutputDataset
 * @see #read
 */
public void write(String filename) throws IOException
  {
      if(filename == null)
	  throw new IOException("Invalid null filename");

      int order = WebNode.BREADTH_FIRST; // Otherwise I cannot read it.

      OutputDataset output  = new OutputDataset(filename);
      output.open();

      output.writeHeader("Description of Web Site : " + getName() +
			 " - WebWader version " + VersionInfo.VERSION);
      /**
       * The server spec and the common directory is written.
       */
      String dir = getTopDirURL();
      if(! dir.endsWith("/")) { // Case http://www.site.org
	  dir += "/";           // Written as http://www.site.org/
      }
      int index = dir.length();

      output.writeLine("Name : " + getName());
      output.writeLine("URL  : " + dir);
      output.writeNL();

      for(Enumeration e = getEnumeration(order); e.hasMoreElements();) {

	  WebNode node  = (WebNode)e.nextElement();
	  /**
	   * Nodes located in an upper directory are skipped.
	   */
	  if(node.isLeaf()) continue;

	  /**
	   * The relative path to the common directory  is written.
	   */
	  StringBuffer buf = null;
	  try {
	      String str = node.getLocation();

	      if(str.length() <= index) str = ""; // Case http://www.site.org
	      else str = str.substring(index);

	      buf = new StringBuffer(str);
	  }
	  catch(StringIndexOutOfBoundsException ex) {
	      ToolBox.warn("Node out " + node.getLocation());
	      continue;
	  }
	  buf.append(" :");

	  for(Enumeration ec = node.children(); ec.hasMoreElements();) {

	      WebNode child  = (WebNode)ec.nextElement();

	      try {
		  buf.append(' ').append(child.getLocation().substring(index));
	      }
	      catch(StringIndexOutOfBoundsException ex) {
		  ToolBox.warn("Child out " + child.getLocation()
			     + " index " + index, ex);
	      }
	  }
	  output.writeLine(buf.toString().trim());
      }

      output.writeNL();
      output.writeFooter();
      output.close();
  }
//______________________________________________________________________________
/**
 * Converts this site to String.
 * @return the string.
 */
public String toString()
  {
      StringBuffer buf = new StringBuffer("WebSite : ").append(getName());

      buf.append(" Server : ").append(top.getServerSpec());
      buf.append(" Number of leaves : " + top.getLeafCount());

      return buf.toString();
  }
//______________________________________________________________________________
/**
 * Writes down the content of this site to an XML file.
 * <Br>
 * A header, the specification of the root, then the list of pages
 * are written in the file.
 * <Br>
 * The order used to traverse the tree is the breadth first order.
 *
 * @param filename the path to the file to write.
 * @throw IOException if the file cannot be written.
 * @see #write
 */
public void writeXML(String filename) throws IOException
  {
      if(filename == null) throw new IOException("Invalid null filename");
      else if(top == null) throw new InvalidDataException("Empty site");

      FileOutputStream  output  = new FileOutputStream(filename);
      PrintWriter printer = new PrintWriter(output);

      printer.println(
      ToolBox.XMLHeader("WebWader version = '" + VersionInfo.VERSION + "'"));
      printer.println(getDocType());

      printer.println("<WebSite name = '" + getName() +
		               "' date = '" + (new Date()) + "'>");

      String dir = getTopDirURL();
      if(! dir.endsWith("/")) { // Case http://www.site.org
	  dir += "/";           // Written as http://www.site.org/
      }

      printer.println("<URL>" + dir + "</URL>\n");

      /**
       * Each node description will be indented by the following indent.
       */
      if(indent == null) {
	  int i = Resources.instance().get(Resources.PREFIX
					+ "XML.Indentation", 0);
	  if(i < 0) i = 0;
	  else if(i > 8) i = 8;
	  
	  indent = ("         ").substring(0, i);
      }

      writeXMLNode(top, dir.length(), printer, "");

      printer.println("</WebSite>");
      printer.close();
  }
//______________________________________________________________________________
/**
 * Writes recursively the content of this node to an XML file.
 *
 * @param node the node to write.
 * @parama dlen the length of the top dir url.
 * @param printer the writer to use.
 * @param tab the indentation.
 */
private void writeXMLNode(WebNode node, int dl, PrintWriter printer, String tab)
  {
      try {
	  String str = node.getLocation();

	  if(str.length() <= dl) str = ""; // Case http://www.site.org
	  else str = str.substring(dl);

	  printer.println(tab + "<WebPage name = '" + node.getName()
		              + "' path = '" + str + "'>");
      }
      catch(StringIndexOutOfBoundsException ex) {
	  ToolBox.warn("Node out " + node.getLocation());
      }
      
      printer.println(tab + "<Title>" + node.getTitle() + "</Title>");

      for(Enumeration e = node.children(); e.hasMoreElements();) {

	  WebNode child  = (WebNode)e.nextElement();
	  writeXMLNode(child, dl, printer, tab + indent);
      }

      printer.println(tab + "</WebPage>");
  }
//______________________________________________________________________________
/**
 * Creates the DTD reference string.
 * <Br>
 * It is <!DOCTYPE WebSite SYSTEM "file:JVS_HOME/lib/xml/webwader.dtd">
 *
 * @return the formatted string.
 */
final private String getDocType()
  {
      StringBuffer buf;
      String dtd = Resources.instance().getLibDirectory() + "xml/webwader.dtd";

      try { dtd = Locator.unescapeChar(Locator.makeLocator(dtd).getLocation());}
      catch(Exception ex) {}

      buf = new StringBuffer("<!DOCTYPE ").append("WebSite");
      buf.append(" SYSTEM '").append(dtd).append("'>");

      return buf.toString();
  }
//______________________________________________________________________________
}



