//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader;

import org.ariane.tools.ToolBox;
import java.net.MalformedURLException;
import java.net.URL;

/**
 * WebPage : a reference to a HTML page.
 * <P>
 * It stores the location of the html file.
 * Most of the time this object is embedded in a WebNode.
 *
 * @see WebSite
 * @see WebNode
 * @version $Id: WebPage.java,v 3.2 2000/11/04 18:16:37 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */

public class WebPage {
  /**
   * @serial The location of the html file.
   */
private Locator locator;
  /**
   * @serial The title of the html file.
   */
private String title;
  /**
   * @serial The name of this page.
   */
private String name;

//______________________________________________________________________________
/**
 * Creates the object.
 * @param locator the location of the page.
 */
public	WebPage(Locator locator)
  {
      this.locator = locator;
      title        = null;
      setName(locator);
  }
//______________________________________________________________________________
/**
 * Creates the object.
 * @param path the path to the page.
 * @throw InvalidLocatorException if the path is not correct.
 */
public	WebPage(String path) throws InvalidLocatorException
  {
      this(Locator.makeLocator(path));
  }
//______________________________________________________________________________
/**
 * Gets the location of the web page.
 * @return the location of the page.
 * @see Locator#getLocation()
 */
final public String getLocation()
  {
      return locator.getLocation();
  }
//______________________________________________________________________________
/**
 * Gets the locator of the web page.
 * @return the locator of the page.
 * @see Locator#getLocation()
 */
final public Locator getLocator()
  {
      return locator;
  }
//______________________________________________________________________________
/**
 * Gets the url of the web page.
 * @return the url of the page.
 * @see Locator#getURL()
 * @throw MalformedURLException if the location is not correct.
 */
final public URL getURL() throws MalformedURLException
  {
      return locator.getURL();
  }
//______________________________________________________________________________
/**
 * Gets the path to the web page.
 * @return the path to the page.
 * @see Locator#getPath()
 */
final public String getPath()
  {
      return locator.getPath();
  }
//______________________________________________________________________________
/**
 * Gets the server spec of this web page.
 * @return the spec.
 * @see Locator#getServerSpec()
 */
final public String getServerSpec()
  {
      return locator.getServerSpec();
  }
//______________________________________________________________________________
/**
 * Gets the title of the web page.
 * <P>
 * If the title has not yet been found in the page the path is returned.
 * @return the title of this page.
 * @see #getPath()
 */
final public String getTitle()
  {
      if(title == null) return "Title of " + getPath();
      return title;
  }
//______________________________________________________________________________
/**
 * Sets the title of the web page.
 * <P>
 * If the argument is null, the title is not changed.
 * @param title the title of this page.
 * @see #getTitle()
 */
final public void setTitle(String title)
  {
      if(title != null) this.title = title;
  }
//______________________________________________________________________________
/**
 * Converts this page to String.
 * @return the string.
 * @see #getName()
 */
public String toString()
  {
      return getName();
  }
//______________________________________________________________________________
/**
 * Gets a name identifying this page.
 * @return the name.
 * @see org.demo.webwader.gui.TreePane
 * @see #setName
 */
final public String getName()
  {
      return name;
  }
//______________________________________________________________________________
/**
 * Sets a name identifying this page.
 * @param the name.
 */
final public void setName(String name)
  {
      this.name = name;
  }
//______________________________________________________________________________
/**
 * Sets the name identifying this page.
 * <P>
 * It is the file name of the locator except when this name is something
 * like <I>index</I>. In this case it is the name of the directory.
 * It is used to display information about the node in a TreePane.
 * @param the assocated locator.
 * @see #getName()
 */
final private void setName(Locator locator)
  {
      String str = locator.getFileName();
      if(str.startsWith("index.")) {
	  str = locator.getParentName();
      }

      setName(str);
  }
//______________________________________________________________________________
}
