//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader;

import javax.swing.tree.DefaultMutableTreeNode;
import java.util.Enumeration;
import java.net.MalformedURLException;
import java.net.URL;
import org.ariane.tools.ToolBox;

/**
 * WebNode : a node of a tree storing a HTML page.
 * <P>
 * The user object of this node is a WebPage.
 * The topmost node of a tree may be used to defined a WebSite.
 *
 * @see WebPage
 * @see WebSite
 * @version $Id: WebNode.java,v 3.3 2000/11/04 18:15:58 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */

public class WebNode extends DefaultMutableTreeNode {
    /**
     * @serial A mode of enumeration.
     */
public static final int BREADTH_FIRST = 0;
    /**
     * @serial A mode of enumeration.
     */
public static final int PREORDER      = 1;
    /**
     * @serial A mode of enumeration.
     */
public static final int POSTORDER     = 2;
    /**
     * @serial The array of enumeration mode.
     */
public static final String[] ENUMERATION_MODE =
                    {"BreadthFirst", "Preorder", "Postorder"};

//______________________________________________________________________________
/**
 * Creates the node with a WebPage.
 * <Br>
 * The locator is used to create the WebPage.
 * @param locator the locator of the page.
 */
protected WebNode(Locator locator)
  {
      WebPage page = new WebPage(locator);
      setUserObject(page);
  }
//______________________________________________________________________________
/**
 * Creates the node with a WebPage.
 * <Br>
 * The path is used to create the WebPage.
 * @param path the path to the page.
 * @throw InvalidLocatorException if the path is invalid.
 */
protected WebNode(String path) throws InvalidLocatorException
  {
      WebPage page = new WebPage(path);
      setUserObject(page);
  }
//______________________________________________________________________________
/**
 * Creates a node with a page.
 * @param page the embedded page
 */
protected WebNode(WebPage page)
  {
      setUserObject(page);
  }
//______________________________________________________________________________
/**
 * Creates an empty node.
 * @see WebRoot
 */
protected WebNode()
  {
  }
//______________________________________________________________________________
/**
 * Gets the path to the web page.
 *
 * @return the path to the page.
 * @see Locator#getPath()
 */
public String getPagePath()
  {
      return ((WebPage)getUserObject()).getPath();
  }
//______________________________________________________________________________
/**
 * Gets the directory depth.
 * <P>
 * The directory depth is the number of directories between the top
 * and the subdirectory where the document of this node is.
 * It may be different form the link length.
 *
 * @return the depth or -1 if there is no path.
 * @see WebNode#getLinkDepth()
 */
final public int getDirDepth()
  {
      return getLocator().getDirDepth();
  }
//______________________________________________________________________________
/**
 * Gets the link depth.
 * <P>
 * The link depth is the number of nodes between the top
 * and this node. It may be different form the dir length.
 *
 * @return the depth.
 * @see WebNode#getDirDepth()
 */
final private int getLinkDepth()
  {
      return getPath().length;
  }
//______________________________________________________________________________
/**
 * Gets the locator of the web page.
 * @return the locator of the page.
 * @see Locator#getLocation()
 */
final public Locator getLocator()
  {
      return ((WebPage)getUserObject()).getLocator();
  }
//______________________________________________________________________________
/**
 * Gets the url of the web page.
 *
 * @return the url of  the page.
 * @see Locator#getURL()
 * @throw MalformedURLException if the location is not correct.
 */
final public URL getURL() throws MalformedURLException
  {
      return ((WebPage)getUserObject()).getURL();
  }
//______________________________________________________________________________
/**
 * Gets the location of the web page.
 *
 * @return the location of the page.
 * @see Locator#getLocation()
 */
final public String getLocation()
  {
      return ((WebPage)getUserObject()).getLocation();
  }
//______________________________________________________________________________
/**
 * Gets the server spec of the web page.
 *
 * @return the server spec of the page.
 * @see Locator#getServerSpec()
 */
final public String getServerSpec()
  {
      return ((WebPage)getUserObject()).getServerSpec();
  }
//______________________________________________________________________________
/**
 * Gets the name of the web page.
 *
 * @return the name of the page.
 * @see WebPage#getName
 */
public String getName()
  {
      return ((WebPage)getUserObject()).getName();
  }
//______________________________________________________________________________
/**
 * Gets the title of the web page.
 *
 * @return the title of the page.
 * @see WebPage#getTitle
 */
public String getTitle()
  {
      return ((WebPage)getUserObject()).getTitle();
  }
//______________________________________________________________________________
/**
 * Sets the title of the web page.
 *
 * @param title the title of the page.
 * @see WebPage#setTitle
 */
public void setTitle(String title)
  {
      ((WebPage)getUserObject()).setTitle(title);
  }
//______________________________________________________________________________
/**
 * Converts a string to the corresponding navigation order.
 * <P>
 * The conversion is case unsensitive. If the string is not recognized
 * the default mode BREADTH_FIRST is returned.
 *
 * @return the navigation mode as an int.
 */
final static public int navigationOrder(String order)
  {
      if(order == null)
	  return BREADTH_FIRST;
      else if(order.equalsIgnoreCase("Preorder"))
	  return PREORDER;
      else if(order.equalsIgnoreCase("Postorder"))
	  return POSTORDER;

      return BREADTH_FIRST;
  }
//______________________________________________________________________________
/**
 * Converts a int to the corresponding navigation order.
 * @return the navigation mode as a string.
 */
final static public String navigationOrder(int order)
  {

      switch(order) {
      case WebNode.PREORDER    :
	  return "PreOrder";

      case WebNode.POSTORDER   :
	  return "PostOrder";

      case WebNode.BREADTH_FIRST   :
	  return "Breadth_First";

      default :
	  return "Unknown";
      }
  }
//______________________________________________________________________________
/**
 * Finds the node in the tree rooted at this node
 * and storing the specified page.
 * <Br>
 * Pages are considered to be identical if they are at the same location.
 * @param location the path of the targetted page.
 * @return the node if it exists or null.
 * @see #breadthFirstEnumeration()
 * @see Locator#getLocation()
 */
final protected WebNode findNode(String location)
{
    if(location == null) return null;

    for(Enumeration e = breadthFirstEnumeration(); e.hasMoreElements();) {

	WebNode node  = (WebNode)e.nextElement();
	String string = node.getLocation();
	if(location.equals(string))
	    return node;
    }

    return null;
}
//______________________________________________________________________________
/**
 * Prints content of the tree rooted at this node.
 * <Br>
 * Minimum, maximum of dir and link depth are computed and stored in a string.
 * @return a string providing results.
 * @see #toString()
 * @see #breadthFirstEnumeration()
 */
final public String getVolume()
{
    int min_dir_depth     = Integer.MAX_VALUE;
    int max_dir_depth     = Integer.MIN_VALUE;
    int min_link_depth    = Integer.MAX_VALUE;
    int max_link_depth    = Integer.MIN_VALUE;

    WebNode min_dir_node  = null;
    WebNode max_dir_node  = null;
    WebNode min_link_node = null;
    WebNode max_link_node = null;

    for(Enumeration e = breadthFirstEnumeration(); e.hasMoreElements();) {
	
	WebNode node = (WebNode)e.nextElement();
	int depth    = node.getDirDepth();

	if(depth > max_dir_depth) {
	    max_dir_depth = depth;
	    max_dir_node  = node;
	}
	if(depth < min_dir_depth) {
	    min_dir_depth = depth;
	    min_dir_node  = node;
	}

	depth    = node.getLinkDepth();

	if(depth > max_link_depth) {
	    max_link_depth = depth;
	    max_link_node  = node;
	}
	if(depth < min_link_depth) {
	    min_link_depth = depth;
	    min_link_node  = node;
	}
	if(false)
	    System.out.println(node.getLocator());
    }

    StringBuffer buf = new StringBuffer("\n");

    buf.append("Dir  min : ").append(min_dir_node.getLocator());
    buf.append(" (depth : ").append(min_dir_depth).append(")\n");

    buf.append("Dir  max : ").append(max_dir_node.getLocator());
    buf.append(" (depth : ").append(max_dir_depth).append(")\n");

    buf.append("Link min : ").append(min_link_node.getLocator());
    buf.append(" (depth : ").append(min_link_depth).append(")\n");

    buf.append("Link max : ").append(max_link_node.getLocator());
    buf.append(" (depth : ").append(max_link_depth).append(")\n");

    return buf.toString();
}
//______________________________________________________________________________
  /**
   * Compares two WebNodes for equality.
   * Returns a boolean that indicates whether this WebNode is equivalent 
   * to the specified Object.
   *
   * @param obj	the Object to compare with.
   * @return true if these Objects are equal; false otherwise.
   */
public boolean	equals(Object obj)
  {
    if (obj == null ||  ! (obj instanceof WebNode))
       return false;

    return getLocation().equals(((WebNode)obj).getLocation());
  }
//______________________________________________________________________________
/**
 * Prints content of the tree rooted at this node.
 *<Br>
 * @see #toString()
 * @see #breadthFirstEnumeration()
 */
final public void dumpBreadthFirst()
{
    for(Enumeration e = breadthFirstEnumeration(); e.hasMoreElements();) {
	System.out.print(((WebNode)e.nextElement()).toString() + ' ');
    }
    System.out.println();
}
//______________________________________________________________________________
/**
 * Prints content of the tree rooted at this node.
 * <Br>
 * @see #postorderEnumeration()
 * @see #toString()
 */
final public void dumpPostorder()
{
    for(Enumeration e = postorderEnumeration(); e.hasMoreElements();) {
	System.out.print(((WebNode)e.nextElement()).toString() + ' ');
    }
    System.out.println();
}
//______________________________________________________________________________
/**
 * Prints content of the tree rooted at this node.
 * <Br>
 * @see #toString()
 * @see #preorderEnumeration()
 */
final public void dumpPreorder()
{
    for(Enumeration e = preorderEnumeration(); e.hasMoreElements();) {
	System.out.print(((WebNode)e.nextElement()).toString() + ' ');
    }
    System.out.println();
}
//______________________________________________________________________________
/**
 * Converts this node to String.
 * @return the string.
 * @see WebPage#toString()
 */
public String toString()
  {
      return getUserObject().toString();
  }
//______________________________________________________________________________
}




