//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader;

import java.io.*;
import org.ariane.tools.ToolBox;

/**
 * PathLocator : a reference to a local document.
 * <P>
 * @version $Id: PathLocator.java,v 3.7 2000/11/02 10:46:50 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */
public class PathLocator extends Locator {
  /**
   * @serial The scheme used to specify a local file.
   */
static final private String FILEL = "file:";
  /**
   * @serial The prefix used to specify a local file.
   */
static final private String FILEP;
  /**
   * @serial The abstract File.
   */
private File file;

static {
    // On Windows append a slash to the scheme.
    FILEP = File.separatorChar == '\\' ? "file:/" : "file:";
}
//______________________________________________________________________________
/**
 * Creates the object.

 * @param location the string providing the location.
 * @throws InvalidLocatorException is the location is not valid.
 * @see File#getCanonicalFile
 */
protected PathLocator(String location) throws InvalidLocatorException
  {
      if(location.startsWith(FILEL)) {
	  location = location.substring(FILEL.length());
      }

      // On Windows replace existing '|' by ':'.
      if(File.separatorChar == '\\') {
	  location = location.replace('|', ':');
      }

      if(location.indexOf(' ') >= 0) {
	  location = escapeSpace(location);
      }

      try {
	  file = new File(removeTag(location)).getCanonicalFile();
	  setLocation(file.toURL()); 
      }
      catch(IOException ex) {
	  throw new InvalidLocatorException(ex.getMessage());
      }
  }
//______________________________________________________________________________
/**
 * Creates the object with a parent and a child.
 * @param parent the string providing the parent location.
 * @param location the string providing the relative location.
 * @throws InvalidLocatorException is the location is not valid.
 * @see File#getCanonicalFile
 */
protected PathLocator(String parent, String location)
                                                  throws InvalidLocatorException
  {
      if(parent.startsWith(FILEL)) {
	  parent  = parent.substring(FILEL.length());
      }

      // On Windows replace existing '|' by ':'.
      if(File.separatorChar == '\\') {
	  location = location.replace('|', ':');
	  parent   = parent.replace('|', ':');
      }

      if(location.indexOf(' ') >= 0) {
	  location = escapeSpace(location);
      }
      if(parent.indexOf(' ') >= 0) {
	  parent = escapeSpace(parent);
      }

      if(! (parent.endsWith("/") || parent.endsWith("\\"))) {
	  parent = new File(removeTag(parent)).getParent();
      }

      try {
	  file = new File(parent, removeTag(location)).getCanonicalFile();
	  setLocation(file.toURL()); 
      }
      catch(IOException ex) {
	  throw new InvalidLocatorException(ex.getMessage());
      }
  }
//______________________________________________________________________________
/**
 * Adds the scheme.
 * @param path the path to process.
 * @return the path prefixed with "file:" if necessary.
 */
static final private String addScheme(String path)
  {
      return path.startsWith(FILEL) ? path : FILEL + path;
  }
//______________________________________________________________________________
/**
 * Gets the path as a File object.
 * <Br>
 * Example : <Code>/path/to/a/page.html</Code>
 * gives : <Code>/path/to/a/page.html</Code>
 * <br>
 * Characters may be encoded.
 * @return the path.
 * @see File
 * @see #getPath
 * @see #getDecodedFilePath
 */
final protected File getFilePath()
  {
      return file;
  }
//______________________________________________________________________________
/**
 * Gets the path as a File object.
 * <Br>
 * Any encoded character is reset to its plain ascii form.
 * @return the path.
 * @see #getFilePath
 */
final private File getDecodedFilePath()
  {
      String path = file.getPath();
      if(path.indexOf('%') < 0) return file; // No escaped character.

      return new File(unescapeChar(path));
  }
//______________________________________________________________________________
/**
 * Gets the server specification.
 *
 * @return the empty string.
 */
final protected String getServerSpec()
  {
      return "";
  }
//______________________________________________________________________________
/**
 * Gets the scheme with the separator appended.
 *
 * @return the string "file:".
 */
final protected String getSchemeServerSpec()
  {
      return FILEP;
  }
//______________________________________________________________________________
/**
 * Returns null since it does not make sense.
 * @return null
 */
final protected String getRedirectedLocation()
  {
      return null;
  }
//______________________________________________________________________________
/**
 * Checks to see if this locator is reachable.
 * <P>
 * The return codes are defined in {@link java.net.HttpURLConnection
 * <Code>HttpURLConnection</Code>}
 * This implementation may return OK, NOT_FOUND or UNAUTHORIZED.
 * @return a code indicating how a request behaves.
 * @see #isReadable
 */
final protected int getReadCode()
  {
      try {
	  return isReadable() ?
	      java.net.HttpURLConnection.HTTP_OK :
	      java.net.HttpURLConnection.HTTP_NOT_FOUND;
      }
      catch(Exception ex) {
	  return java.net.HttpURLConnection.HTTP_UNAUTHORIZED;
      }
  }
//______________________________________________________________________________
/*
 * Checks to see if this locator is reachable.
 * <P>
 * It tries to open a stream from this location.
 * If it not possible false is returned.
 *
 * @return true if stream can be open from this location.
 * @see File#canRead
 * @see #getReadCode
 */
final protected boolean isReadable()
  {
      try {
	  return getDecodedFilePath().canRead();
      }
      catch(Exception ex) {
	  return false;
      }
  }
//______________________________________________________________________________
/**
 * Checks to see if this locator points to an existing document.
 * <P>
 * If it is a plain path, it is easy to find out if the file exists.
 * @return true if this location points to an existing document.
 * @see File#exists
 */
final protected boolean exists()
  {
      return getDecodedFilePath().exists();
  }
//______________________________________________________________________________
/**
 * Gets the type of the document pointed by this locator.
 *
 * @return TYPE_UNKNOWN .
 */
final protected int getContentType()
  {
      return TYPE_UNKNOWN;
  }
//______________________________________________________________________________
/**
 * Gets a reader object to the document pointed by this locator.
 *
 * @return the reader.
 * @exception IOException when the document cannot be opened.
 * @see #isReadable
 * @see File#getCanonicalPath
 * @see FileInputStream
 * @see InputStreamReader
 */
protected Reader getReader() throws IOException
  {
      InputStream input = new FileInputStream(getDecodedFilePath().getPath());

      return new InputStreamReader(input);
  }
//______________________________________________________________________________
}

