//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader;

import java.net.*;
import java.io.*;
import org.ariane.tools.ToolBox;

/**
 * NetLocator : a reference to a document accessible by HTTP.
 * <P>
 * It is the specific version of a locator for a document accessible with
 * a protocol.
 *
 * @see #main
 * @see URL
 * @version $Id: NetLocator.java,v 3.1 2000/10/25 16:15:02 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */

public class NetLocator extends Locator {

//______________________________________________________________________________
/**
 * Creates the object.
 * <Br>
 * @param location the string providing the location.
 * @throws InvalidLocatorException is the location is not valid.
 */
protected NetLocator(String location) throws InvalidLocatorException
  {
      super(location);
  }
//______________________________________________________________________________
/**
 * Creates the object with a parent and a child.
 * <Br>
 * @param parent the string providing the parent location.
 * @param location the string providing the relative location.
 * @throws InvalidLocatorException is the location is not valid.
 */
protected NetLocator(String parent, String location)
                                                  throws InvalidLocatorException
  {
      super(parent, location);
  }
//______________________________________________________________________________
/**
 * Gets the server specification.
 * <P>
 * It returns the protocol, hostname, port string.
 * <Br>
 * Example : <Code>http://www.site.org:8080/path/to/a/page</Code>
 * gives : <Code>http://www.site.org:8080</Code>
 *
 * @return the protocol-host.
 */
final protected String getServerSpec()
  {
      URL url = getURL();
      StringBuffer buf = new StringBuffer();

      String spec = url.getProtocol();
      if(spec.length() > 0) buf.append(spec).append(":");

      spec = url.getHost();
      if(spec.length() > 0) buf.append("//").append(spec);

      int port = url.getPort();
      if(port > 0) buf.append(":").append(port);

      return buf.toString();
  }
//______________________________________________________________________________
/**
 * Gets the server spec with the scheme.
 *
 * @return the server spec.
 * @see #getServerSpec
 */
final protected String getSchemeServerSpec()
  {
      return getServerSpec();
  }
//______________________________________________________________________________
/**
 * Checks to see if this locator is reachable.
 * <P>
 * It tries to open a connection to this location. If it not possible
 * the exception is caught and false is returned.
 * It the return code is 4xx it returns true
 *
 * @return true if a stream can be open from this location.
 * @see #getReadCode()
 */
final protected boolean isReadable()
  {
      int rc = getReadCode();
      return (200 <= rc && rc < 300);
  }
//______________________________________________________________________________
/**
 * Checks to see if this locator points to an existing document.
 * <P>
 * If it is an http url, the method tries to open a connection to
 * this <Code>location</Code>. If the returned header contains
 * <Code>OK</Code>, the document exists.
 * If the header has an <Code>Error 404</Code>, the document
 * it means that it is not found.
 *
 * @return true if this location points to an existing document.
 * @see #getReadCode
 */
final protected boolean exists()
  {
      int rc = getReadCode();
      return rc == 200;
  }
//______________________________________________________________________________
/**
 * Gets the type of the document pointed by this locator.

 * @return the type of the document.
 * @see URLConnection#getContentType
 */
final protected int getContentType()
  {
      try {
	  URL url = getURL();
	  HttpURLConnection c = (HttpURLConnection)url.openConnection();
	  c.setRequestMethod("HEAD");

	  String s = c.getContentType();
	  if(s == null) return TYPE_UNKNOWN;

	  return s.toLowerCase().startsWith("text/html") ?
	         TYPE_HTML : TYPE_NOT_HTML;
      }
      catch(Exception ex) {
	  return TYPE_UNKNOWN;
      }
  }
//______________________________________________________________________________
/**
 * Gets the redirected location of this URL.
 * <Br>
 * If the document is located elsewhere and if the server is able to
 * find the new location, this location is returned. In any other cases
 * the method returns null.
 *
 * @return the new location or null.
 * @see URLConnection#getHeaderField
 */
final protected String getRedirectedLocation()
  {
      try {
	  URL url = getURL();
	  HttpURLConnection c = (HttpURLConnection)url.openConnection();
	  c.setRequestMethod("HEAD");

	  int rc = c.getResponseCode();
	  return (300 <= rc && rc < 400) ? c.getHeaderField("Location") : null;
      }
      catch(Exception ex) {
	  return null;
      }
  }
//______________________________________________________________________________
/**
 * Gets the response code from an http server.
 * <P>
 * See also RFC 2616.
 *
 * @return the response code to the request.
 * @see HttpURLConnection
 */
final protected int getReadCode()
  {
      try {
	  HttpURLConnection c = (HttpURLConnection)getURL().openConnection();
	  c.setRequestMethod("HEAD");

	  return c.getResponseCode();
      }
      catch(Exception ex) {
	  return HttpURLConnection.HTTP_BAD_REQUEST;
      }
  }
//______________________________________________________________________________
/**
 * Gets a reader object to the document pointed by this locator.
 *
 * @exception IOException  when the document cannot be opened.
 * @see #isReadable
 * @see URL#openStream
 * @see InputStreamReader
 */
protected Reader getReader() throws IOException
  {
      InputStream input = getURL().openStream();

      return new InputStreamReader(input);
  }
//______________________________________________________________________________
}




