//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader;

import java.net.*;
import java.io.*;
import java.util.StringTokenizer;
import org.ariane.tools.ToolBox;

/**
 * Locator : a reference to a document.
 * <P>
 * It is built from strings. The result can be a plain path
 * to a local file or an URL to a document accessible on the network.
 * <P>
 * Different components of the locator can be accessed.
 * For a document located at <Code>http://www.site.org/path/to/doc.html</Code>
 * <Ul>
 * <Li>The <Code>location</Code> is the full specification of the Locator.
 * (<Code>http://www.site.org/path/to/doc.html</Code>)
 *
 * <Li>The <Code>server spec</Code> is the protocol-hostname-port part.
 * (<Code>http://www.site.org</Code>)
 *
 * <Li>The <Code>path</Code> is the pathname of the document.
 * (<Code>/path/to/doc.html</Code>)
 *
 * <Li>The <Code>dir</Code> is the directory of the document.
 * (<Code>/path/to</Code>)
 *
 * <Li>The <Code>name</Code> is the filename of the document.
 * (<Code>doc.html</Code>)
 * </Ul>
 *
 * @see #main
 * @see URL
 * @version $Id: Locator.java,v 3.6 2000/10/25 16:15:02 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */

abstract public class Locator {
  /**
   * @serial The url.
   */
private URL location;
  /**
   * @serial When the type of the document is not known.
   */
static final protected int TYPE_UNKNOWN = -1;
  /**
   * @serial When the  document is HTML.
   */
static final protected int TYPE_HTML = 1;
  /**
   * @serial When the  document is plain text.
   */
static final protected int TYPE_PLAIN_TEXT = 2;
  /**
   * @serial When the document is something else.
   */
static final protected int TYPE_NOT_HTML = 0;

//______________________________________________________________________________
/**
 * Creates the object. It is a factory method.
 * <P>
 * Depending on the structure of the location a specific locator is
 * created.
 *
 * @param location the string providing the location.
 * @return a locator : either a PathLocator or a NetLocator.
 * @throws InvalidLocatorException is the location is not valid.
 * @see PathLocator
 * @see NetLocator
 */
static public Locator makeLocator(String location)
                                   throws InvalidLocatorException
  {
      if(isLocalPath(location))
	  return new PathLocator(location);
      else
	  return new NetLocator(location);  
  }
//______________________________________________________________________________
/**
 * Creates the object. It is a factory method.
 * <P>
 * @param parent the string providing the parent location.
 * @param location the string providing the location.
 * @return a locator : either a PathLocator or a NetLocator.
 * @throws InvalidLocatorException is the location is not valid.
 * @see PathLocator
 * @see NetLocator
 */
static public Locator makeLocator(String parent, String location)
                                   throws InvalidLocatorException
  {
      if(isLocalPath(parent) && isLocalPath(location))
	  return new PathLocator(parent, location);
      else
	  return new NetLocator(parent, location);  
  }
//______________________________________________________________________________
/**
 * Creates the object. It is a factory method.
 * <Br>
 * @param location the string providing the location.
 * @param parent the parent location.
 * @return a locator : either a PathLocator or a NetLocator.
 * @throws InvalidLocatorException is the location is not valid.
 * @see PathLocator
 * @see NetLocator
 */
static public Locator makeLocator(Locator parent, String location)
                                   throws InvalidLocatorException
  {
      return makeLocator(parent.getLocation(), location);
  }
//______________________________________________________________________________
/**
 * Checks to see if a string represents a local path or an remote URL.
 * <Br>
 * If there is no scheme at the beginning of the string (i.e.
 * there is no colon ':') or if this scheme is "file" the spec is
 * assumed to be a plain pathname.
 * On Windows if the spec starts with a letter followed by ':' the spec
 * is considered as local.
 *
 * @return true if the spec denotes a local file.
 */
static private boolean isLocalPath(String spec)
  {
      int i = spec.indexOf(':');
      if(i < 0 || spec.substring(0, i).equals("file")) { // No problemo
	  return true;
      }
      else if(i == 1 && File.separatorChar != '/') { // Windows C:\path
	  return Character.isLetter(spec.charAt(0));
      }

      return false;
  }
//______________________________________________________________________________
/**
 * Creates the object.
 */
protected Locator()
  {
      location = null;
  }
//______________________________________________________________________________
/**
 * Creates the object.
 * <Br>
 * @param location the string providing the location.
 * @throws InvalidLocatorException is the location is not valid.
 * @see URL#URL(String)
 */
protected Locator(String location) throws InvalidLocatorException
  {
      setLocation(location); 
  }
//______________________________________________________________________________
/**
 * Creates the object with a parent and a child.
 * <P>
 * @param parent the string providing the parent location.
 * @param location the string providing the relative location.
 * @throws InvalidLocatorException is the location is not valid.
 * @see URL#URL(String)
 */
protected Locator(String parent, String location) throws InvalidLocatorException
  {
      setLocation(parent, location); 
  }
//______________________________________________________________________________
/**
 * Creates the URL object with a string.
 * <P>
 * @param location the string providing the location.
 * @throws InvalidLocatorException is the location is not valid.
 * @see URL#URL(String)
 */
final protected void setLocation(String location) throws InvalidLocatorException
  {
      try {
	  if(location.indexOf(' ') >= 0) location = escapeSpace(location);
	  this.location = new URL(removeTag(location));
      }
      catch(MalformedURLException ex) {
	  throw new InvalidLocatorException(ex.getMessage());
      }
  }
//______________________________________________________________________________
/**
 * Sets the URL object.
 * @param location the location.
 */
final protected void setLocation(URL location)
  {
      this.location = location;
  }
//______________________________________________________________________________
/**
 * Creates the URL object with a parent and a child.
 * <P>
 * @param parent the string providing the parent location.
 * @param location the string providing the relative location.
 * @throws InvalidLocatorException is the location is not valid.
 * @see URL#URL(String)
 */
final protected void setLocation(String parent, String location)
                                               throws InvalidLocatorException
  {
      try {
	  if(location.indexOf(' ') >= 0) location = escapeSpace(location);
	  this.location = new URL(new URL(parent), removeTag(location));
      }
      catch(MalformedURLException ex) {
	  throw new InvalidLocatorException(ex.getMessage());
      }
  }
//______________________________________________________________________________
/**
 * Removes any fragment from the URL spec.
 * <Br>
 * If there is a tag indicated by a '#' in the spec, it is discarded.
 *
 * @return the cleaned up spec.
 */
static final protected String removeTag(String spec)
  {
      int i = spec.indexOf('#');
      return (i < 0) ? spec : spec.substring(0, i);
  }
//______________________________________________________________________________
/**
 * Converts embedded white space characters into their escaped form.
 * <B>
 * /a b/c  / gives /a%20b/c%20%20/
 * @param path the string to check.
 * @return the new string with space escaped.
 */
static final protected String escapeSpace(String path)
  {
      byte[] src = path.getBytes();
      int n      = src.length;
      byte[] dst = new byte[3 * n]; // At most, if all chars are spaces
      int m      = 0;

      for(int i = 0; i < n; i++) {
	  if(src[i] == ' ') {
	      dst[m++] = (byte)'%'; dst[m++] = (byte)'2'; dst[m++] = (byte)'0';
	  }
	  else {
	      dst[m++] = src[i];
	  }
      }

      return new String(dst, 0, m);
  }
//______________________________________________________________________________
/**
 * Translates back escaped characters to their ascii form.
 * <B>
 * /a%20b/c%20%20/ gives /a b/c  .
 * @param path the string to decode.
 * @return the new string with ascii characters.
 */
static final protected String unescapeChar(String path)
  {
      byte[] src = path.getBytes();
      int n      = src.length;
      byte[] dst = new byte[n];
      byte[] val = new byte[2];
      int m      = 0;

      for(int i = 0; i < n; i++) {
	  if(src[i] == '%') {
	      try {
		  val[0]   = src[i + 1];
		  val[1]   = src[i + 2];
		  dst[m++] = Byte.parseByte(new String(val), 16);
		  i += 2;
	      }
	      catch(NumberFormatException ex) {
		  dst[m++] = src[i]; // Give up decoding
	      }
	  }
	  else {
	      dst[m++] = src[i];
	  }
      }

      return new String(dst, 0, m);
  }
//______________________________________________________________________________
/**
 * Gets the location.
 * <P>
 * It is the external form of the URL.
 * @return the location as a string.
 * @see URL#toExternalForm
 */
final protected String getLocation()
  {
      return location.toExternalForm();
  }
//______________________________________________________________________________
/**
 * Gets the url object.
 * @return the URL.
 */
final protected URL getURL()
  {
      return location;
  }
//______________________________________________________________________________
/**
 * Gets the last component of the location.
 * <Br>
 * Example : <Code>http://www.site.org:8080/path/to/a/page.html</Code>
 * gives : <Code>page.html</Code>
 *
 * @return the string.
 * @see #getFilePath
 */
final protected String getFileName()
  {
      return isRoot() ? "/" : getFilePath().getName();
  }
//______________________________________________________________________________
/**
 * Gets the directory part of the path.
 * <Br>
 * Example : <Code>http://www.site.org:8080/path/to/a/page.html</Code>
 * gives : <Code>/path/to/a</Code>
 * <Br>
 * The separator is always '/' even on Windows.
 *
 * @return the directory or "."
 * @see #getFilePath
 */
final protected String getParentPath()
  {
      if(isRoot()) return "/";
      String path = getFilePath().getParent();
      if(path == null)
	  return ".";
      else if(File.separatorChar == '/')
	  return path;

      return path.replace(File.separatorChar, '/');
  }
//______________________________________________________________________________
/**
 * Gets the name of the last directory in the path.
 * <Br>
 * Example : <Code>http://www.site.org:8080/path/to/a/page.html</Code>
 * gives : <Code>a</Code>
 *
 * @return the name of the directory ".".
 * @see #getFilePath
 */
final protected String getParentName()
  {
      if(isRoot()) return "/";

      File parent = getFilePath().getParentFile();
      if(parent == null) return ".";

      String name =  parent.getName();
      return name.equals("") ? "/" : name;
  }
//______________________________________________________________________________
/**
 * Gets the path as a File object.
 * <Br>
 * Example : <Code>http://www.site.org:8080/path/to/a/page.html</Code>
 * gives : <Code>/path/to/a/page.html</Code>
 * <Br>
 * Care must be taken when manipulating the separator !
 * @return the path.
 * @see File
 * @see #getPath
 */
protected File getFilePath()
  {
      return new File(getPath());
  }
//______________________________________________________________________________
/**
 * Gets the path as a String object.
 * <Br>
 * Example : <Code>http://www.site.org:8080/path/to/a/page.html</Code>
 * gives : <Code>/path/to/a/page.html</Code>
 *
 * @return the path.
 * @see URL#getFile
 */
final protected String getPath()
  {
      return isRoot() ? "/" : location.getFile();
  }
//______________________________________________________________________________
/**
 * Checks to see if the path is root.
 *
 * @return true if the path is "/" or "".
 * @see URL#getFile
 */
final private boolean isRoot()
  {
      String path = location.getFile();
      return path.equals("/") || path.equals("");
  }
//______________________________________________________________________________
/**
 * Gets the server specification.
 * @return the protocol-host part or ""
 */
abstract protected String getServerSpec();
/**
 * Gets the server specification.
 * @return the protocol-host part or "file:"
 */
abstract protected String getSchemeServerSpec();
/**
 * Gets the new location if the document was moved permanently.
 * @return the new location or null
 */
abstract protected String getRedirectedLocation();

//______________________________________________________________________________
/**
 * Gets the directory depth.
 * <P>
 * The directory depth is the number of directories between the top
 * and the subdirectory where the file pointed by this locator is.
 * It is estimated as the number of '/' in the path.
 * <Br>
 * Example : <Code>http://www.site.org:8080/path/to/a/page.html</Code>
 * gives a depth of 4.
 *
 * @return the depth.
 */
final protected int getDirDepth()
  {
      return (new StringTokenizer(getPath(), "/")).countTokens();
  }
//______________________________________________________________________________
/**
 * Checks to see if this locator is below a specified path.
 * <P>
 * If the path to this locator starts with the one passed on argument
 * it is considered as below. If they are equal, it is not below.
 * <P>
 * If the path is <Code>http://www.site.org/a</Code> and if 
 * this locator is <Code>http://www.site.org/a/b/index.html</Code>
 * the locator is below. If the locator is 
 * <Code>http://www.site.org/c/index.html</Code> it is not below.
 * 
 * @return true if it is below the given path.
 */
final protected boolean isBelow(String path)
  {
      String here = getLocation();
      return here.startsWith(path) && here.length() > path.length();
  }
//______________________________________________________________________________
/**
 * Checks to see if a string may represent a location.
 * <Br>
 * It rejects strings beginning with
 * "mailto", "javascript", "ftp", "news", "#".
 * @return true if it is handled by this version of WebWader.
 */
final static protected boolean isAcceptable(String string)
  {
      if(string == null)
	  return false;
      else if(string.startsWith("mailto:"))
	  return false;
      else if(string.startsWith("javascript:"))
	  return false;
      else if(string.startsWith("ftp:"))
	  return false;
      else if(string.startsWith("news:"))
	  return false;
      else if(string.startsWith("#"))
	  return false;
      return true;
  }
//______________________________________________________________________________
/**
 * Converts this Locator to String.
 * @return the location.
 */
final public String toString()
  {
      return location.toString();
  }
//______________________________________________________________________________
/**
 * Gives a list of the various parts of this locator.
 * @return the concatenation of the substrings.
 */
final public String paramString()
  {
      StringBuffer buf = new
      StringBuffer("Location : ").append(getLocation());
      buf.append(" Server : ").append(getServerSpec());
      buf.append(" Path : ").append(getPath());
      buf.append(" FilePath : ").append(getFilePath().getPath());
      buf.append(" DirPath : ").append(getParentPath());
      buf.append(" DirName : ").append(getParentName());
      buf.append(" Name : ").append(getFileName());
      buf.append(" Exists : ").append(exists() ? "yes" : "no");
      buf.append(" Readable : ").append(isReadable() ? "yes" : "no");

      return buf.toString();
  }
//______________________________________________________________________________
/**
 * Checks to see if this locator is reachable.
 * <P>
 * The return codes are defined in {@link java.net.HttpURLConnection
 * <Code>HttpURLConnection</Code>}
 * @return a code indicating how a request behaves.
 * @see #isReadable
 */
abstract protected int getReadCode();

//______________________________________________________________________________
/**
 * Checks to see if this locator is reachable.
 * @return true if a stream can be open from this location.
 * @see #getReadCode
 */
abstract protected boolean isReadable();

//______________________________________________________________________________
/**
/**
 * Checks to see if this locator points to an existing document.
 * @return true if this location points to an existing document.
 */
abstract protected boolean exists();

//______________________________________________________________________________
/**
 * Gets the type of the document pointed by this locator.
 *
 * @return the type of the document.
 */
abstract protected int getContentType();

//______________________________________________________________________________
/**
 * Gets a reader object to the document pointed by this locator.
 *
 * @return the reader.
 * @exception IOException  when the document cannot be opened.
 */
abstract protected Reader getReader() throws IOException;

//______________________________________________________________________________
  /**
   * Starts the test program.
   * <P>
   * Usage : java org.demo.webwader.Locator [parent] child
   * <P>
   * It accepts one or two arguments.
   * @param args the array of arguments on the command line.
   * @see #makeLocator
   */
public static void main(String args[])
  {
      if(args.length < 1) {
	  printUsage();
	  return;
      }
	  
      try {
	  Locator l = Locator.makeLocator(args[0]);
	  if(args.length > 1) {
	      Locator p = Locator.makeLocator(l, args[1]);
	      l = p;
	  }
	  ToolBox.warn(l.paramString());
      }
      catch(InvalidLocatorException ex) {
	  ToolBox.warn("Can't make locator : " + args[0], ex);
      }
      catch(IOException ex) {
	  ToolBox.warn("Can't connect : " + args[0], ex);
      }
  }
//______________________________________________________________________________
    /**
     * Prints how to use the program.
     */
final static private void printUsage()
  {
      System.out.println(
      "Usage   : java org.demo.webwader.Locator [parent] child");
  }
//______________________________________________________________________________
}

