//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.tools;

import java.util.Date;
import java.util.Locale;
import java.util.StringTokenizer;
import java.util.NoSuchElementException;
import java.io.File;
import java.text.*;

/**
 * Toolbox : a collection of useful procedures.
 * <P>
 * It is a singleton : the unique instance can be obtained with
 * the static method : <Code>ToolBox.instance()</Code>.
 * @version $Id: ToolBox.java,v 3.7 2001/05/14 15:01:33 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */
public	final	class ToolBox extends Object {

  /**
   * @serial Decides whether some useful information is output or not.
   */
public static 	boolean verbose	= false;
  /**
   * @serial Decides whether some debugging information is output or not.
   */
public static 	boolean debug	= false;
  /**
   * @serial the character marking a file a file as temporary. See emacs.
   */
public static final char tempfileMarker	= '~';
  /**
   * @serial Inverse of  natural logarithm of 10.
   */
public static 	final double VLN10	= (double)1 / Math.log(10.);
  /**
   * @serial A tiny value to test against 0.
   */
public static 	final double EPSILON	= (double)0.00001;
  /**
   * @serial A huge value to test against 0.
   */
public static 	final double HUGEVALUE = (double)1.e09;
  /**
   * A constant used by exit() when something went wrong.
   */
public static 	final int WRONG = 1;
  /**
   * @serial A constant used by exit() when a process was sucessful.
   */
public static 	final int OK    = 0;
  /**
   * @serial The unique instance.
   */
private  static	ToolBox toolbox = null;
  /**
   * @serial The array of white space characters.
   */
private  static	char[] spaces;
static {
    int n  = 80;
    spaces = new char[n];
    for(int i = 0; i < n; i++)
	spaces[i] = ' ';
}

//______________________________________________________________________________
/**
 * Gets an unique instance of ToolBox.
 * It is a singleton
 */
public static	ToolBox instance()
  {
    if(toolbox == null)
	toolbox = new ToolBox();

    return toolbox;
  }
//______________________________________________________________________________
  /**
   * Creates the Tool Box. It is private to ensure that only one
   * instance is created.
   * @see #instance
   */
private ToolBox()
  {
  }
//______________________________________________________________________________
  /**
   * Prints the current date to stdout.
   * @see Date
   */
final public void  printDate()
  {
     Date today = new Date();
     System.out.println(today);
  }
//______________________________________________________________________________
/**
 * Changes the default locale.
 * <Br>
 * If the ident is null, simply returns the current default.
 * Otherwise, it must be made of 2 (possibly 3) tokens separated by
 * underscore : language_country_variant. e.g. : en_US, fr_FR_sci
 *
 * @param ident identifies a locale.
 * @return the locale
 */
public static Locale setLocale(String ident)
  {
      if(ident == null) return Locale.getDefault();

      StringTokenizer stk = new StringTokenizer(ident, "_");
      String language = null;
      String country  = null;

      try {
	  language = stk.nextToken();
	  country  = stk.nextToken();
      }
      catch (NoSuchElementException ex) {
	  ToolBox.warn("Invalid locale " + ident);
	  return null;
      }

      Locale locale;

      if(stk.hasMoreTokens()) {
	  String variant = stk.nextToken();
	  locale = new Locale(language, country, variant);
      }
      else {
	  locale = new Locale(language, country);
      }

      Locale.setDefault(locale);

      return locale;
  }
//______________________________________________________________________________
/**
 * Gets a decimal format in scientific notation.
 * <Br>
 * An instance with en_US locale forced is generated. So the decimal
 * separator is set to . even if the current locale is fr_FR.
 * The pattern is applied.
 *
 * @param pattern the template used to display numbers.
 * @return the format.
 * @see FxpN
 */
public static DecimalFormat scientificFormat(String pattern)
  {
      DecimalFormat format = (DecimalFormat)NumberFormat.getInstance(Locale.US);
      DecimalFormatSymbols dfs = format.getDecimalFormatSymbols();
      dfs.setGroupingSeparator(' ');
      format.setDecimalFormatSymbols(dfs);
      format.setGroupingUsed(true);
      format.applyPattern(pattern);

      return format;
  }
//______________________________________________________________________________
/**
 * Prints a message on stdout.
 * @param msg	a string.
 * @param ex	an exception.
 */
final public static void warn(String msg, Exception ex)
  {
      warn(msg + " : " + ex);
  }
//______________________________________________________________________________
/**
 * Prints a message on stdout.
 * @param msg a string.
 */
final public static void warn(String msg)
  {
      System.out.println(msg + " !");
  }
//______________________________________________________________________________
/**
 * Prints a dot on stdout.
 */
public static void dot()
  {
      System.out.print('.'); System.out.flush();
  }
//______________________________________________________________________________
/**
 * Reads a line from the standard input
 * @return the string read
 */
public static String readInputString()
  {
      String str = "";
      int    max = 100;
      byte[] buf = new byte[max];

      try {
	  int r = System.in.read(buf, 0, max);

	  if(r < 1) {
	      ToolBox.warn("Can't read bytes");
	  }
	  else {
	      str = new String(buf, 0, r - 1); // Get rid of LF
	  }
      }
      catch(Exception ex) {
	  ToolBox.warn("Can't get input string", ex);
      }

      return str;
  }
//______________________________________________________________________________
/**
 * Creates a centered string of the specified width with the input string.
 * <P>
 * If the string is larger than the width, it takes the first characters.
 * The number of padding spaces must be less than 80.
 * @param  str	the string to center.
 * @param  width the requested width.
 * @return the created string.
 */
public static  String centerString(String str, int width)
  {
      int l = str.length();

      if(l > width)
	  return str.substring(0, width);

      int n = (width - l) / 2;
      StringBuffer buf = new StringBuffer(width);
      buf.append(spaces, 0, n).append(str).append(spaces, 0, width - l - n);

      return buf.toString();
  }
//______________________________________________________________________________
/**
 * Creates a string of the specified width padded with white spaces.
 * <P>
 * if the width parameter is less than 1, it just returns a copy.
 * If the string is larger than the width, it truncates the string.
 * The number of padding spaces must be less than 80.
 * @param  str	the string to center.
 * @param  width the requested width.
 * @return a new substring.
 */
public static  String padString(String str, int width)
  {
      if(width < 1)
	  return new String(str);

      int l = str.length();

      if(l > width)
	  return str.substring(0, width);

      StringBuffer buf = new StringBuffer(str);
      buf.append(spaces, 0, width - l);

      return buf.toString();
  }
//______________________________________________________________________________
/**
 * Formats a float number into a string.
 * @param x the number to convert.
 * @param n the number of digits after the decimal point.
 * @return the created string.
 */
public static  String format(float x, int n)
  {
      String str = String.valueOf(x);
      int i = str.indexOf('.');

      return str.substring(0, i + n + 1);
  }
//______________________________________________________________________________
/**
 * Formats a long number into a right justified string.
 * <br>
 * If the requested width is smaller than the size of the number
 * its string representation is simply returned unchanged.
 * @param x the number to convert.
 * @param n the width of the string.
 * @return the created string.
 */
public static  String format(long x, int n)
  {
      String str = String.valueOf(x);
      int l = str.length();
      if(n <= l) return str;

      try {
	  final String
	  blanks = "                                                           ";
	  return blanks.substring(0, n - l).concat(str);
      }
      catch(Exception ex) {
	  return str;
      }
  }
//______________________________________________________________________________
/**
 * Checks if the browser used to run the applet is Netscape.
 * <Br>
 * If the property <I>browser</I> is not readable, it returns false.
 * @return true if under Netscape.
 * @see System#getProperty(String, String)
 */
public static boolean underNetscape()
  {
    try {
      return System.getProperty("browser", "Unknown").startsWith("Netscape");
    }
    catch(Exception ex) {
      ToolBox.warn("Can't find browser type", ex);
      return false;
    }
  }
//______________________________________________________________________________
/**
 * Checks if XML classes are available.
 *
 * @return true if XML can be used by this application.
 */
public static boolean hasXML()
  {
      try {
	  Class.forName("org.xml.sax.helpers.ParserFactory");
	  return true;
      }
      catch(Exception ex) {
	  return false;
      }
  }
//______________________________________________________________________________
/**
 * Creates a couple of lines which can be used as header for a XML file.
 * <br>
 * Encoding is set to iso-8859-1.
 * @param title a line inserted in the header. Can be null.
 * @return a string which can be used as an header.
 */
public static String XMLHeader(String title)
  {
      StringBuffer buf;

      buf = new StringBuffer("<?xml version='1.0' encoding='ISO8859_1'?>");
      buf.append("\n<!-- Created by ").append(System.getProperty("user.name"));
      buf.append(" on ").append(new Date()).append(" with the JVS.  -->");
      if(title != null) buf.append("\n<!-- ").append(title).append(" -->");
      buf.append("\n");

      return buf.toString();
  }
//______________________________________________________________________________
/**
 * Gets a good rounded value.
 * For instance : 12 -> 20, 450 -> 500, 632 -> 1000, -5 -> -10.
 * For v < epsilon returns 0.
 * @param value the value to be rounded.
 * @return the rounded value.
 */
static	public	double	upperRoundValue(double v)
  {
      boolean neg = false;

      if(v < 0.) {
	  neg = true;
	  v   = -v;
      }

      if(v < EPSILON)
	  return (double)0.0;

      double t = powerOfTen(v);

      v /= t;

           if(v <= 1)
	  v = 1;
      else if (v <= 2.)
	  v = 2.;
      else if (v <= 5.)
	  v = 5.;
      else if (v <= 10.)
	  v = 10.;

      if(neg) v = -v;

      return v * t;
  }
//______________________________________________________________________________
/**
 * Gets the upper power of ten.	If 10**n-1 <= v < 10**n return 10**n-1
 * @param value the value to be rounded.
 * @return the power of ten.
 */
public	static	double	powerOfTen(double v)
  {
      boolean neg = false;

      if(v < 0.) {
	  neg = true;
	  v   = -v;
      }
           if(v < 10.)
	  v = 1;
      else if (v < 100.)
	  v = 10.;
      else if (v < 1000.)
	  v = 100.;
      else if (v < 1000.)
	  v = 100.;
      else if (v < 10000.)
	  v = 1000.;
      else if (v < 100000.)
	  v = 10000.;
      else
	  v = Math.pow(10., (int)(Math.log(v) * VLN10));

      return neg ? -v : v;
  }
//______________________________________________________________________________
/**
 * Renames the files : filename <-> filename~
 * <P>
 * It is meant to swap files <I>filename</I> and <I>filename~</I> when
 * a function has to update a file named <I>filename</I>
 * and must keep a backup copy.
 * <P>
 * First the function should write in <I>filename~</I>, then if the operation
 * succeeds the original <I>filename</I> is moved to <I>filename~</I> and
 * the new file is renamed <I>filename</I>.
 * <Br>
 * If the operation failed, the file <I>filename</I> is left unchanged.
 * <Br>
 * Note that <I>filename%</I> is used as a temporaty file by this method.
 *
 * @see File#renameTo
 * @exception FileOpException if a file cannot be renamed.
 */
public static final void renameFile(String filename) throws FileOpException
  {
      File niu = new File(markAsTemp(filename));
      File old = new File(filename);

      if(! old.exists()) {
	  if(! niu.renameTo(old))
	      throw new FileOpException("Can't rename " + niu + " to " + old);
	  return;
      }

      File tmp = new File(filename + '%');

      if(! old.renameTo(tmp)) {
	  throw new FileOpException("Can't rename " + old + " to " + tmp);
      }
      else if(! niu.renameTo(old)) {
	  throw new FileOpException("Can't rename " + niu + " to " + old);
      }
      else if(! tmp.renameTo(niu)) {
	  throw new FileOpException("Can't rename " + tmp + " to " + niu);
      }
  }
//______________________________________________________________________________
    /**
     * Adds a mark to a filename to indicate that it is a temporary file.
     * @param filename the name of the file.
     * @return filename + tempfileMarker;
     */
public static final String markAsTemp(String filename)
  {
      return ((new StringBuffer(filename)).append(tempfileMarker)).toString();
  }
//______________________________________________________________________________
/**
 * Checks if the OS is a Unix.
 * @return true if the application is running under Unix, false otherwise.
 * @see File
 */
public static final boolean isUnderUnix()
  {
      return File.separatorChar == '/';
  }
//______________________________________________________________________________
/**
 * Checks if the OS is Windows.
 * @return true if the application is running under Win32, false otherwise.
 * @see File
 */
public static final boolean isUnderWindows()
  {
      return File.separatorChar == '\\';
  }
//______________________________________________________________________________
/**
 * Finds the origin of the running JVS application.
 * <br>
 * The class path is scanned to find out the first instance of the JVS
 * either the src directory or the jvs.jar.
 * <br>
 * The strings "jvs.jar" or "JVS/src" are searched.
 * @return the path to the classes used to run the application.
 */
public static final String getJVSOrigin()
  {
      String path         = System.getProperty("java.class.path");
      StringTokenizer stk = new StringTokenizer(path, File.pathSeparator);
      final String jvsjar = "jvs.jar";
      final String jvssrc = "JVS" + File.separatorChar + "src";

      while(stk.hasMoreTokens()) {

	  String origin = stk.nextToken();
	  if(origin.indexOf(jvsjar) >= 0 || origin.indexOf(jvssrc) >= 0) {
	      return origin;
	  }
      }

      return null;
  }
//______________________________________________________________________________
}


