//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.tools;

import java.util.Properties;
import java.util.StringTokenizer;
import java.io.*;
import java.awt.Color;
import java.awt.Font;

/**
 * Resources :  gives access to the encapsulated properties.
 * <P>
 * This object provides convenient methods to get parameters
 * from a dataset.
 * <P>
 * This is a singleton. The unique instance can be accessed by
 * the method {@link #instance <Code>instance()</Code>}.
 * <Br>
 * This object is initialized from the JVS property
 * file found in the lib directory of the JVS. Then a file specific
 * to an application can be loaded.
 * Finally a user's property file is merged into the set.
 * <P>
 * To check the list of resources currently defined, run
 * {@link org.ariane.test.TryProp <Code>TryProp</Code>}.
 *
 * @see Properties
 * @see org.ariane.test.TryProp
 * @see org.ariane.gui.GuiResources
 * @version $Id: Resources.java,v 3.4 2000/10/30 12:30:16 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */
public class Resources extends Object {
  /**
   * @serial the name of the property giving the home of the JVS.
   */
public static final String  JVS_HOME  = "jvs.home";
  /**
   * @serial The properties.
   */
private Properties prop;
    /**
     * @serial The url where to find data files.
     */
private String libdir;
  /**
   * @serial The unique instance.
   */
protected  static  Resources resources = null;

//______________________________________________________________________________
/**
 * Gets an unique instance of Resources.
 * <Br>
 * It is a singleton
 * @return the unique instance.
 */
public static	Resources instance()
  {
    if(resources == null)
	resources = new Resources();

    return resources;
  }
//______________________________________________________________________________
  /**
   * Creates the Resources. Sets up properties ...
   */
protected Resources()
  {
      try {
	  setUpProperties();
      }
      catch(Exception ex) {
	  ToolBox.warn("Can't set up properties", ex);
      }
  }
//______________________________________________________________________________
  /**
   * Sets up properties.
   * <Br>
   * The JVS home directory is got.
   * @see Properties#Properties()
   * @see #getJVSHome
   */
final private void setUpProperties()
  {
      prop = new Properties();
      prop.put("Software", "ariane");
      getJVSHome();
  }
//______________________________________________________________________________
  /**
   * Loads properties : reads them from file.
   * <Br>
   * Reads from default file : <Code>jvs.home/lib/jvs.properties</Code>.
   * Then merges, if it exists, the file <Code>user.home/.jvs.properties</Code>.
   * No specific file is loaded.
   *
   * @see #load(String)
   */
public void  load()
  {
      load(null);
  }
//______________________________________________________________________________
  /**
   * Loads properties : reads them from file.
   * <Br>
   * Reads from default file : <Code>jvs.home/lib/jvs.properties</Code>.
   * <Br>
   * Then merges, if it exists, the file given as argument and located in the
   * <Code>jvs.home</Code> directory.
   * <Br>
   * Finally merges, if it exists, the file
   * <Code>user.home/.jvs.properties</Code>.
   * <P>
   * If something goes wrong, the exception is caught and a message is issued.
   *
   * @param filename the application specific property file.
   */
public void load(String filename)
  {
      try {
	  read(getLibDirectory() + "jvs.properties"); // JVS standard

	  if(filename != null)
	      read(getLibDirectory() + filename);  // Application specific

	  String path = getUserHomeDirectory() + ".jvs.properties"; 
	  File file   = new File(path);              // User's preference

	  if( file.exists()) {
	      read(path);
	  }
      }
      catch(IOException ex) {
	  ToolBox.warn("Can't read properties ", ex);
      }
      catch(Exception ex) {
	  ToolBox.warn("Can't load properties ", ex);
      }
  }
//______________________________________________________________________________
  /**
   * Reads a particular list of properties : read them from file.
   * <Br>
   * New definitions are merged in the existing set.
   *
   * @param filename the file containing the properties.
   * @see Properties#load
   * @throw IOException if the file cannot be read.
   */
final private void  read(String filename) throws IOException
  {
      if(ToolBox.debug) ToolBox.warn("Reading " + filename);
      InputDataset input = new InputDataset(filename);
      prop.load(input.getStream());
      input.close();
  }
//______________________________________________________________________________
/**
 * Gets the home directory from which data files can be fetched.
 * <Br>
 * The default value is used : <Code>user.home</Code>.
 * The separator is appended.
 * @return the directory if it exists or null.
 */
final public String getUserHomeDirectory()
  {
    try {
	String sep  = System.getProperty("file.separator");
	String home = System.getProperty("user.home");
	String path = home + sep;

	return path;
    }
    catch(Exception ex) {
      System.err.println("Can't get user home directory !"); 
      System.err.println("Exception caught : " + ex + " !"); 
      return null;
    }
  }
//______________________________________________________________________________
/**
 * Gets the lib directory from which data files can be fetched.
 * <Br>
 * The default value is used : <Code>user.home/lib</Code>. It the value is not
 * found or is not a directory returns null.
 * @return the directory if it exists.
 * @deprecated replaced by {@link #getUserHomeDirectory()}
 */
final private String getUserLibDirectory()
  {
    try {
	String sep  = System.getProperty("file.separator");
	String home = System.getProperty("user.home");
	String path = home + sep + "lib" + sep;

	File dir    = new File(path);
	if(dir.isDirectory()) return path;
    }
    catch(Exception ex) {
      System.err.println("Can't get user lib directory !"); 
      System.err.println("Exception caught : " + ex + " !"); 
      return null;
    }

    return null;
  }
//______________________________________________________________________________
/**
 * Sets the default directory from which data files can be fetched.
 * <Br>
 * It is choosen as the jvs.home/lib directory.
 * @see #getJVSHome
 * @see #setLibDirectory(String)
 */
final private void setLibDirectory()
  {
      String sep  = System.getProperty("file.separator");
      setLibDirectory(getJVSHome() + sep + "lib");
  }
//______________________________________________________________________________
/**
 * Sets the directory from which data files can be fetched.
 * <Br>
 * The directory can specified as an url.
 * If the path is not terminated by the separator, it is appended.
 * @param dir the path to the directory.
 */
final private void setLibDirectory(String dir)
  {
      if(dir == null) throw new IllegalArgumentException("Null lib directory");
      libdir     = dir;
      String sep = "/";
      try {
	  if(! dir.startsWith("http:")) // For an url the sep is always '/'
	      sep = System.getProperty("file.separator");
      }
      catch(Exception ex) {
	  System.err.println("Exception caught : " + ex + " !"); 
      }

      char last = libdir.charAt(libdir.length() - 1);
      if(last != '\\' && last != '/')
	  libdir += sep;
  }
//______________________________________________________________________________
/**
 * Gets the directory from which data files can be fetched.
 * <br>
 * The path separator is present at the end of the string.
 * @return the path to the directory
 */
final public String getLibDirectory()
  {
      if(libdir == null) setLibDirectory();
      return libdir;
  }
//______________________________________________________________________________
/**
 * Gets the directory where some JVS executables are found.
 * <br>
 * It is JVS_HOME/bin.
 * The path separator is present at the end of the string.
 * @return the path to the directory
 * @see #getLibDirectory
 */
final public String getBinDirectory()
  {
      String bindir = getLibDirectory();
      int index     = bindir.length() - 4; // - "lib/" length
      return bindir.substring(0, index) +
	     "bin" + System.getProperty("file.separator");
  }
//______________________________________________________________________________
/**
 * Gets the properties mananaged by the JVS.
 * @return the properties
 */
final public Properties getProperties()
  {
      return prop;
  }
//______________________________________________________________________________
  /**
   * Saves properties : write them to file.
   * <Br>
   * The file is named 'Properties' in the current directory.
   */
public void  saveProperties()
  {
    try{
      FileOutputStream file = new FileOutputStream("Properties");
      prop.store(file, " Ariane Resources");
      file.close();
    }
    catch(IOException ex) {
	System.err.println("Exception caught : " + ex + " !"); 
   }
  }
//______________________________________________________________________________
  /**
   * Returns as a string the list of known properties.
   * <P>
   * In case of error, returns null.
   * @return the list.
   * @see Properties#list
   */
public String getPropertyValues()
  {
    try {
	StringWriter sw  = new StringWriter();
	PrintWriter pw   = new PrintWriter(sw);
	prop.list(pw);

	return sw.toString();
    }
    catch(Exception ex) {
	return null; 
   }
  }
//______________________________________________________________________________
  /**
   * Dumps properties.
   */
public void dump()
  {
    try{
	ToolBox.warn("Dumping resources " + getLibDirectory());
	prop.list(System.out);
    }
    catch(Exception ex) {
	ToolBox.warn("Resources Dump", ex); 
   }
  }
//______________________________________________________________________________
  /**
   * Sets the directory where the JVS is installed.
   * @param  jvs_home the directory.
   * @see #getJVSHome()
   */
final public void setJVSHome(String jvs_home)
  {
      prop.put(JVS_HOME, jvs_home);
  }
//______________________________________________________________________________
  /**
   * Retrieves the directory where the JVS is installed.
   * <Br>
   * If the property <Code>jvs.home</Code> exists in the JVS properties
   * its content is returned. If it is in the System properties (may be
   * set from the command line) it is stored in the JVS properties and
   * returned.
   * <Br>
   * Otherwise the Class path is scanned to find where the jvs.jar file is
   * located. It is assumed to be in the JVS lib directory. This value is kept
   * in the property <Code>jvs.home</Code>.
   *
   * @return the directory.
   * @throw ResourcesException if the directory can'be found.
   */
final public String getJVSHome()
  {
      String jvs_home = prop.getProperty(JVS_HOME);
      if(jvs_home != null) return jvs_home;

      jvs_home = System.getProperty(JVS_HOME);
      if(jvs_home != null) {
	  setJVSHome(jvs_home);
	  return jvs_home;
      }

      String path = System.getProperty("java.class.path");
      String psep = System.getProperty("path.separator");
      StringTokenizer stk = new StringTokenizer(path, psep);

      while(stk.hasMoreTokens()) {

	  final String target = "jvs.jar";
	  String str = stk.nextToken();

	  int i = str.indexOf(target) - 4; // lib/jvs.jar
	  if(i >= 0) {
	      jvs_home = (i == 0) ? "." : str.substring(0, i - 1);
	      setJVSHome(jvs_home);
	      return jvs_home;
	  }
      }

      if(ToolBox.debug) {
	  ToolBox.warn("java.class.path " + path);
	  ToolBox.warn("path.separator " + psep);
      }

      throw new ResourcesException("Can't find JVS home directory");
  }
//______________________________________________________________________________
  /**
   * Copies a property from the System properties set to this set.
   * <Br>
   * It the value was not defined nothing happens.
   *
   * @param key the key.
   * @see System#getProperty(String)
   * @see Property#setProperty(String, String)
   */
final public void copyFromSystem(String key)
  {
      String value = System.getProperty(key);
      if(value != null) prop.put(key, value);
  }
//______________________________________________________________________________
  /**
   * Copies a property from this set to the System properties set.
   * <Br>
   * It the value was not defined nothing happens.
   * @param key the key.
   * @see System#getProperty(String)
   * @see Property#setProperty(String, String)
   */
final public void copyToSystem(String key)
  {
      String value = prop.getProperty(key);
      if(value != null) System.setProperty(key, value);
  }
//______________________________________________________________________________
  /**
   * Stores a property with a value provided.
   * @param key the key.
   * @param value the value.
   * @see Property#get(String)
   */
final public void put(String key, String value)
  {
      prop.put(key, value);
  }
//______________________________________________________________________________
  /**
   * Returns a property.
   * @param key the key.
   * @return the value.
   * @see Property#getProperty(String)
   */
final public String get(String key)
  {
      return prop.getProperty(key);
  }
//______________________________________________________________________________
  /**
   * Returns property with a default value provided.
   * @param key the key.
   * @param def the default.
   * @return the value.
   * @see Property#getProperty(String, String)
   */
final public String get(String key, String def)
  {
      return prop.getProperty(key, def);
  }
//______________________________________________________________________________
  /**
   * Returns the source of a JVS image defined by its name.
   * <Br>
   * Inserts automatically the JVS lib path. For example :
   * <Br>
   * Picture=picture.gif
   * <Br>
   * "/opt/share/JVS/lib/images/picture.gif" = getImage("Picture")
   * <Br>
   * Returns a default path if the value is not found.
   * @parameter key the key.
   * @return the path to the source.
   */
final public String getImagePath(String key)
  {
      StringBuffer buf = new StringBuffer(getLibDirectory());
      buf.append("images");
      buf.append(System.getProperty("file.separator"));
      buf.append(get(key, "unknown.gif"));

      return buf.toString();
  }
//______________________________________________________________________________
  /**
   * Returns boolean property with a default value provided.
   * <Br>
   * Default value is used whenever property is not present.
   * @param key the key.
   * @param def the default.
   * @return the value.
   */
final public boolean get(String key, boolean def)
  {
      String str = prop.getProperty(key);

      if(str == null)  return def;

      return str.equalsIgnoreCase("true");
  }
//______________________________________________________________________________
  /**
   * Returns integer property with a default value provided.
   * <Br>
   * Default value is used whenever property is not present or is not a number.
   * @param key the key.
   * @param def the default.
   * @return the value.
   */
final public int get(String key, int def)
  {
      String str = prop.getProperty(key);

      if(str == null)  return def;

      int i;
      try {
	  i = Integer.parseInt(str);
      }
      catch(Exception ex) {
	  return def;
      }
      return i;
  }
//______________________________________________________________________________
  /**
   * Return float property with a default value provided.
   * <Br>
   * Default value is used whenever property is not present or is not a number.
   * @param key the key.
   * @param def the default.
   * @return the value.
   */
final public float get(String key, float def)
  {
      String str = prop.getProperty(key);

      if(str == null)  return def;

      Float x;
      try {
	  x = Float.valueOf(str);
      }
      catch(Exception ex) {
	  return def;
      }
      return x.floatValue();
  }
//______________________________________________________________________________
  /**
   * Returns Font property with a default value provided.
   * <Br>
   * Default value is used whenever property is not present.
   * The string describing the font should comply with the format :
   * name.style.size style being R, P, I, B
   * @param key the key.
   * @param defname the default name.
   * @param defstyle the default style.
   * @param defsize the default size.
   * @return the value.
   */
final public Font get(String key, String defname, int defstyle, int defsize)
  {
      String fdesc = prop.getProperty(key);
      if(fdesc == null)
	  return new Font(defname, defstyle, defsize);

      String name = "Times";
      int style   = Font.PLAIN;
      int size	  = 16;

      StringTokenizer stk = new StringTokenizer(fdesc, ".");
      name = stk.nextToken().trim();

      String str  = stk.nextToken().trim();
      if(str.equalsIgnoreCase("P"))
	  style = Font.PLAIN;
      else if(str.equalsIgnoreCase("R"))
	  style = Font.PLAIN;
      else if(str.equalsIgnoreCase("I"))
	  style = Font.ITALIC;
      else if(str.equalsIgnoreCase("B"))
	  style = Font.BOLD;

      str  = stk.nextToken().trim();
      try {
	  size = Integer.parseInt(str);
      }
      catch(Exception ex) {};

      return new Font(name, style, size);
  }
//______________________________________________________________________________
  /**
   * Returns Color property with a default value provided.
   * <Br>
   * Default value is used whenever property is not present.
   * Currently accepted : the name of the constants defined in the Class
   * color, the format #rrggbb (see the
   * <A Href = "http://www.multimania.com/lefevre/graphics/webcolors.html">
   * Web color page</A>)
   *
   * @param key the key.
   * @param def the default.
   * @return the value.
   */
final public Color get(String key, Color def)
  {
      String bg = prop.getProperty(key);

      if(bg == null) {		
	     return def;
      }
      else if(bg.indexOf('#') == 0) {
	  try {
	      return new Color(Integer.parseInt(bg.substring(1), 16));
	  }
	  catch(Exception ex) {
	      ToolBox.warn("Can't parse " + key, ex);
	      return def; 
	  }
      }
      else if(bg.equalsIgnoreCase("black")) 	return Color.black;
      else if(bg.equalsIgnoreCase("blue")) 	return Color.blue;
      else if(bg.equalsIgnoreCase("cyan")) 	return Color.cyan;
      else if(bg.equalsIgnoreCase("darkGray")) 	return Color.darkGray;
      else if(bg.equalsIgnoreCase("gray")) 	return Color.gray;
      else if(bg.equalsIgnoreCase("green")) 	return Color.green;
      else if(bg.equalsIgnoreCase("lightGray"))	return Color.lightGray;
      else if(bg.equalsIgnoreCase("magenta"))	return Color.magenta;
      else if(bg.equalsIgnoreCase("orange")) 	return Color.orange;
      else if(bg.equalsIgnoreCase("pink"))	return Color.pink;
      else if(bg.equalsIgnoreCase("red"))       return Color.red;
      else if(bg.equalsIgnoreCase("white"))	return Color.white;
      else if(bg.equalsIgnoreCase("yellow"))	return Color.yellow;

    return def;
  }
//______________________________________________________________________________
}
