//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.tools;

import java.util.AbstractList;
import java.util.Enumeration;
import java.io.Serializable;

/** 
 * PlainList : a very simple container implementing the List interface.
 * <P>
 * The motivation to create this collection was the necessity to work
 * with a {@link java.util.ListIterator <Code>ListIterator</Code>}
 * instead of a plain {@link java.util.Enumeration <Code>Enumeration</Code>}.
 * <P>
 * This implementation stores the content of an <Code>Enumeration</Code>
 * in an array.
 *
 * @version $Id: PlainList.java,v 3.1 2000/11/02 20:04:22 lefevre Exp $
 * @author Jean-Paul Le Fvre
 * @see	java.util.Vector
 */
public	class PlainList extends AbstractList implements Serializable {

  /**
   * @serial the array storing the elements.
   */
  protected	Object[] 	list;
  /**
   * @serial the number of elements in the list.
   */
  private	int 	nbr;
  /**
   * @serial the default initial capacity.
   */
static final private int INITIAL_CAPACITY = 100;
  /**
   * @serial the default size increment.
   */
static final private int CAPACITY_INCREMENT = 100;

//______________________________________________________________________________
  /**
   * Creates a list with the initial capacity specified.
   * @param initialCapacity the initial size of the array.
   */
public PlainList(int initialCapacity)
  {
      super();
      list = new Object[initialCapacity];
      nbr  = 0;
  }
//______________________________________________________________________________
  /**
   * Creates a list with INITIAL_CAPACITY slots.
   * @see #PlainList(int)
   */
public PlainList()
  {
      this(INITIAL_CAPACITY);
  }
//______________________________________________________________________________
  /**
   * Creates a list storing the content of the enumeration.
   * @param e the collection to store.
   */
public PlainList(Enumeration e)
  {
      this();
      set(e);
  }
//______________________________________________________________________________
  /**
   * Creates a string giving information about the object.
   * @return the string.
   */
public String toString()
  {
      StringBuffer buf = new StringBuffer("PlainList : ");
      buf.append(" [");

      for (java.util.Iterator it = iterator(); it.hasNext() ;) {
	  buf.append(" ").append(String.valueOf(it.next()));
      }
      buf.append(" ]");

      return buf.toString();
  }
//______________________________________________________________________________
    /**
     * Adds an object in the list. If the capacity is too small, the list
     * is recreated with more room available.
     * @param object the object to add.
     * @return true. Duplication is not checked.
     * @see java.util.AbstractCollection#add
     */
public final synchronized boolean add(Object object)
  {
      if(size() >= capacity())
	  enlarge();

      list[nbr++] = object;

      return true;
  }
//______________________________________________________________________________
/**
 * Gets the element at the specified index.
 * @param i the index in the list.
 * @return the object.
 * @throw IndexOutOfBoundsException if the index is bad.
 */
final public Object get(int i)
    {
	return list[i];
    }
//______________________________________________________________________________
  /**
   * Gets the number of elements in the list.
   * @return the number.
   */
final public int size()
  {
      return nbr;
  }
//______________________________________________________________________________
  /**
   * Loads the content of the enumeration in the list.
   * <Br>
   * The list adapts its size if necessary.
   * @param e the collection to store.
   */
final public void set(Enumeration e)
  {
      for(int i = 0; e.hasMoreElements(); i++) {
	  add(e.nextElement());
      }
  }
//______________________________________________________________________________
  /**
   * Gets the length of the array storing the list.
   * @return the capacity.
   */
final public int capacity()
  {
      return list.length;
  }
//______________________________________________________________________________
  /**
   * Creates a larger list in place of the current one.
   * The capacity is incremented by CAPACITY_INCREMENT.
   * Moves elements to the new list.
   */
private final synchronized void enlarge()
  {
      Object[] larger = new Object[capacity() + CAPACITY_INCREMENT];
      for(int i = 0; i < size(); i++)
	  larger[i] = list[i];

      list = larger;
  }
//______________________________________________________________________________
}




