//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.tools;

import java.util.Enumeration;
import java.util.Iterator;
import java.util.Vector;
import java.util.NoSuchElementException;
import java.io.*;

/** 
 * NobjSet : a container for named object.
 * <P>
 * The implementation is based on the Vector's one. Note that it is not
 * an extension but a 'use' relationship.
 *
 * @version $Id: NobjSet.java,v 3.0 2000/08/23 13:15:35 lefevre Exp $
 * @author Jean-Paul Le Fvre
 * @see	Nobject
 * @see	Vector
 */
public	class NobjSet extends Nobject {

  /**
   * @serial the array storing the elements.
   */
  protected	Vector 	table;

//______________________________________________________________________________
  /**
   * Creates a Set. Initial capacity 50, increment 10.
   * @parameter str the name for the set.
   * @see Vector#Vector(int, int)
   * @see	Nobject
  */
public NobjSet(String name)
  {
   super(name);
   table = new Vector(50, 10);
  }
//______________________________________________________________________________
  /**
   * Reads a set from an input serial file.
   * It is a factory method.
   * @parameter input_name the name of the input file.
   * @returns a set if the operation was successfull, null otherwise.
   */
public	static NobjSet readSerial(String input_name)
  {
    if(ToolBox.verbose)
      ToolBox.warn("Reading serial " + input_name + " ...");

    NobjSet set;

    try {

      FileInputStream istream = new FileInputStream(input_name);
      ObjectInputStream p     = new ObjectInputStream(istream);

      set = (NobjSet)p.readObject();

      istream.close();

    }
    catch(Exception ex) {
      ToolBox.warn("Can't read file " + input_name);
      ToolBox.warn("Exception caught", ex); 
      return null;
    }
 
    return set;
  }
//______________________________________________________________________________
  /**
   * Writes down the Set to an Ascii file.
   * Write an header and a footer. Format names.
   * @param 	filename	the name of the file.
   * @return	false if something goes wrong, true otherwise.
   */
public boolean	write(String filename)
  {
    if(ToolBox.verbose)
      System.out.println("Writing " + getName() + " to " + filename + " !");

    OutputDataset file = new OutputDataset(filename);
    int max = getNameMaxLength();

    try {

      file.open();
      file.writeHeader(getName());

      for (Iterator e = table.iterator(); e.hasNext() ;) {

	Nameable n = (Nameable)e.next();
	String str = n.toString();
	file.writeLine(str);
      }

      file.writeFooter();
    }
    catch (Exception e) {
      System.err.println("Can't write file " + file.getName() + " !");
      System.err.println("Exception caught : " + e + " !"); 
      return false;
    }

    return true;
  }
//______________________________________________________________________________
  /**
   * Writes dwon the set to a  serial file.
   * @see Serializable
   * @parameter output_name the name of the output file.
   * @returns true is the operation was successfull, false otherwise.
   */
public	boolean writeSerial(String output_name)
  {
    try {

      if(ToolBox.verbose) {
	ToolBox.warn("Writing " + size() + " sites to serial "
			   + output_name + " ...");
      }

      FileOutputStream ostream = new FileOutputStream(output_name);
      ObjectOutputStream p     = new ObjectOutputStream(ostream);

      p.writeObject(this);

      p.flush();
      ostream.close();
    }
    catch(Exception ex) {
      ToolBox.warn("Can't write file " + output_name);
      ToolBox.warn("Exception caught", ex); 
      return false;
    }
 
    return true;
  }
//______________________________________________________________________________
  /*
   * Returns the length of the longest name in the set
   * Used to format output.
   * @return	the longest length.
   */
private final int getNameMaxLength()
  {
    int	max = 0;

    for (Iterator e = table.iterator(); e.hasNext() ;) {

      Nameable 	n = (Nameable)e.next();
      int	l = n.getName().length();
      if(l > max) max = l;
    }

    return max;
  }
//______________________________________________________________________________
  /**
   * Returns an iterator on the set.
   * @return	the iterator.
   */
public final synchronized Iterator iterator()
{
    return table.iterator();
}
//______________________________________________________________________________
  /**
   * Minimizes used memory.
   * @see Vector#trimToSize()
   */
public final synchronized void trimToSize()
  {
    table.trimToSize();
  }
//______________________________________________________________________________
  /**
   * Returns the number of elements in the set.
   * @return the size.
   * @see Vector#size
   */
public final int size()
  {
    return table.size();
  }
//______________________________________________________________________________
  /**
   * Returns the array of names of the elements in the set.
   * @see NobjSet#size
   * @return the array.
   */
public final String[] namesArray()
  {
      String[] array = new String[table.size()];

      int i = 0;

      for (Iterator e = table.iterator(); e.hasNext() ;) {

	  Nameable n = (Nameable)e.next();
	  array[i++] = new String(n.getName());
      }

      return array;
  }
//______________________________________________________________________________
  /**
   * Sorts the table of Nobject.
   * @see Nobject#compareTo
   */
public final void sort()
  {
      int size = table.size();
      Nobject tmp;
      Nobject[] array = new Nobject[size];

//    						Move nobjects to the temp array
      table.copyInto(array);
      table.removeAllElements();

//						Sort the array
      for(int i = size - 2; i >= 0; i--) {

	  int j;
	  for(j = i + 1; j < size; j++) {

	    if(array[i].compareTo(array[j]) > 0) {

		tmp = array[i];
		for(int k = i + 1; k < j; k++)
		    array[k-1] = array[k];
		array[j-1] = tmp;
		break;
	    }
	  }

	  if(j == size) {
		tmp = array[i];
		for(int k = i + 1; k < size; k++)
		    array[k-1] = array[k];
		array[size - 1] = tmp;
	  }
      }
//						Put back nobjects
      for(int i = 0; i < size; i++)
	  table.addElement(array[i]);

  }
//______________________________________________________________________________
  /**
   * Adds a Nameable element to the set.
   * @see	Vector#add
   * @parameter obj the item to add.
   */
public final void add(Nameable obj)
  {
    table.add(obj);
  }
//______________________________________________________________________________
  /**
   * Retrieves an element named 'str' in the set.
   * @parameter str the name of element to lookup.
   * @return a reference to the element or null if not found.
   */
public final Nameable	get(String str)
  {
    Nobject obj = new Nobject(str);
    int index   = table.indexOf(obj);

    if(index <  0)
      return null;

    return (Nameable)table.elementAt(index);
  }
//______________________________________________________________________________
  /**
   * Retrieves the element at position 'index' in the set.
   * @param  index the position.
   * @return a reference to the element.
   * @see	Vector#elementAt
   */
public final Nameable	get(int index)
  {
    return (Nameable)table.elementAt(index);
  }
//______________________________________________________________________________
  /**
   * Retrieves the first element in the set.
   * @return a reference to the element.
   * @see	Vector#elementAt
   */
public final Nameable	getFirst()
  {
    return (Nameable)table.elementAt(0);
  }
//______________________________________________________________________________
  /**
   * Retrieves the last element in the set.
   * @return a reference to the element.
   * @see	Vector#elementAt
   */
public final Nameable	getLast()
  {
    return (Nameable)table.elementAt(table.size() - 1);
  }
//______________________________________________________________________________
  /**
   * Retrieves the next element in the set. Return the first one if the input
   * object is the last one.
   * @param	an element in the set.	
   * @return 	a reference to the element found.
   * @exception NoSuchElementException if the input object is not in the set.
   */
public final Nameable	getForward(Nameable obj)
  {
    int index   = table.indexOf(obj);

    if(index < 0) throw new NoSuchElementException(obj.getName());

    if(++index >= table.size())	index = 0;


    return (Nameable)table.elementAt(index);
  }
//______________________________________________________________________________
  /**
   * Retrieves the previous element in the set. Return the last one if the input
   * object is the first one.
   * @param	an element in the set.	
   * @return 	a reference to the element found.
   * @exception NoSuchElementException if the input object is not in the set.
   */
public final Nameable	getBackward(Nameable obj)
  {
    int index   = table.indexOf(obj);

    if(index < 0) throw new NoSuchElementException(obj.getName());

    if(--index < 0)	index = table.size() - 1;

    return (Nameable)table.elementAt(index);
  }
//______________________________________________________________________________
  /**
   * Returns the position of an element in the set.
   * @param	the name of the element.
   * @return 	the position.
   * @see	Vector#indexOf
   */
public final int indexOf(String str)
  {
    Nobject obj = new Nobject(str);
    return table.indexOf(obj);
  }
//______________________________________________________________________________
  /**
   * Returns the position of an element in the set.
   * @param	the reference to the element.
   * @return 	the position.
   * @see	Vector#indexOf
   */
public final int indexOf(Nobject obj)
  {
    return table.indexOf(obj);
  }
//______________________________________________________________________________
  /**
   * Prints the content of the set  on stdout.
   */
public void dump()
  {
    System.out.println(
    "Set : " + getName() + " size : " + table.size() + " (" + table.capacity()
    + ") objects !");

    for (Iterator e = table.iterator(); e.hasNext() ;) {
      Nameable n = (Nameable)e.next();
      System.out.println(n.toString());
    }

    System.out.println();
  }
//______________________________________________________________________________
}




