//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.tools;

import java.util.AbstractList;
import java.util.Iterator;
import java.io.Serializable;

/** 
 * MiniSet : a very simple container meant to store 5 or 6 elements.
 * <P>
 * It is an oversimplified version of a Vector which can be used to
 * encapsulate an array of objects. It has the capability of growing
 * if needed : one more slot is allocated when the number of elements
 * is getting larger than the size of the array. Only the mandatory
 * methods are implemented. If more functionnalities are needed a
 * <I>Vector</I> should be used.
 *
 * @version $Id: MiniSet.java,v 3.0 2000/08/23 13:15:35 lefevre Exp $
 * @author Jean-Paul Le Fvre
 * @see	java.util.Vector
 */
public	class MiniSet extends AbstractList implements Serializable {
 /**
  * The version ID used to manage evolution.
  */
static final long serialVersionUID = -3453650378541511920L;

  /**
   * @serial the array storing the elements.
   */
  protected	Object[] 	set;
  /**
   * @serial the number of elements in the set.
   */
  private	int 	nbr;
  /**
   * @serial the default initial capacity.
   */
static final private int INITIAL_CAPACITY = 1;
  /**
   * @serial the default size increment.
   */
static final private int CAPACITY_INCREMENT = 1;
    /**
     * A very simple iterator which can be used to scan
     * a MiniSet. It implements the two methods <I>next()</I>
     * and <I>hasNext()</I>.
     */
    private class MiniIterator implements Iterator {
	/**
	 * The current index in the set.
	 */
	transient int index;
	/**
	 * The last index in the set. It is capacity - 1.
	 */
	transient int last_index;
	//______________________________________________________________________
	/**
	 * Creates and initializes the iterator.
	 * Must be checked : thread safe.
	 */
	public MiniIterator()
	{
	   last_index = size() - 1;
	   index = 0;
	}
	/**
	 * Gets the next element in the set.
	 * @return the next object.
	 */
	public final Object next()
	{
	    return set[index++];
	}
	/**
	 * Checks if there are more elements in the set.
	 * @return true if other element are available.
	 */
	public final boolean hasNext()
	{
	    return index <= last_index;
	}
	/**
	 * Removes from the underlying collection the last element returned
	 * by the iterator.
	 */
	public void remove()
	{
	    for(int i = index; i < nbr; i++) {
		set[i - 1] = set[i];
	    }
	    index--;
	    set[nbr - 1] = null;
	    nbr--;
	}
 	//______________________________________________________________________
   }
//______________________________________________________________________________
  /**
   * Creates a set with the initial capacity specified.
   * @param initialCapacity the initial size of the array.
   */
public MiniSet(int initialCapacity)
  {
      super();
      set = new Object[initialCapacity];
      nbr = 0;
  }
//______________________________________________________________________________
  /**
   * Creates a set with just one slot.
   * @see #MiniSet(int)
   */
public MiniSet()
  {
      this(1);
  }
//______________________________________________________________________________
  /**
   * Creates a string giving information about the object.
   * @return the string.
   */
public String toString()
  {
      StringBuffer buf = new StringBuffer();
      buf.append("[");

      for (java.util.Iterator it = iterator(); it.hasNext() ;) {
	  buf.append(" ").append(String.valueOf(it.next()));
      }
      buf.append(" ]");

      return buf.toString();
  }
//______________________________________________________________________________
    /**
     * Adds an object in the set. If the capacity is too small, the set
     * is recreated with more room available.
     * @param object the object to add.
     * @return true if the collection changed as a result of the call.
     * @see java.util.AbstractCollection#add
     */
public final synchronized boolean add(Object object)
  {
      if(contains(object))
	 return false;

      if(size() >= capacity())
	  enlarge();

      set[nbr++] = object;

      return true;
  }
//______________________________________________________________________________
/**
 * Removes an object from the set.
 * @param index the index of the object to remove.
 * @return the object removed.
 * @see java.util.AbstractList#remove
 * @throw IndexOutOfBoundsException if the index is bad.
 */
public final synchronized Object remove(int index)
  {
      Object object = set[index];

      for(int i = index + 1; i < nbr; i++) {
	  set[i - 1] = set[i];
      }
      set[nbr - 1] = null;
      nbr--;

      return object;
  }
//______________________________________________________________________________
/**
 * Gets the element at the specified index.
 * @param i the index in the set.
 * @return the object.
 * @throw IndexOutOfBoundsException if the index is bad.
 */
public final Object get(int i)
    {
	return set[i];
    }
//______________________________________________________________________________
  /**
   * Gets the number of elements in the set.
   * @return the number.
   */
public final int size()
  {
      return nbr;
  }
//______________________________________________________________________________
  /**
   * Gets the length of the array storing the set.
   * @return the capacity.
   */
public final int capacity()
  {
      return set.length;
  }
//______________________________________________________________________________
  /**
   * Creates a larger set in place of the current one.
   * The capacity is incremented by CAPACITY_INCREMENT.
   * Moves elements to the new set.
   */
private final synchronized void enlarge()
  {
      Object[] larger = new Object[capacity() + CAPACITY_INCREMENT];
      for(int i = 0; i < size(); i++)
	  larger[i] = set[i];

      set = larger;
  }
//______________________________________________________________________________
  /**
   * Gets an iterator over the elements contained in this set.
   * @return the iterator.
   * @see MiniIterator
   */
public final synchronized Iterator iterator()
  {
      return new MiniIterator();
  }
//______________________________________________________________________________
  /**
   * Prints the content of the set to stdout.
   */
public void dump()
  {
    System.out.println(toString());
    System.out.println();
  }
//______________________________________________________________________________
}




