//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.tools;

import java.io.*;
import java.net.*;

/**
 * InputDataset : an object designed to read data from.
 * <P>
 * It can be either a plain file or an url.
 * To scan the data a reader can be used.
 * One can choose different readers depending on the need.
 * A basic reader is already available in this object.
 * <P>
 * This object was designed at the very beginning of the
 * Java adventure. It would be a good idea to take advantage
 * of the new features of java.io package.
 *
 * @see	LinesReader
 * @see	Nobject
 * @see	DataInputStream
 * @version $Id: InputDataset.java,v 3.0 2000/08/23 13:15:35 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */

public	final	class InputDataset extends Nobject {
  /**
   * @serial The input stream.
   */
protected	DataInputStream input = null;
  /**
   * @serial The type file or url.
   */
private	boolean isurl;
  /**
   * @serial The current reader
   */
private	LinesReader	reader = null;
  /**
   * @serial The last modification time.
   */
private	long		modif;
  /**
   * The basic reader
   */
public 	static final int      RAW    = 0;
  /**
   * The bufferized reader
   */
public 	static final int      BUFFER = 1;
  /**
   * The shared bufferized reader
   */
public 	static final int      SHARED = 2;

//______________________________________________________________________________
/**
 * Creates an input data set.
 * <P>
 * If the substring "://" is found in name it is supposed to be an URL.
 *
 * @param name the name of the file or the url.
 */
public InputDataset(String name)
  {
    super(name);
    isurl = name.indexOf("://") >= 0;
  }
//______________________________________________________________________________
/**
 * Opens the input stream.
 * @exception	java.io.IOException	when the file cannot be open.
 */
final public void open() throws IOException
  {
      createReader();
  }
//______________________________________________________________________________
/**
 * Opens the input stream.
 * @exception	java.io.IOException	when the file cannot be open.
 */
final protected void openStream() throws IOException
  {
    if(isurl)
      openURL();
    else
      openFile();
  }
//______________________________________________________________________________
/**
 * Checks whether the dataset has been modified since the last call to Open.
 * If an exception occurs, return true.
 * @return true if it has been changed.
 */
public boolean isModified()
  {
      long last;
      try {
	  if(isurl) {
	      URL url = new URL(getName());
	      URLConnection co = url.openConnection();
	      last   = co.getLastModified();
	  }
	  else {
	      File file = new File(getName());
	      last      = file.lastModified();
	  }
      }
      catch(Exception ex) {
	  ToolBox.warn("Can't check dataset", ex);
	  return true;
      }

      return last > modif;
  }
//______________________________________________________________________________
/**
 * Opens the input stream when the dataset is a plain file.
 * @exception	java.io.IOException	when the file cannot be opened.
 */
private	void openFile() throws IOException
  {
    File file = new File(getName());
    modif     = file.lastModified();

    FileInputStream fin =  new FileInputStream(file.getPath());
    input = new DataInputStream(fin);

    if(false)
      System.err.println("Input file " + getName() + " open !");
  }
//______________________________________________________________________________
/**
 * Opens the input stream when the dataset is accessed thru an URL.
 * @exception	java.io.IOException	when the file cannot be open.
 */
private	void openURL() throws IOException
  {
    URL url = new URL(getName());
    URLConnection co = url.openConnection();
    modif   = co.getLastModified();

    input = new DataInputStream(co.getInputStream());

    if(false)
      System.err.println("Input url " + getName() + " open !");
  }
//______________________________________________________________________________
/**
 * Gets the source of a data set : either a file or an url.
 * Make the decision by looking at the substring ://
 *
 * @param str the location of the object.
 * @return the filename or the url object.
 */
public static Object getSource(String str)
  {
    if(str.indexOf("://") >= 0) {
      try {
	  return new URL(str);
      }
      catch(Exception ex) {
	  System.err.println("Invalid url " + str + " !");
	  return null;
      }
    }
    return str;
  }
//______________________________________________________________________________
/**
 * Creates a Reader Object of the specified type.
 * <P>
 * A reference is kept in this dataset.
 *
 * @param type the type of the LinesReader.
 * @return the LinesReader.
 * @see StreamReader
 * @see BufferReader
 * @see SharedReader
 * @exception	java.io.IOException	when the file cannot be opened.
 */
public LinesReader createReader(int type) throws IOException
  {
      openStream();

      switch(type) {
      case BUFFER :
	  reader = new BufferReader(this);
	  break;

      case SHARED :
	  reader = new SharedReader();
	  break;

      default :
	  reader = new StreamReader(this);
      }

      return reader;
  }
//______________________________________________________________________________
/**
 * Creates a basic Reader Object.
 * @return a StreamReader.
 * @exception	java.io.IOException	when the file cannot be opened.
 * @see #createReader(int)
 */
final public StreamReader createReader() throws IOException
  {
      return (StreamReader)createReader(RAW);
  }
//______________________________________________________________________________
/**
 * Closes the input stream.
 * @exception	java.io.IOException	when the file cannot be closed.
 */
final public void close() throws IOException
  {
    input.close(); input = null;
  }
//______________________________________________________________________________
  /**
   * Returns the Input Stream. Open the stream if necessary.
   * @return	the stream.
   * @exception	java.io.IOException	when the file cannot be closed.
   */
final public DataInputStream getStream() throws IOException
  {
      if(input == null) open();
      return input;
  }
//______________________________________________________________________________
  /**
   * Reads the next line in the file.
   * <P>
   * Skip empty lines. Get rid of comments.
   * @return	the line.
   * @see	StreamReader#nextLine
   */
final public String getNextLine()
  {
    return reader.nextLine();
  }
//______________________________________________________________________________
  /**
   * Returns the number of the current line.
   * @return	the number.
   * @see	StreamReader#lineNumber
   */
final public int lineNumber()
  {
    return reader.lineNumber();
  }
//______________________________________________________________________________
  /**
   * Prints info on stdout.
   */
public void dump()
  {
    System.out.println("Dumping InputDataset named : " + getName() + " !");
    File file = new File(getName());

    if(! file.exists()) {
      System.out.println("File does not exist !");
      return;
    }

    System.out.print("Path : "); System.out.println(file.getAbsolutePath());
  }
//______________________________________________________________________________
}



