//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.tools;
import java.io.DataInputStream;
import java.util.Vector;

/**
 * BufferReader : a bufferized Reader.
 * <br>
 * The lines read from the dataset are kept internally
 * in a buffer. So it is possible to rescan easily
 * the dataset if necessary.
 * @see	InputDataset
 * @version $Id: BufferReader.java,v 3.0 2000/08/23 13:15:35 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */

public	class BufferReader extends StreamReader {
  /**
   * The associated buffer.
   */
private	Vector	buffer;
  /**
   * The initial size of the buffer.
   */
private static final int      SIZE = 100;
  /**
   * The increment for  the buffer.
   */
private static final int      INCR = 100;

//______________________________________________________________________________
  /**
   * Creates a Reader. Allocates a buffer to store content of the dataset.
   * @param input the associated Dataset.
   * @see Vector
   */
public BufferReader(InputDataset input)
  {
      super(input);
      buffer = new Vector(SIZE, INCR);
  }
//______________________________________________________________________________
/**
 * Resets the Reader to the head of the buffer.
 */
final public synchronized void rewind()
  {
      reset();
  }
//______________________________________________________________________________
/**
 * Returns the number of elements in the buffer.
 * @return the size.
 */
final public synchronized int size()
  {
      return buffer.size();
  }
//______________________________________________________________________________
  /**
   * Adds a new line into the buffer.
   * @param str the line to add to the buffer.
   */
final public synchronized void addLine(String str)
  {
      buffer.addElement(str);
  }
//______________________________________________________________________________
  /**
   * Replaces a line in the buffer.
   * @param index the index of the line to replace.
   * @param str the new line to insert.
   */
final public synchronized void replaceLine(int index, String str)
  {
      buffer.set(index, str);
  }
//______________________________________________________________________________
  /**
   * Reads the next line in the file.
   * @return	the line.
   * @see #readNextLine
   * @see #getNextLine
   */
final public synchronized String nextLine()
  {
      return (line < (buffer.size())) ? getNextLine() : readNextLine();
  }
//______________________________________________________________________________
  /**
   * Reads a specified line either from the file or from the buffer.
   * @parameter index the number of the line to get.
   * @return the line.
   * @see #nextLine
   */
public synchronized String getLine(int index)
  {
      String str;

      if(index < buffer.size())
	  str = (String)buffer.elementAt(index);
      else
	  str = readLine(index);

      return str;
  }
//______________________________________________________________________________
  /**
   * Reads a specified line from the file.
   * @return the line.
   */
private	synchronized String readLine(int index)
  {
      String str = null;
      int nbread = buffer.size();

      for(int i = nbread; i <= index; i++) {
	  str = readNextLine();
      }

      return str;
  }
//______________________________________________________________________________
  /**
   * Reads the next line in the file.
   * @return the line.
   */
final private synchronized String readNextLine()
  {
      String str = super.nextLine();
      if(str != null) buffer.addElement(str);

      return str;
  }
//______________________________________________________________________________
  /**
   * Gets the next line from the buffer.
   * @return the line.
   */
final private synchronized String getNextLine()
  {
      String str = (String)buffer.elementAt(line);
      line++;

      return str;
  }
//______________________________________________________________________________
}
