//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.time;

import org.ariane.tools.ToolBox;
import java.util.Date;
import java.util.TimeZone;
import java.text.SimpleDateFormat;
import java.text.DateFormat;
import java.text.ParseException;

/**
 * Hour : manage a date like hh:mm.
 * @see	Date
 * @version $Id: Hour.java,v 3.0 2000/08/23 13:15:34 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */

public	final	class Hour extends Object implements Cloneable {
   /**
    * Used to separate hours from minutes.
    */
    private	static	String sep = ":";
   /**
    * The local time zone in Paris.
    */
    private	static	TimeZone TZ = Calendar.LTZ;
   /**
    * @serial The associated date.
    */
     private	Date when = null;

//______________________________________________________________________________
/**
 * Hour : Sets to the current time.
 * @see Date
 */
public Hour()
  {
    /*
     * The simple statement when = new Date() does not work properly.
     * If used, the time stored internally by the date is not consistent
     * with the one computed when the other constructor is called.
     * Therefore, comparisons of hours fail.
     */
      this(toString(new Date()));
  }
//______________________________________________________________________________
/**
 * Hour : parses a string hh:mm.
 * @param string the hour.
 * @see SimpleDateFormat
 */
public Hour(String string)
  {
      try {
	  SimpleDateFormat fmt = new SimpleDateFormat("H:mm");
	  fmt.setTimeZone(TZ);
	  when = fmt.parse(string.trim());
      }
      catch(ParseException ex) {
	  ToolBox.warn("Can't parse Hour " + string, ex);
      }
      catch(Exception ex) {
	  ToolBox.warn("Can't create Hour " + string, ex);
      }
  }
//______________________________________________________________________________
  /**
   * Creates a clone of the Hour. A new instance is allocated and a 
   * bitwise clone of the current object is place in the new object.
   * @return		a clone of this Hour.
   * @exception	OutOfMemoryError If there is not enough memory.
   */
public Object clone()
  {
    try {
	Hour h = (Hour)super.clone();
	h.when = new Date();
	h.when.setTime(when.getTime());
	return h;
    } catch (CloneNotSupportedException ex) {
      // this shouldn't happen, since we are Cloneable
      throw new InternalError();
    }
  }
//______________________________________________________________________________
  /**
   * Converts a date in String hh:mm . Adds leading 0 if necessary.
   * @return the string.
   */
private static String toString(Date date)
  {
      SimpleDateFormat fmt = new SimpleDateFormat("HH:mm");
      fmt.setTimeZone(TZ);

      return fmt.format(date);
  }
//______________________________________________________________________________
  /**
   * Converts Hour in String hh:mm . Adds leading 0 if necessary.
   * @return the string.
   */
public String toString()
  {
      return toString(when);
  }
//______________________________________________________________________________
  /**
   * Returns the number of milliseconds since
   * January 1, 1970, 00:00:00 GMT represented by this hour.
   * @return the number.
   */
final public long getTime()
  {
      return when.getTime();
  }
//______________________________________________________________________________
  /**
   * Adds dm minutes to this Hour.
   * @param dm	the number of minutes to add.
   * @see Calendar
   */
final public void addMinutes(int dm)
  {
    Calendar cal = new Calendar();
    cal.setCurrentTime(when);
    cal.add(Calendar.MINUTE, dm);
    when = cal.getCurrentTime();
  }
//______________________________________________________________________________
  /**
   * Checks whether this hour comes before the specified hour.
   * @param h the hour to compare
   * @return true if the original hour comes before the specified one.
   */
final public boolean before(Hour h)
  {
      return  when.before(h.when);
  }
//______________________________________________________________________________
  /**
   * Checks whether this hour comes after the specified hour.
   * @param h the hour to compare.
   * @return true if the original hour comes after the specified one.
   */
final public boolean after(Hour h)
  {
      return when.after(h.when);
  }
//______________________________________________________________________________
}


