//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.time;
import java.util.Date;
import java.util.TimeZone;
import java.util.SimpleTimeZone;
import java.text.SimpleDateFormat;

/**
 * Calendar : the Ariane's implementation.
 * @version $Id: Calendar.java,v 3.0 2000/08/23 13:15:34 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */

public	final	class Calendar extends java.util.GregorianCalendar {
   /**
    * @serial The local time zone in Paris.
    */
    public	static	TimeZone LTZ = TimeZone.getTimeZone("GMT");
   /**
    * @serial The reference time zone.
    */
    public	static	TimeZone UTC = new SimpleTimeZone(0, "UTC");
   /**
    * @serial The number of days in a month.
    */
    private	static	final int[] days_nbr = {31, 27, 31, 30, 31, 30,
				                31, 31, 30, 31, 30, 31};
   /**
    * @serial The number of days in a month.
    */
    private	static	final String[] month_name =
    {"January", "February", "March",     "April",   "May",      "June",
     "July",    "August",   "September", "October", "November", "December"};
   /**
    * @serial The number of milliseconds in a day.
    */
    private	static	final long DAY_MS = 24 * 3600 * 1000;


//______________________________________________________________________________
/**
 * Calendar : Constructs a default GregorianCalendar.
 */
public Calendar()
  {
      super();
      setTimeZone(LTZ);
      computeFields();
  }
//______________________________________________________________________________
/**
 * Calendar : Constructs a GregorianCalendar for the specified date.
 * @parameter d the date.
 */
public Calendar(Date d)
  {
      this();
      setCurrentTime(d);
  }
//______________________________________________________________________________
/**
 * Returns a calendar in the UTC timezone.
 * @parameter d the date.
 */
public static Calendar getUTCCalendar(Date d)
  {
      Calendar calendar = new Calendar();
      calendar.setTimeZone(Calendar.UTC);
      calendar.setCurrentTime(d);

      return calendar;
  }
//______________________________________________________________________________
  /**
   * Converts Calendar into String.
   * @return str	the string.
   */
public String toString()
  {
      SimpleDateFormat fmt = new SimpleDateFormat("dd/MM/yy HH:mm:ss");
      fmt.setTimeZone(LTZ);

      return fmt.format(getTime());
  }
//______________________________________________________________________________
  /**
   * Gets the number of days in the month. 
   * @return the number.
   */
final public int getMonthNumberOfDays()
  {
      int month = get(Calendar.MONTH);
      int nbr   = days_nbr[month];

      if(month == Calendar.FEBRUARY && isLeapYear(get(Calendar.YEAR))) {
	  nbr = 28;
      }

      return nbr;
  }
//______________________________________________________________________________
  /**
   * Gets the day of week for the n th day in month.
   * @parameter nth the index of the day in the month.
   */
final public void setDayInMonth(int nth)
  {
      if(nth < getMinimum(DAY_OF_MONTH) || nth > getMaximum(DAY_OF_MONTH))
	  throw new IllegalArgumentException("Invalid day " + nth);

      set(DAY_OF_MONTH, nth);
      complete();
  }
//______________________________________________________________________________
  /**
   * Gets the name of the month. 
   * @return the name.
   */
final public String getMonthName()
  {
      return month_name[get(Calendar.MONTH)];
  }
//______________________________________________________________________________
  /**
   * Gets this Calendar's current time as a long. 
   * @return current time.
   */
final public long getTimeInMillis()
  {
      complete();
      return super.getTimeInMillis();
  }
//______________________________________________________________________________
  /**
   * Gets this Calendar's current time. 
   * @return current time.
   */
final public Date getCurrentTime()
  {
      complete();
      return super.getTime();
  }
//______________________________________________________________________________
  /**
   * Sets this Calendar's current time. 
   * @param d the given Date
   */
final public void setCurrentTime(Date d)
  {
      super.setTime(d);
      computeFields();
  }
//______________________________________________________________________________
  /**
   * Increments or decrement the current month. 
   * @param forward if true set the month in the future, if false in the past.
   */
final public void moveMonth(boolean forward)
  {
      add(Calendar.MONTH, forward ? + 1 :  - 1);
      complete();
   }
//______________________________________________________________________________
  /**
   * Sets this Calendar to an other day either forward of backward. 
   * @param day the targetted day of the week.
   * @param forward if true set the day in the future, if false in the past.
   */
public void setToWeekDay(int day, boolean forward)
  {
      if(day < getMinimum(DAY_OF_WEEK) || day > getMaximum(DAY_OF_WEEK))
	  throw new IllegalArgumentException("Invalid day " + day);

      int today = get(DAY_OF_WEEK);
      int delta = day - today;

      if(delta <= 0 && forward)
	  delta += 7;

      else if(delta >= 0 && ! forward)
	  delta -= 7;

      setTimeInMillis(getTimeInMillis() + delta * DAY_MS);
      computeFields();
  }
//______________________________________________________________________________
/**
 * Fixes  the date.
 */
final public void fixFields()
  {
      complete();computeFields();
  }
//______________________________________________________________________________
  /**
   * Checks this Calendar settings. 
   */
public void checkFields()
  {
      if(areFieldsSet) 			System.out.println("FS 1");
      else				System.out.println("FS 0");

      if(isSet(Calendar.MILLISECOND)) 	System.out.println("MS 1");
      else				System.out.println("MS 0");
      if(isSet(Calendar.SECOND)) 	System.out.println("S  1");
      else				System.out.println("S  0");
      if(isSet(Calendar.MINUTE)) 	System.out.println("M  1");
      else				System.out.println("M  0");
      if(isSet(Calendar.HOUR)) 		System.out.println("H  1");
      else				System.out.println("H  0");
      if(isSet(Calendar.AM_PM)) 	System.out.println("AP 1");
      else				System.out.println("AP 0");

      if(isSet(Calendar.DAY_OF_MONTH))  System.out.println("DM 1");
      else				System.out.println("DM 0");
      if(isSet(Calendar.MONTH))  	System.out.println("Mo 1");
      else				System.out.println("Mo 0");
      if(isSet(Calendar.DAY_OF_WEEK))  	System.out.println("DW 1");
      else				System.out.println("DW 0");
      if(isSet(Calendar.WEEK_OF_YEAR))  System.out.println("WY 1");
      else				System.out.println("WY 0");

      System.out.println();
  }
//______________________________________________________________________________
  /**
   * Checks this Calendar settings. 
   */
public void printFields()
  {
      System.out.println(get(Calendar.HOUR)   + " h "  +
			 get(Calendar.MINUTE) + " m "  +
			 get(Calendar.AM_PM)  + " apm");

      System.out.println(get(Calendar.DAY_OF_MONTH) + " / " +
			 get(Calendar.MONTH)        + " / " +
			 get(Calendar.YEAR));

      System.out.println(get(Calendar.DAY_OF_WEEK)  + " dw " +
			 get(Calendar.DAY_OF_YEAR)  + " dy " +
			 get(Calendar.WEEK_OF_YEAR) + " wy");

      System.out.println("For " + toString() + "\n");
  }
//______________________________________________________________________________
}



