//______________________________________________________________________________

//	Java Test program
//______________________________________________________________________________

package org.ariane.test;

import org.ariane.tools.*;
import org.ariane.security.*;
import java.io.*;

/**
 * A test program for Controller.
 * <P>
 * The program can be used to sign the content of a file or
 * to verify the signature.
 * <Br>
 * Usage   : java org.ariane.test.TryController [-v|-d] [-c|-s]
 *                                              -u user [-p password] filename
 * <P>
 * It -s is specified the file is signed, if it is -c it the file is checked.
 * The signature is supposed to be in filename.sig~
 * The location of the keystore can be set on the command line using
 * the java -D flag, eg : java -DKeyStore.File=/mnt/share/adm/keystore ...
 *
 * @see Controller
 * @version $Id: TryController.java,v 3.1 2000/09/14 10:23:27 lefevre Exp $
 * @author  Jean-Paul Le Fvre
 */

class TryController extends TryProg {

    /**
     * @serial the user's name.
     */
private	String user = null;
    /**
     * @serial the user's password. Needed only to sign.
     */
private	String passwd = null;
    /**
     * @serial the file to sign.
     */
private	String filename = null;
    /**
     * @serial the extension for the signature file.
     */
private	final static String signext = ".sig";
    /**
     * @serial sign or check the file ?.
     */
private	boolean dosign = false;

//______________________________________________________________________________

public static void main (String args[])
  {

    TryController prog = new TryController("TryController");

    if (! prog.parseArg(args)) {
      System.err.println("Can't parse command line !");
      System.err.println("Usage : [-vd] [-c|s] -u user [-p password] filename");
      return;
    }

    if (! prog.run()) {
      System.err.println("Can't execute TryController !");
      return;
    }

  }
//______________________________________________________________________________
    /**
     * Creates the program.
     * @param name the name of the program.
     */
public TryController(String name)
  {
    super(name);
  }
//______________________________________________________________________________
    /**
     * Executes the test program.
     * @return false if it fails.
     */
public	boolean run()
  {
    try {

	Controller ctrl = new Controller();
	if(ToolBox.debug) ctrl.dump();

	String phrase = getPhrase();
	if(false) {
	    System.out.println();
	    System.out.println(phrase);
	}

	if(dosign) {
	    signPhrase(ctrl, phrase);
	}
	else {
	    boolean flag = verifyPhrase(ctrl, phrase);
	    if(flag) {
		ToolBox.warn("Congratulations Mister " + user);
	    }
	    else {
		ToolBox.warn("You're a liar Mister " + user);
	    }
	}
    }
    catch(Exception ex) {
	ToolBox.warn("Can't run TryController", ex);
	return false;
    }

    return true;
  }
//______________________________________________________________________________
    /**
     * Reads the phrase in the input file.
     * @return the string or null if it fails.
     */
final private String getPhrase()
  {
    try {
	FileInputStream input = new FileInputStream(filename);
	int size   = input.available();
	byte[] buf = new byte[size];
	int got    = input.read(buf);
	input.close();

	return new String(buf);
    }
    catch(Exception ex) {
	ToolBox.warn("Can't read " + filename, ex);
	return null;
    }
  }
//______________________________________________________________________________
    /**
     * Signs the phrase.
     * @param ctrl the controller.
     * @param phrase the phrase to sign.
     */
final private void signPhrase(Controller ctrl, String phrase)
  {
    try {
	long ti = System.currentTimeMillis();
	byte[] signature = ctrl.signPhrase(phrase, user, passwd.toCharArray());
	ToolBox.warn("Phrase signed in " + (System.currentTimeMillis() - ti)
	    + " ms. P/S sizes : " + phrase.length() + "/" + signature.length);

	FileOutputStream output =
	    new FileOutputStream(ToolBox.markAsTemp(filename + signext));
	output.write(signature);
	output.close();
    }
    catch(Exception ex) {
	ToolBox.warn("Can't sign the phrase", ex);
    }
  }
//______________________________________________________________________________
    /**
     * Verifies the phrase.
     * @param ctrl the controller.
     * @param phrase the phrase to check.
     * @return true is the phrase was written by the user.
     */
final private boolean verifyPhrase(Controller ctrl, String phrase)
  {
    try {
	FileInputStream input =
	    new FileInputStream(ToolBox.markAsTemp(filename + signext));
	int size         = input.available();
	byte[] signature = new byte[size];
	int got          = input.read(signature);
	input.close();

	return ctrl.verifyPhrase(phrase, user, signature);
    }
    catch(Exception ex) {
	ToolBox.warn("Can't verify the phrase", ex);
	return false;
    }
  }
//______________________________________________________________________________
    /**
     * Parses the command line.
     * <Br>
     * Usage   : [-v|d] [-c|s] -u user [-p password] filename
     */
public	boolean parseArg(String args[])
  {
    int i     = 0;
    int error = 0;

    while(i < args.length && args[i].startsWith("-")) {

      String arg = args[i++];
      int j;

      for (j = 1; j < arg.length(); j++) {

	char flag = arg.charAt(j);

	switch (flag) {
	case 's':
	  dosign = true;
	  break;

	case 'v':
	  ToolBox.verbose = true;
	  break;

	case 'd':
	  ToolBox.verbose = true;
	  ToolBox.debug   = true;
	  break;

	case 'u':
	  user = args[i++];
	  break;

	case 'p':
	  passwd = args[i++];
	  break;
	}
      }
    }

    if(i < args.length)
      filename = args[i];
    else
      return false;

    resources.copyFromSystem("KeyStore.File");

    return (error == 0);
  }
//______________________________________________________________________________
}

