//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.stats;

import java.util.Enumeration;
import java.awt.Dimension;
import org.ariane.tools.ToolBox;
import org.ariane.tools.Nobject;

/** 
 * ValueCountSet : a container for value countable objects.
 *
 * @version $Id: ValueCountSet.java,v 3.0 2000/08/23 13:15:34 lefevre Exp $
 * @author Jean-Paul Le Fvre
 * @see	java.util.Vector
 * @see	ValueCount
 */
public	class ValueCountSet extends CountSet implements Distribuable {

  /**
   * @serial the lower bound for the count. Outside that range
   * values are ignored.
   */
  private int mini = Integer.MIN_VALUE;
  /**
   * @serial the upper bound for the count.
   */
  private int maxi = Integer.MAX_VALUE;

//______________________________________________________________________________
  /**
   * Creates a Set named 'str'.
   * @see Nobject
   */
public ValueCountSet(String str)
  {
   super(str);
  }
//______________________________________________________________________________
  /**
   * Counts this new value. If out of bounds ignore it.
   * If already stored increment the count. Otherwise add the value.
   * @param val the value to count.
   */
public void count(int val)
  {
    if(val < mini || val > maxi) {
	return;
    }

    ValueCount vc = find(val);

    if(vc == null)
	add(new ValueCount(val, 1));
    else
	vc.increment();
  }
//______________________________________________________________________________
  /**
   * Finds the Object storing the value.
   * @param val the value to find.
   * @return the object or null if not found.
   */
private ValueCount find(int val)
  {
    ValueCount vc = new ValueCount(val);

    int index     = table.indexOf(vc);

    if(index <  0)
      return null;

    return (ValueCount)table.elementAt(index);
  }
//______________________________________________________________________________
/**
 * Adds 0 count to value not already added.
 * Adds 0 at the end to get nice histogram.
 * @see	#sort
*/
public final void fillUp()
{
    int max = maxValue() + 2;

    for(int i = 0; i < max; i++) {
	if(find(i) == null)
	    add(new ValueCount(i, 0));
    }
}
//______________________________________________________________________________
  /**
   * Sorts the table of ValueCounts according to the values (not to the counts).
   * @see ValueCount#compareTo
   */
public final void sort()
  {
      int size = table.size();
      ValueCount tmp;
      ValueCount[] array = new ValueCount[size];

//    						Move objects to the temp array
      table.copyInto(array);
      table.removeAllElements();

//						Sort the array
      for(int i = size - 2; i >= 0; i--) {

	  int j;
	  for(j = i + 1; j < size; j++) {

	    if(array[i].compareTo(array[j]) > 0) {

		tmp = array[i];
		for(int k = i + 1; k < j; k++)
		    array[k-1] = array[k];
		array[j-1] = tmp;
		break;
	    }
	  }

	  if(j == size) {
		tmp = array[i];
		for(int k = i + 1; k < size; k++)
		    array[k-1] = array[k];
		array[size - 1] = tmp;
	  }
      }
//						Put back objects
      for(int i = 0; i < size; i++)
	  table.addElement(array[i]);

  }
//______________________________________________________________________________
/**
 * Finds out the max value and the max count.
 * If the set if empty return 0,0
 * @return the dimension holding the values.
 * @see	#maxValue
 * @see	CountSet#maxCount
 */
public Dimension getBounds()
{
    if(size() < 1)
	return new Dimension();

    Dimension dim = new Dimension(maxValue(), maxCount());

    if(ToolBox.debug) System.out.println("Bounds : " + dim.toString());

    return dim;
}
//______________________________________________________________________________
/**
 * Gets the maximum value.
 * The set is NOT supposed to be sorted. The count is not checked.
 * @return the value.
 * @see	CountSet#maxCount
 */
public final int maxValue()
{
    int max = mini;

    for (Enumeration e = table.elements(); e.hasMoreElements() ;) {

	int value = ((ValueCount)e.nextElement()).value();
	if(value > max) max = value;
    }

    return max;
}
//______________________________________________________________________________
/**
 * Computes the mean value of the set.
 * @return the mean value.
 * @see	#meanSquareValue
 */
public final float meanValue()
{
    int value = 0;
    int total = 0;
    Enumeration e;

    for (e = table.elements(); e.hasMoreElements() ;) {
	ValueCount vc = (ValueCount)e.nextElement();
	int	  count = vc.getCount();
	total += count;
	value += count * vc.value();
    }

    if(total < 1) return 0;

    return ((float)value / (float)total);
}
//______________________________________________________________________________
/**
 * Computes the mean square value of the set.
 * Details are taken from Numerical Recipes.
 * @return the mean square value.
 * @see	#meanValue
 */
public final float meanSquareValue()
{
    float mean   = meanValue();
    double value = 0.;
    int total    = 0;
    Enumeration e;

    for (e = table.elements(); e.hasMoreElements() ;) {

	ValueCount vc = (ValueCount)e.nextElement();
	int	  count = vc.getCount();
	float	delta   = mean - vc.value(); 
	total += count;
	value += count * delta * delta;
    }

    if(total < 1) return 0;

    return (float)(Math.sqrt(value / (double)(total - 1)));
}
//______________________________________________________________________________
/**
 * Prints the content of the set on stdout.
 */
public void dump()
{
    super.dump();

    System.out.println("Mean value : " + meanValue() +
    " Mean square value : " + meanSquareValue() + " !");
    System.out.println("Bounds : " + getBounds().toString() + " !");

    System.out.println();
}
//______________________________________________________________________________
/**
 * Fills up arrays for the Plot.
 * @param v  the array of values.
 * @param c  the array of counts.
 */
public void getValues(float v[], float c[])
{
    int size = table.size();

    for(int i = 0; i < size; i++) {
	ValueCount vc = (ValueCount)table.elementAt(i);
	c[i] = vc.getCount();
	v[i] = vc.value();
    }
}
//______________________________________________________________________________
}
