//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.stats;

import java.io.IOException;
import java.util.Date;
import java.util.Calendar;
import java.util.Enumeration;
import org.ariane.tools.*;
import org.ariane.time.Duration;

/** 
 * DatedCountSet : a container for dated countable objects.
 *
 * @version $Id: DatedCountSet.java,v 3.0 2000/08/23 13:15:34 lefevre Exp $
 * @author Jean-Paul Le F&egrave;vre
 * @see	java.util.Vector
 */
public	class DatedCountSet extends CountSet {

//______________________________________________________________________________
  /**
   * Creates a Set named 'str'.
   * @see Nobject
   */
public DatedCountSet(String str)
  {
   super(str);
  }
//______________________________________________________________________________
  /**
   * Reads data from Ascii files.
   * <Br>
   * The data directory is libdir.
   * @param path the url where to find data file.
   * @exception IOException if data cannot be read.
   * @see InputDataset .
   * @see StreamReader .
   */
public void read(String path) throws IOException
  {
    if(ToolBox.debug) {
      System.out.print("Reading Dated Count Set " + getName());
      System.out.println(" in file " + path + " ...");
    }

    InputDataset input  = new InputDataset(path);
    StreamReader reader = input.createReader();

    String line;
    long   total = 0;

    while((line = reader.nextLine()) != null) {
	DatedCount dc = new DatedCount(line) ;
	add(dc);
	total += dc.getCount();
	if(ToolBox.debug) System.out.println(dc.toString());
    }

    input.close();
    trimToSize();	// Minimize used memory

    if(ToolBox.verbose) {
      System.out.println(getName() + " read " + size() + " elements. Total : "
      + total + " !");
    }
  }
//______________________________________________________________________________
  /**
   * Gets the duration between the first count and the last.
   * <P>
   * Note that the set is supposed to be sorted. No verification is performed.
   * If the set has less than 2 items a duration of 0 is returned.
   * @return the time interval.
   */
public Duration getDuration()
  {
      if(size() < 2)
	  return new Duration();

      Date d1 = ((DatedCount)getFirst()).date();
      Date d2 = ((DatedCount)getLast()).date();

      return new Duration(d1, d2);
  }
//______________________________________________________________________________
  /**
   * Rounds the date to hh:qh:00 0 for every elements in the set.
   * @see DatedCount#roundToQuarter
   */
final public void roundToQuarter()
  {
    Enumeration e;

    for (e = table.elements(); e.hasMoreElements() ;) {
       DatedCount dc = (DatedCount)e.nextElement();
       dc.roundToQuarter();
    }
  }
//______________________________________________________________________________
/**
 * Expands in a new set. The purpose is to get a regular set with samples
 * for every 0.5 h.
 * If the interval between two items in this set is greater than 0.5 h,
 * new items are created for the time missing with a count of 0
 * and inserted in the new set.
 * @return the new set.
 */
public DatedCountSet expand()
{
    DatedCountSet set = new DatedCountSet(getName());

    Enumeration e     = table.elements();
    DatedCount pc     = (DatedCount)e.nextElement();

    for (; e.hasMoreElements() ;) {

	DatedCount dc = (DatedCount)e.nextElement();
	int dh        = (new Duration(pc.date(), dc.date())).inHalfHours();

	if(dh < 1) {
	    ToolBox.warn("DatedCountSet : value ignored : " + dc.toString());
	    continue;
	}

	set.add(pc);

	if(dh > 1) {

	    DatedCount nc = pc;

	    for(int h = 1; h < dh; h++) {

		nc     = new DatedCount(nc.date(), 30);
		set.add(nc);
	      }
	}
	pc = dc;
    }

    set.add(pc);

    set.trimToSize();

    return set;
}
//______________________________________________________________________________
/**
 * Compress in a new set. Summ up the counts.
 * The interval is specified in number of 0.5 h.
 * The set is supposed to be regular.
 * @param dt  the interval of time.
 * @return the new set.
 */
public DatedCountSet compress(int dt)
{
    if(dt < 2)
	return this;

    DatedCountSet set = new DatedCountSet(getName());

    Enumeration e     = table.elements();
    DatedCount pc     = null;
    int count         = 0;

    for (; e.hasMoreElements() ;) {

	DatedCount dc = (DatedCount)e.nextElement();

	if(pc == null) {
	    count = dc.getCount();
	    pc    = dc;
	    continue;
	}

	int dh = (new Duration(pc.date(), dc.date())).inHalfHours() + 1;

	if(dh < dt) {
	    count += dc.getCount();
	}
	else if(dh == dt) {
	    count        += dc.getCount();
	    DatedCount nc = new DatedCount(pc.date(), 0);
	    nc.setCount(count);
	    set.add(nc);
	    pc = null;
	}
    }

    set.trimToSize();
  
    return set;
}
//______________________________________________________________________________
}




