//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package	org.ariane.stats;

import java.util.Date;
import org.ariane.time.Time;
import org.ariane.time.Calendar;
import java.text.SimpleDateFormat;
import java.text.ParseException;

/**
 * DatedCount : a class managing some count at some date.
 *
 * @see Calendar
 * @version $Id: DatedCount.java,v 3.0 2000/08/23 13:15:34 lefevre Exp $
 * @author  Jean-Paul Le Fvre
 */
public	class DatedCount extends ObjectCount {

//______________________________________________________________________________
  /**
   * Initialisation to : now and 0.
   */
public DatedCount()
  {
      super(new Calendar(), 0);
      getCalendar().setTime(new Date());
  }
//______________________________________________________________________________
  /**
   * Initialisation to : the date + dm minutes and 0.
   */
public DatedCount(Date d, int dm)
  {
      super(new Calendar(), 0);
      getCalendar().setTime(d);
      getCalendar().add(Calendar.MINUTE, dm);
  }
//______________________________________________________________________________
  /**
   * Initializes from a line 'date : count'
   * The date must comply with the syntax of ParseDate().
   * It seems necessary to specify the time zone for Netscape.
   * @param str the date & count.
   * @exception IllegalArgumentException if str is not correct.
   * @see #parseDate
   */
public DatedCount(String str) throws IllegalArgumentException
  {
      String sep  = new String(" : ");	// The separator
      int    lep  = 4;			// Its length

      int i  = str.indexOf(sep);
      if(i < 0) throw new IllegalArgumentException("No separator found");

      try {
	  object = parseDate(str.substring(0, i));
	  count  = Integer.parseInt(str.substring(i+lep).trim());
      }
      catch(Exception ex) {
	  throw new IllegalArgumentException("Can't parse " + str);
      }
  }
//______________________________________________________________________________
  /**
   * Returns the day of the date.
   * @see Time
   * @return the string dd/mm.
   */
final public String day()
  {
      Time t = new Time(getCalendar());

      return t.dayMonth();
  }
//______________________________________________________________________________
  /**
   * Returns the day of the week of the date.
   * I don't understand when this day is changed !
   * @see Calendar
   * @return a value between 0 and 6. Sunday : 0.
   */
final public int weekDay()
  {
      return getCalendar().get(Calendar.DAY_OF_WEEK);
  }
//______________________________________________________________________________
  /**
   * Returns the day of the month of the date.
   * @see Calendar
   * @return a value between 1 and 31.
   */
final public int monthDay()
  {
      return getCalendar().get(Calendar.DAY_OF_MONTH);
  }
//______________________________________________________________________________
  /**
   * Returns a reference to the current date.
   * @return the date object.
   */
final public Date date()
  {
    return getCalendar().getTime();
  }
//______________________________________________________________________________
  /**
   * Returns a reference to the current calendar.
   * @return the calendar object.
   */
final private Calendar getCalendar()
  {
    return (Calendar)object;
  }
//______________________________________________________________________________
  /**
   * Builds the line : 'object : count'
   * @return the string.
   */
public String	toString()
  {
      return date().getTime() + " " + weekDay() + " " + monthDay()
	  + " " + " : " + count;
  }
//______________________________________________________________________________
  /**
   * Rounds the date to hh:qh:00 0 the minutes are rounded to the quarter.
   */
final public void roundToQuarter()
  {
      Calendar c = getCalendar();

      c.set(Calendar.MILLISECOND, 0);
      c.set(Calendar.SECOND,     0);
      int m = 15 * (c.get(Calendar.MINUTE) / 15);
      c.set(Calendar.MINUTE, m);
  }
//______________________________________________________________________________
/**
 * Checks if the count is in the window [t1-tf]
 * @param ti  the initial hour defining the window
 * @param tf  the final hour defining the window 
 * @return true if the count is in the interval
 */
final public boolean isInInterval(int ti, int tf)
{
    if(ti < 1 && tf > 23)
	return true;

    int h = getCalendar().get(Calendar.HOUR);

    if(ti < tf) {
	return ti <= h && h <= tf;
    }
    else if (ti > tf) {
	return (ti <= h && h <= 23) || (0 <= h && h <= tf);
    }

    return false;
}
//______________________________________________________________________________
  /**
   * Parses the date.
   * The date must comply with the format : mm/dd/yy hh:mm:ss
   * @param str the date.
   * @see SimpleDateFormat
   * @exception ParseException
   */
private Calendar parseDate(String str) throws ParseException
  {
      SimpleDateFormat fmt = new SimpleDateFormat("MM/dd/yyyy HH:mm:ss");
      Calendar         cal = new Calendar();
      fmt.setTimeZone(Calendar.LTZ);
      cal.setTime(fmt.parse(str.trim()));

      return cal;
  }
//______________________________________________________________________________
    /**
     * Test program used to check this ojbect.
     * @see #parseDate
     */
public static void main (String args[])
    {
	if(args.length < 1) {
	    System.err.println(
		   "Usage : java org.ariane.stats.DatedCount string");
	    return;
	}
	
	System.out.println("string : " + args[0]);
	DatedCount dc = new DatedCount(args[0]);
	System.out.println(dc.toString());
    }
//______________________________________________________________________________
}



