//______________________________________________________________________________

//	The Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.security;

import java.security.*;
import java.security.spec.*;
import java.security.cert.Certificate;
import java.io.InputStream;
import java.io.FileInputStream;
import java.io.IOException;
import org.ariane.tools.ToolBox;
import org.ariane.tools.Resources;

/**
 * Controller : a object used to control identity.
 * <Br>
 * It provides methods to sign a phrase and to verify a signature.
 * <P>
 * See the <A Href =
 * "http://java.sun.com/docs/books/tutorial/security1.2/apisign/index.html">
 * related pages</A> in the JDK tutorial.
 *
 * @version $Id: Controller.java,v 3.0 2000/08/23 13:15:34 lefevre Exp $
 * @author Jean-Paul Le Fvre
 * @see org.ariane.test.TryController
 * @see org.ariane.orbec.DobjectFactory
 */
//______________________________________________________________________________

public	class Controller {
    /**
     * @serial the keystore used to manage the keys.
     */
private KeyStore keystore = null;
    /**
     * @serial the signature object.
     */
private Signature signer = null;

//______________________________________________________________________________
/**
 * Creates a Controller.
 * <Br>
 * It loads the keystore from a file.
 * The signature object is created with the algorithm SHA1withDSA.
 *
 * @throw SecurityException if the keystore cannot be loaded.
 * @see #loadKeyStore
 * @see Signature
 */
public Controller() throws SecurityException
  {
      loadKeyStore();
      try {
	  signer = Signature.getInstance("SHA1withDSA");
      }
      catch(GeneralSecurityException ex) {
	  throw new SecurityException("Can't create signer " + ex.getMessage());
      }
  }
//______________________________________________________________________________
/**
 * Loads the keystore.
 * <Br>
 * It loads the keystore from a file. The name of the file is taken
 * in the resources, the property is <Code>KeyStore.File</Code>.
 * If not found the default
 * <Code>user.home/.keystore</Code> is tried.
 * 
 * @throw SecurityException if the keystore cannot be loaded.
 */
private void loadKeyStore() throws SecurityException
  {
      Resources rsrc = Resources.instance();
      String deffile = System.getProperty("user.home")
                     + System.getProperty("file.separator") + ".keystore";
      String file    = rsrc.get("KeyStore.File", deffile);

      try {

	  InputStream input = new FileInputStream(file);
	  keystore          = KeyStore.getInstance(KeyStore.getDefaultType());
	  keystore.load(input, null);
	  if(ToolBox.debug) ToolBox.warn("Keystore loaded from " + file);
      }
      catch(IOException ex) {
	  throw new SecurityException("Can't read " + file);
      }
      catch(GeneralSecurityException ex) {
	  keystore = null;
	  throw new SecurityException("Can't load keystore " + ex.getMessage());
      }
  }
//______________________________________________________________________________
/**
 * Signs a phrase.
 * <Br>
 * The user's key is retrieved in the keystore. A genuine private key
 * is generated from it. The spec is supposed to be PKCS8.
 * The signature is carried out.
 *
 * @param phrase the phrase to sign.
 * @param alias the name of the person who signs.
 * @param password the password of the person who signs.
 * @return the signature.
 * 
 * @throw SecurityException if the phrase cannot be signed.
 * @see KeyFactory#generatePrivate
 * @see Signature#sign
 */
final public byte[] signPhrase(String phrase, String alias, char[] password)
                                                     throws SecurityException
  {
      if(phrase == null || alias == null || password == null)
	  throw new IllegalArgumentException("Invalid null arguments");

      try {
	  Key key = keystore.getKey(alias, password);
	  if(key == null) throw new SecurityException("No key for " + alias);
	      
	  PKCS8EncodedKeySpec spec = new PKCS8EncodedKeySpec(key.getEncoded());

	  KeyFactory factory = KeyFactory.getInstance("DSA");
	  PrivateKey prive   = factory.generatePrivate(spec);

	  signer.initSign(prive);
	  signer.update(phrase.getBytes());

	  return signer.sign();
      }
      catch(GeneralSecurityException ex) {
	  throw new SecurityException("Can't sign the phrase ("
				      + ex.getMessage() + ")");
      }
  }
//______________________________________________________________________________
/**
 * Signs a phrase.
 * <Br>
 * The user's key is retrieved in the keystore. A genuine private key
 * is generated from it. The spec is supposed to be PKCS8.
 * The signature is carried out.
 *
 * @param phrase the phrase to sign.
 * @param alias the name of the person who signs.
 * @param password the password of the person who signs.
 * @return the signature.
 * 
 * @throw SecurityException if the phrase cannot be signed.
 * @see KeyFactory#generatePrivate
 * @see Signature#sign
 * @deprecated
 */
final public byte[] SignPhrase(String phrase, String alias, char[] password)
                                                     throws SecurityException
    {
	return signPhrase(phrase, alias, password);
  }
//______________________________________________________________________________
/**
 * Verifies the signature of a phrase.
 * <P>
 * The user's certificate is retrieved in the keystore. The embedded
 * public key is taken inside and used to check the signature.
 *
 * @param phrase the phrase to check.
 * @param alias the name of the person who signs.
 * @param signature the signature of the person who signs.
 * @return true if the signature is correct.
 * @see Certificate
 * @see Signature#verify
 */
final public boolean verifyPhrase(String phrase, String alias, byte[] signature)
  {
      if(phrase == null || alias == null || signature == null)
	  throw new IllegalArgumentException("Invalid null arguments");

      try {
	  Certificate certif = keystore.getCertificate(alias);
	  if(certif == null) {
	      ToolBox.warn("User " + alias + " unknown");
	      return false;
	  }

	  PublicKey key = certif.getPublicKey();
	  signer.initVerify(key);
	  signer.update(phrase.getBytes());

	  return signer.verify(signature);
      }
      catch(GeneralSecurityException ex) {
	  if(ToolBox.debug) ex.printStackTrace();
	  ToolBox.warn("Can't verify signature", ex);
	  return false;
      }
      catch(Exception ex) {
	  if(ToolBox.debug) ex.printStackTrace();
	  ToolBox.warn("Can't verify signature", ex);
	  return false;
      }
  }
//______________________________________________________________________________
/**
 * Gets a user's certificate.
 * @param alias the user's name.
 * @return the certicate.
 * @see KeyStore#getCertificate
 * @throw KeyStoreException if the certificate cannot be got.
 */
final public Certificate getCertificate(String alias) throws KeyStoreException
  {
    return keystore.getCertificate(alias);
  }
//______________________________________________________________________________
/**
 * Converts Controller to String.
 * @return str	the string.
 */
public String toString()
  {
    return "Controller";
  }
//______________________________________________________________________________
/**
 * Checks content of this object.
 */
public void dump()
  {
      if(keystore == null) {
	  System.out.println("Keystore is null");
	  return;
      }

      StringBuffer buf = new StringBuffer("Keystore type ");
      buf.append(keystore.getType()).append(" provider ");
      buf.append(keystore.getProvider().getName());
      
      System.out.println(buf);
  }
//______________________________________________________________________________
}
