//______________________________________________________________________________

//	The Java Virtual Shelf

//______________________________________________________________________________

package org.ariane.net;

import org.ariane.tools.ToolBox;
import org.ariane.exec.RunnableTask;
import java.net.Socket;
import java.io.*;

/**
 * TagConnection : a object handling a tag request.
 * <br>
 * This object is created whenever a connection to the server takes place.
 *
 * @see TagServer
 * @version $Id: TagConnection.java,v 3.0 2000/08/23 13:15:33 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */
//______________________________________________________________________________

public	class TagConnection extends RunnableTask {
  /**
   * @serial The server.
   */
private	TagServer server;
  /**
   * @serial The client socket.
   */
private	Socket socket;
  /*
   * @serial The mark for the server host.
   */
private	HostTag tag;
  /*
   * @serial The command hello.
   */
public	static	final	byte	HI  = 1;
  /*
   * @serial The command request tag.
   */
public	static	final	byte	TAG = 2;
  /*
   * @serial The command shutdown.
   */
public	static	final	byte	DOWN = 3;
  /*
   * @serial The command bybye.
   */
public	static	final	byte	BYE = 4;
  /*
   * @serial The undefined command.
   */
public	static	final	byte	NDF = 9;

//______________________________________________________________________________
/**
 * Creates a TagConnection. It stores references on the server and the hosttag.
 * @param h the server.
 * @param s the established socket.
 * @param t the host tag.
 */
public TagConnection(TagServer h, Socket s, HostTag t)
  {
      server = h;
      socket = s;
      tag    = t;
  }
//______________________________________________________________________________
  /**
   * Launches the task. Reads the command.
   * <br>
   * Touch the tag if requested. Send back the tag to the client.
   * May be used to kill the server.
   * @see HostTag#touch
   */
public void run()
{
    done    = false;
    if(ToolBox.debug) ToolBox.warn("TagConnection  started");

    try {
	DataInputStream  in  = new DataInputStream(socket.getInputStream());
	byte cmd	     = receiveCommand(in);
	DataOutputStream out = new DataOutputStream(socket.getOutputStream());

	switch(cmd) {

	case TAG : 
	    tag.touch();
	    ToolBox.warn("Connection " + tag.toString());
	    sendString(out, tag.toString());
	    break;

	case DOWN :
	    ToolBox.warn("Server goes dead");
	    server.stop();
	    break;

	default :
	    ToolBox.warn("Unknown command " + cmd);
	    sendString(out, "What ?");
	}
    }
    catch(IOException ex) {
	ToolBox.warn("Can't communicate", ex);
    }

    done = true;

    if(ToolBox.debug) ToolBox.warn("TagConnection done");
}
//______________________________________________________________________________
  /**
   * Receives a command from the stream.
   * @parameter in the input stream.
   * @return the code for the command.
   * @exception IOException if the command cannot be read.
   */
private byte receiveCommand(DataInputStream in) throws IOException
  {
      int   max  = 4;
      byte[] cmd = new byte[max];

      in.readFully(cmd, 0, max);

      if(cmd[0] != HI || cmd[2] != BYE) {
	  ToolBox.warn("Can't understand cmd " +
		       cmd[0] + " " + cmd[1] + " " + cmd[2] + " " + cmd[3]);
	  return NDF;
      }

      return cmd[1];
  }
//______________________________________________________________________________
  /**
   * Sends the tag into the stream.
   * @parameter out the output stream.
   * @parameter str the string to send.
   * @exception IOException if the command cannot be written.
   */
private void sendString(DataOutputStream out, String str) throws IOException
  {
      out.writeBytes(str); out.writeByte('\n') ;out.flush();
  }
//______________________________________________________________________________
}


