//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.net;

import org.ariane.tools.TryProg;
import org.ariane.tools.ToolBox;

import org.ariane.net.TagConnection;
import org.ariane.net.TagServer;
import java.net.Socket;
import java.io.*;

/**
 * TagClient : a client requesting tags from a server.
 * <Br>
 * It can be used to shutdown the server.
 * It enters a loop asking for an option from the user.
 *
 * @version $Id: TagClient.java,v 3.0 2000/08/23 13:15:33 lefevre Exp $
 * @author  Jean-Paul Le F&egrave;vre
 */

class TagClient extends TryProg {
  /**
   * The name of the server of tags.
   */
private	String 	server;
  /**
   * The port associated to the service.
   */
private	int 	port;

//______________________________________________________________________________

public static void main (String args[])
  {
      ToolBox.debug = true;
      TagClient prog = new TagClient();

      if (! prog.parseArg(args)) {
	  ToolBox.warn("Can't parse command line");
	  return;
      }

      if (! prog.run()) {
	  ToolBox.warn("Can't execute command");
	  return;
      }
  }
//______________________________________________________________________________
  /**
   * Creates the client.
   * The server and the port number are read from resources.
   */
public TagClient()
  {
    super("TagClient");
    server = resources.get(TagServer.APPLI + ".Server");
    port   = resources.get(TagServer.APPLI + ".Port", -1);

    if(ToolBox.debug) ToolBox.warn("Connection to " + server + " " + port);
  }
//______________________________________________________________________________
  /**
   * Executes the program : it launches a loop and wait for an answer.
   */
public	boolean run()
  {
    boolean again = true;
    ToolBox.warn("TagClient started ...");

    while(again) {

	System.out.println();
	System.out.print("Option : "); System.out.flush();
	String answer = ToolBox.readInputString();

	if(answer.equalsIgnoreCase("t")) {

	    requestTag();
	}
	else if(answer.equalsIgnoreCase("s")) {

	    shutDown();
	    again = false;
	}
	else if(answer.equalsIgnoreCase("q")) {
	    again = false;
	}
	else if(answer.equalsIgnoreCase("x")) {
	    again = false;
	}
	else {
	    ToolBox.warn("Valid options : t(ag) s(shutdown) q(uit)");
	}
    }

    ToolBox.warn("TagClient terminated");

    return true;
  }
//______________________________________________________________________________
  /**
   * Shutdowns the server : send the specific command to the server.
   */
private void shutDown()
  {
      if(ToolBox.debug) ToolBox.warn("ShutDown " + server);

      try {
	  Socket socket	       = new Socket(server, port);
	  DataOutputStream out = new DataOutputStream(socket.getOutputStream());

	  sendCommand(out, TagConnection.DOWN);
	  requestTag();
      }
      catch(IOException ex) {
	  ToolBox.warn("Can't shutdown " + server, ex);
      }
  }
//______________________________________________________________________________
  /**
   * Asks for a tag from a server. Reads the reply.
   */
private void requestTag()
  {
      if(ToolBox.debug) ToolBox.warn("RequestTag " + server);

      try {
	  Socket socket        = new Socket(server, port);

	  DataOutputStream out = new DataOutputStream(socket.getOutputStream());
	  sendCommand(out, TagConnection.TAG);

	  BufferedReader  in  = new BufferedReader(
			        new InputStreamReader(socket.getInputStream()));
	  String tag           = receiveString(in);

	  System.out.println(tag);
      }
      catch(IOException ex) {
	  ToolBox.warn("Can't get tag from " + server, ex);
      }
  }
//______________________________________________________________________________
  /**
   * Sends a command into the stream.
   * @parameter out the output stream.
   * @parameter cmd the code for the command.
   * @exception IOException if the command cannot be written.
   */
private void sendCommand(DataOutputStream out, byte cmd) throws IOException
  {
      out.writeByte(TagConnection.HI);
      out.writeByte(cmd);
      out.writeByte(TagConnection.BYE);
      out.writeByte('\n');out.flush();
  }
//______________________________________________________________________________
  /**
   * Receives a tag from the stream.
   * @parameter in the input stream.
   * @return the string representing the tag.
   * @exception IOException if the command cannot be read.
   */
private String receiveString(BufferedReader in) throws IOException
  {
      return in.readLine();
  }
//______________________________________________________________________________
  /**
   * Parses the command line.
   * Usage   : -v|d
   */
public	boolean parseArg(String args[])
  {
    int i     = 0;
    int error = 0;

    while(i < args.length && args[i].startsWith("-")) {

      String arg = args[i++];
      int j;

      for (j = 1; j < arg.length(); j++) {

	char flag = arg.charAt(j);

	switch (flag) {
	case 'v':
	  ToolBox.verbose = true;
	  break;

	case 'd':
	  ToolBox.verbose = true;
	  ToolBox.debug   = true;
	  break;
	}
      }
    }

    return (error == 0);
  }
//______________________________________________________________________________
}

