//______________________________________________________________________________

//	The Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.gui;

import org.ariane.tools.Resources;
import java.awt.*;
import javax.swing.BorderFactory;
import javax.swing.border.Border;
import javax.swing.ImageIcon;
import javax.swing.Icon;
import javax.swing.JComponent;

/**
 * A GUI Resources object.
 * <Br>
 * It provides values for various parameters used to display a GUI.
 * Some values, the most frequently needed, are cached inside this
 * object. The name of the attributes is built with a prefix depending
 * on the client application. A concrete Class inheriting from this
 * Class must implement the method returning this prefix.
 *
 * @version $Id: GuiResources.java,v 3.1 2000/10/10 19:37:56 lefevre Exp $
 * @author Jean-Paul Le Fvre
 * @see org.ariane.tools.Resources
 */
//______________________________________________________________________________

abstract public class GuiResources implements GuiResourcesOp {
    /**
     * @serial The font used for headers.
     */
private Font header_font;
    /**
     * @serial The font used for labels.
     */
private Font label_font;
    /**
     * @serial The font used for texts.
     */
private Font text_font;
    /**
     * @serial the main background color.
     */
private Color main_bg;
    /**
     * @serial the main foreground color.
     */
private Color main_fg;
    /**
     * @serial the pane background color.
     */
private Color pane_bg;
    /**
     * @serial the pane foreground color.
     */
private Color pane_fg;
    /**
     * @serial the board background color.
     */
private Color board_bg;
    /**
     * @serial the board foreground color.
     */
private Color board_fg;
    /**
     * @serial the selection color.
     */
private Color select_bg;
    /**
     * @serial the common border.
     */
private Border border;
    /**
     * @serial the common low border.
     */
private Border in_border;
    /**
     * @serial the common high border.
     */
private Border out_border;
    /**
     * @serial the common line border.
     */
private Border line_border;
    /**
     * @serial the common matte border.
     */
private Border matte_border;
    /**
     * @serial the common empty border.
     */
private Border margin_border;
    /**
     * @serial the value used to defined spaces.
     */
private int space;

//______________________________________________________________________________
/**
 * Creates the GuiResources object.
 */
protected GuiResources()
  {
      String prefix  = getAppplicationPrefix();
      Resources rsrc = Resources.instance();

      header_font = rsrc.get(prefix + "Header.Font","Helvetica", Font.BOLD,14);
      label_font  = rsrc.get(prefix + "Label.Font", "Helvetica", Font.PLAIN,14);
      text_font   = rsrc.get(prefix + "Text.Font",  "Helvetica", Font.PLAIN,12);

      main_bg     = rsrc.get(prefix + "Background", Color.cyan);
      main_fg     = rsrc.get(prefix + "Foreground", Color.black);
      pane_bg     = rsrc.get(prefix + "Pane.Background", Color.white);
      pane_fg     = rsrc.get(prefix + "Pane.Foreground", Color.black);
      board_bg    = rsrc.get(prefix + "Board.Background", pane_bg);
      board_fg    = rsrc.get(prefix + "Board.Foreground", pane_fg);
      select_bg   = rsrc.get(prefix + "Selection.Background", Color.yellow);
      space       = rsrc.get(prefix + "Space", 2);

      border        = BorderFactory.createEtchedBorder();
      in_border     = BorderFactory.createLoweredBevelBorder();
      out_border    = BorderFactory.createRaisedBevelBorder();
      line_border   = BorderFactory.createLineBorder(main_fg);
      matte_border  = BorderFactory.createMatteBorder(
				   space,space,space,space, main_bg);
      margin_border = BorderFactory.createEmptyBorder(
				   space,space,space,space);
  }
//______________________________________________________________________________
/**
 * Gets the prefix used to create attibute names.
 * <Br>
 * The separator '.' is expected to be appended.
 * <Br>
 * Example : if the prefix returned by this method is
 * <Code>"MyAppli."<Code> the background color will be the value
 * defined by <Code>"MyAppli.Background=white"<Code>
 *
 * @return the prefix.
 */
abstract public String getAppplicationPrefix();

//______________________________________________________________________________
    /**
     * Gets an integer value with a possible default value.
     * The application prefix is automatically inserted in the key.
     * @param key the key.
     * @param def the default.
     * @return the value.
     * @see org.ariane.tools.Resources#get(String, int)
     */
final public int get(String key, int def)
  {
      return Resources.instance().get(getAppplicationPrefix() + key, def);
  }
//______________________________________________________________________________
    /**
     * Gets a string value with a possible default value.
     * The application prefix is automatically inserted in the key.
     * @param key the key.
     * @param def the default.
     * @return the value.
     * @see org.ariane.tools.Resources#get(String, String)
     */
final public String get(String key, String def)
  {
      return Resources.instance().get(getAppplicationPrefix() + key, def);
  }
//______________________________________________________________________________
    /**
     * Gets a boolean value with a default value.
     * The application prefix is automatically inserted in the key.
     * @param key the key.
     * @param def the default.
     * @return the value.
     * @see org.ariane.tools.Resources#get(String, boolean)
     */
final public boolean get(String key, boolean def)
  {
      return Resources.instance().get(getAppplicationPrefix() + key, def);
  }
//______________________________________________________________________________
    /**
     * Gets a string value.
     * The application prefix is automatically inserted in the key.
     * @param key the key.
     * @return the value or null if not found.
     * @see org.ariane.tools.Resources#get(String)
     */
final public String get(String key)
  {
      return Resources.instance().get(getAppplicationPrefix() + key);
  }
//______________________________________________________________________________
    /**
     * Gets a color with a possible default value.
     * The application prefix is automatically inserted in the key.
     * @param key the key.
     * @param def the default.
     * @return the value.
     * @see org.ariane.tools.Resources#get(String, Color)
     */
final public Color get(String key, Color def)
  {
      return Resources.instance().get(getAppplicationPrefix() + key, def);
  }
//______________________________________________________________________________
    /**
     * Gets the font for the headers.
     * @return the font.
     */
final public Font getHeaderFont()
  {
      return header_font;
  }
//______________________________________________________________________________
    /**
     * Gets the font for the labels.
     * @return the font.
     */
final public Font getLabelFont()
  {
      return label_font;
  }
//______________________________________________________________________________
    /**
     * Gets the font for the texts.
     * @return the font.
     */
final public Font getTextFont()
  {
      return text_font;
  }
//______________________________________________________________________________
    /**
     * Gets the background for the main windows.
     * @return the color.
     */
final public Color getMainBackground()
  {
      return main_bg;
  }
//______________________________________________________________________________
    /**
     * Gets the foreground for the main windows.
     * @return the color.
     */
final public Color getMainForeground()
  {
      return main_fg;
  }
//______________________________________________________________________________
    /**
     * Gets the background for the panes.
     * @return the color.
     */
final public Color getPaneBackground()
  {
      return pane_bg;
  }
//______________________________________________________________________________
    /**
     * Gets the foreground for the panes.
     * @return the color.
     */
final public Color getPaneForeground()
  {
      return pane_fg;
  }
//______________________________________________________________________________
    /**
     * Gets the background for the boards.
     * @return the color.
     */
final public Color getBoardBackground()
  {
      return board_bg;
  }
//______________________________________________________________________________
    /**
     * Gets the foreground for the boards.
     * @return the color.
     */
final public Color getBoardForeground()
  {
      return board_fg;
  }
//______________________________________________________________________________
    /**
     * Gets the color used for selections.
     * @return the color.
     */
final public Color getSelectionColor()
  {
      return select_bg;
  }
//______________________________________________________________________________
    /**
     * Gets the common border.
     * @return the border.
     */
final public Border getBorder()
  {
      return border;
  }
//______________________________________________________________________________
    /**
     * Gets the lowered border.
     * @return the border.
     */
final public Border getInBorder()
  {
      return in_border;
  }
//______________________________________________________________________________
    /**
     * Gets the raised border.
     * @return the border.
     */
final public Border getOutBorder()
  {
      return out_border;
  }
//______________________________________________________________________________
    /**
     * Gets the line border.
     * @return the border.
     */
final public Border getLineBorder()
  {
      return line_border;
  }
//______________________________________________________________________________
    /**
     * Gets the matte border.
     * @return the border.
     */
final public Border getMatteBorder()
  {
      return matte_border;
  }
//______________________________________________________________________________
    /**
     * Gets the margin border.
     * @return the border.
     */
final public Border getMarginBorder()
  {
      return margin_border;
  }
//______________________________________________________________________________
 /**
  * Gets the space size.
  * @return the size.
  */
final public int getSpace()
    {
	return space;
    }
//______________________________________________________________________________
  /**
   * Returns the icon defined by its filename.
   * <Br>
   * Searches automatically in the JVS lib directory.
   *
   * @parameter filename the name of the file.
   * @return the image icon.
   * @see ImageIcon
   */
final public Icon getIcon(String filename)
  {
      return new ImageIcon(Resources.instance().getLibDirectory()
			   + "images/" + filename);
  }
//______________________________________________________________________________
 /**
  * Prints to stdout the various sizes of a widget.
  * <P>
  * This utility gives the current, minimum, maximum and preferred sizes.
  *
  * @param label a string identifying the component.
  * @param widget the component to investigate.
  */
static public void printSizes(String label, JComponent widget)
    {
	Dimension d;

	System.out.println();
	System.out.println("Geometry of    : " + label);
	d = widget.getSize(null);
	System.out.println("Current size w : " + d.width + " h : " + d.height);
	d = widget.getMinimumSize();
	System.out.println("Minimum size w : " + d.width + " h : " + d.height);
	d = widget.getPreferredSize();
	System.out.println("Preferd size w : " + d.width + " h : " + d.height);
	d = widget.getMaximumSize();
	System.out.println("Maximum size w : " + d.width + " h : " + d.height);
    }
//______________________________________________________________________________
}
