//______________________________________________________________________________

//	The Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.gui;

import java.awt.Dimension;
import java.awt.Frame;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.border.Border;
import org.ariane.tools.ToolBox;

/**
 * A customized Dialog.
 * <Br>
 *
 * @version $Id: Dialog.java,v 3.1 2000/09/10 16:00:33 lefevre Exp $
 * @author Jean-Paul Le Fvre
 */
//______________________________________________________________________________

abstract public class Dialog extends JDialog implements ActionListener {
    /**
     * @serial The button OK label.
     */
protected static final String OK   = "OK";
    /**
     * @serial the button OK.
     */
protected JButton ok_button;
    /**
     * @serial The button Cancel label.
     */
protected static final String CANCEL   = "Cancel";
    /**
     * @serial the button Cancel.
     */
protected JButton cancel_button;
    /**
     * @serial the answer to the question.
     */
protected boolean confirmed = false;
    /**
     * @serial the main component shown by the Dialog.
     */
protected JComponent component;
    /**
     * @serial the set of properties.
     */
protected GuiResourcesOp resources;

//______________________________________________________________________________
/**
 * Creates a Dialog without title.
 * @param owner the frame from which the dialog is displayed
 */
protected Dialog(Frame owner)
  {
      this(owner, null);
  }
//______________________________________________________________________________
/**
 * Creates a Dialog with a title but without a icon.
 * @param owner the frame from which the dialog is displayed
 * @param title the header of the dialog.
 */
protected Dialog(Frame owner, String title)
  {
      this(owner, title, null);
  }
//______________________________________________________________________________
 /**
  * Creates a Dialog with a title.
  *
  * @param owner the frame from which the dialog is displayed
  * @param title the header of the dialog. Can be null.
  * @param icon_file the file name of the image. Can be null.
  * @see #build
  */
protected Dialog(Frame owner, String title, String icon_file)
  {
      super(owner, "Dialog", true);
      setResources();

      build(title, icon_file);
  }
//______________________________________________________________________________
 /**
  * Creates a Dialog with a title.
  *
  * @param owner the frame from which the dialog is displayed
  * @param title the header of the dialog. Can be null.
  * @param icon_file the file name of the image. Can be null.
  * @param modal true for a modal dialog, false otherwise.
  * @see #build
  */
protected Dialog(Frame owner, String title, String icon_file, boolean modal)
  {
      super(owner, "Dialog", modal);
      setResources();

      build(title, icon_file);
  }
//______________________________________________________________________________
 /**
  * Builds the window.
  * @param title the header of the dialog. Can be null.
  * @param icon_file the file name of the image. Can be null.
  * @see #createHeader
  */
private void build(String title, String icon_file)
  {
      int space        = resources.getSpace();
      Dimension vspace = new Dimension(0, space);
      setBackground(resources.getMainBackground());
      setForeground(resources.getMainForeground());

      JPanel box = new JPanel();
      setContentPane(box);
      box.setLayout(new BoxLayout(box, BoxLayout.Y_AXIS));

      box.setBackground(resources.getMainBackground());
      box.setForeground(resources.getMainForeground());
      box.setFont(resources.getHeaderFont());

      box.add(Box.createRigidArea(new Dimension(0, 3*resources.getSpace())));
      
      if(title != null) {		// Make a title if requested.
	  JComponent header = createHeader(title, icon_file);
	  box.add(header);
	  box.add(Box.createRigidArea(vspace));
      }

      component = createComponent();	// Done in the derived Classes.
      component.setAlignmentX(JComponent.CENTER_ALIGNMENT);

      box.add(component);

      JPanel bottom = new JPanel();
      bottom.setLayout(new BoxLayout(bottom, BoxLayout.Y_AXIS));
      bottom.setOpaque(false);

      bottom.add(Box.createRigidArea(vspace));

      JSeparator sep = new JSeparator(SwingConstants.HORIZONTAL);
      bottom.add(sep);
      bottom.add(Box.createRigidArea(vspace));

      JComponent buttons =  createButtons();
      buttons.setAlignmentX(JComponent.CENTER_ALIGNMENT);
      bottom.add(buttons);
      bottom.add(Box.createRigidArea(vspace));

      Dimension d = new Dimension(2000, 100);
      bottom.setMaximumSize(d);

      box.add(bottom);

      pack();
  }
//______________________________________________________________________________
/**
 * Builds the main component.
 * @return the component.
 */
abstract protected JComponent createComponent();

//______________________________________________________________________________
/**
 * Installs the resources used to build the Dialog.
 */
abstract protected void setResources();

//______________________________________________________________________________
/**
 * Sets the initial size.
 * <br>
 * Sizes are found in the resources.
 * @param window the name of the window.
 * @param width the default width.
 * @param height the default height.
 * @see #getInitialSize
 */
final protected void setInitialSize(String window, int width, int height)
{
    width  = resources.get(window + ".Width",  width);
    height = resources.get(window + ".Height", height);
    setBestSize(width, height);
}
//______________________________________________________________________________
/**
 * Gets the initial size.
 * @return the preferred dimension.
 * @see #setInitialSize
 */
final protected Dimension getInitialSize()
{
    return ((JComponent)getContentPane()).getPreferredSize();
}
//______________________________________________________________________________
/**
 * Sets the preferred size.
 * @param width the wished width.
 * @param height the wished height.
 */
final public void setBestSize(int width, int height)
{
    JComponent box = (JComponent)getContentPane();
    
    box.setPreferredSize(new Dimension(width, height));
}
//______________________________________________________________________________
/**
 * Creates the header.
 * <Br>
 * The icon file must be located in the JVS lib/images directory.
 * @return the component.
 * @param title the header of the dialog.
 * @param icon_file the file name of the image. Can be null.
 */
private JComponent createHeader(String title, String icon_file)
{
    JPanel row = new JPanel();
    row.setLayout(new BoxLayout(row, BoxLayout.X_AXIS));
    row.setOpaque(false);
    row.add(Box.createRigidArea(new Dimension(resources.getSpace(), 0)));
    JLabel label;

    if(icon_file != null) {

	Icon icon = resources.getIcon(icon_file);
	label     = new JLabel(icon);
	label.setAlignmentX(JComponent.LEFT_ALIGNMENT);
	label.setBorder(resources.getLineBorder());
	row.add(label);

	row.add(Box.createHorizontalGlue());

	label = new JLabel(title, JLabel.LEFT);
	label.setAlignmentX(JComponent.RIGHT_ALIGNMENT);
    }
    else {
	label = new JLabel(title, JLabel.CENTER);
	label.setAlignmentX(JComponent.CENTER_ALIGNMENT);
    }

    label.setFont(resources.getHeaderFont());
    label.setForeground(resources.getMainForeground());
    row.add(label);

    return row;
}
//______________________________________________________________________________
/**
 * Builds the buttons row.
 * @return the panel of buttons.
 */
protected JComponent createButtons()
{
    JPanel row       = new JPanel();
    Dimension hspace = new Dimension(2*resources.getSpace(), 0);

    row.setLayout(new BoxLayout(row, BoxLayout.X_AXIS));
    row.add(Box.createRigidArea(hspace));
    row.setOpaque(false);

    ok_button = new JButton(OK);
    ok_button.addActionListener(this);
    ok_button.setBorder(resources.getOutBorder());
    ok_button.setToolTipText("To accept the choice");
    ok_button.setAlignmentX(JComponent.LEFT_ALIGNMENT);
    ok_button.setOpaque(false);
    ok_button.setForeground(resources.getMainForeground());
    row.add(ok_button);

    row.add(Box.createHorizontalGlue());

    cancel_button = new JButton(CANCEL);
    cancel_button.addActionListener(this);
    cancel_button.setBorder(resources.getOutBorder());
    cancel_button.setToolTipText("To cancel the operation");
    cancel_button.setAlignmentX(JComponent.RIGHT_ALIGNMENT);
    cancel_button.setOpaque(false);
    cancel_button.setForeground(resources.getMainForeground());
    row.add(cancel_button);
    row.add(Box.createRigidArea(hspace));

    return row;
}
//______________________________________________________________________________
/**
 * Carries out the operation when a button item is pressed.
 * <Br>
 * Actually, simply updates the variable <Code>confirmed</Code>, and
 * makes the window unvisible.
 *
 * @param ev the event generated.
 */
public void actionPerformed(ActionEvent ev)
{
    String cmd = ev.getActionCommand();
    confirmed  = cmd.equals(OK);
    setVisible(false);
}
//______________________________________________________________________________
/**
 * Displays the confirmer and waits for a button pressed.
 * <Br>
 * @return true if the answer is OK, false otherwise.
 * @see #actionPerformed
 */
public boolean display()
{
    setVisible(true);

    return confirmed;
}
//______________________________________________________________________________
/**
 * Creates a border for a text area component.
 * <Br>
 * It is composed of a LoweredBevelBorder and an EmptyBorder.
 * @return the border.
 * @see BorderFactory
 */
final protected Border makeMarginBorder()
{
    int space  = resources.getSpace();
    Border out = resources.getInBorder();
    Border in  = BorderFactory.createEmptyBorder(2*space, 2*space, space,space);

    return BorderFactory.createCompoundBorder(out, in);
}
//______________________________________________________________________________
}

