//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.exec;

import java.awt.Canvas;
import java.awt.Dimension;
import java.awt.Graphics;
import org.ariane.tools.ToolBox;
import org.ariane.tools.Resources;

/**
 * RunnerBar : a object indicating that some task is running.
 * Is displays a small cursor going forth and back.
 *
 * Resources :
 * RunnerBar.Priority
 * RunnerBar.IdleTime
 * RunnerBar.StepsNumber
 * RunnerBar.Width
 * RunnerBar.Height
 *
 * @version $Id: RunnerBar.java,v 3.0 2000/08/23 13:15:33 lefevre Exp $
 * @author Jean-Paul Le F&egrave;vre
 */

public	class RunnerBar extends Canvas  implements Runnable {
  /**
   * @serial The name of the Class.
   */
private static final String  BAR = "RunnerBar";
  /**
   * @serial The number of steps performed
   */
private int    nb;
  /**
   * @serial the width of the cursor
   */
private int    dw;
  /**
   * @serial The current position
   */
private int    ps;
  /**
   * @serial The current direction
   */
private boolean forth;
  /**
   * @serial Is the bar busy ?
   */
private	boolean running = false;
  /**
   * @serial The associated thread.
   */
private	Thread thread = null;
  /**
   * @serial The priority of the thread.
   */
private	int priority = Thread.MIN_PRIORITY;
  /**
   * @serial The idle time.
   */
private	long laps = 1000;

//______________________________________________________________________________
/**
 * The object RunnerBar.
 * @parameter w the width of the bar
 * @parameter h the height of the bar
 */
public	RunnerBar(int w, int h)
  {
      build(w, h, 10);
  }
//______________________________________________________________________________
/**
 * The object RunnerBar.
 * Attributes are set from the resources.
 */
public	RunnerBar()
  {
    Resources resources = Resources.instance();

    priority = resources.get(BAR + ".Priority",    Thread.MIN_PRIORITY);
    laps     = resources.get(BAR + ".IdleTime",   2000);
    int n    = resources.get(BAR + ".StepsNumber",  10);
    int w    = resources.get(BAR + ".Width",       100);
    int h    = resources.get(BAR + ".Height",        3);

    build(w, h, n);
  }
//______________________________________________________________________________
/**
 * Builds the geometry.
 * @parameter w the width of the bar
 * @parameter h the height of the bar
 * @parameter n the number of steps
 */
private	void build(int w, int h, int n)
  {
      nb    = n > 1 ? n : 2;
      ps    = 0;
      forth = true;

      setSize(w, h);
  }
//______________________________________________________________________________
/** 
 * Reshapes the Component to the specified bounding box.
 * Recompute width to an integer number of dw.
 * @param x the x coordinate
 * @param y the y coordinate
 * @param width the width of the component
 * @param height the height of the component
 * @see java.awt.Component#setBounds
 */
public synchronized void setBounds(int x, int y, int width, int height)
{
    dw  = width / nb;
    if(dw < 2) dw = 2;

    super.setBounds(x, y, dw * nb, height);
}
//______________________________________________________________________________
/**
 * Paints the Bar.
 * @param g the specified Graphics window
 */
public void paint(Graphics g)
{
    Dimension dim = getSize();
    g.setColor(getForeground());
    g.fillRect(ps, 0, dw, dim.height);
}
//______________________________________________________________________________
/**
 * Creates then launches the thread.
 */
public synchronized void start()
  {
      start(null);
  }
//______________________________________________________________________________
/**
 * Creates then launches the thread.
 * @parameter group the group to which the task belongs.
 * @see Thread#start
 */
public synchronized void start(ThreadGroup group)
  {
    if(thread != null && thread.isAlive()) {
	if(ToolBox.verbose) ToolBox.warn(BAR + " already alive");
	return;
    }

    ps	    = 0;
    running = true;
    forth   = true;
    thread  = group == null ? new Thread(this) : new Thread(group, this);
    thread.setPriority(priority);

    if(ToolBox.debug) ToolBox.warn(BAR + " started");
    thread.start();
  }
//______________________________________________________________________________
/**
 * Stops the Runner if it is running.
 * @see Thread#stop
 */
public synchronized void stop()
  {
      if(thread == null || ! thread.isAlive()) {
	if(ToolBox.verbose) ToolBox.warn(BAR + " already dead");
	return;
      }

      ps      = -dw;
      forth   = true;
      running = false;
      thread  = null;

      if(ToolBox.debug) ToolBox.warn(BAR + " stopped");
      repaint();
  }
//______________________________________________________________________________
/**
 * Makes the Runner running.
 */
public void run()
  {
      Dimension dim = getSize();
      int max = dim.width - dw;
      int min = dw;

      try {

	while(running) {

	    if(forth) {
		ps += dw;
		if(ps >= max) {
		    forth = false;
		}
	    }
	    else {
		ps -= dw;
		if(ps < min) {
		    forth = true;
		}
	    }

	    repaint();
	    Thread.sleep(laps);
	}
      }
      catch(InterruptedException ex) {}
  }
//______________________________________________________________________________
/**
 * Converts Task to String.
 * @return str	the string.
 */
public String toString()
  {
    return thread != null ? thread.toString() : "dead";
  }
//______________________________________________________________________________
}



