//______________________________________________________________________________

//	Java Virtual Shelf.
//______________________________________________________________________________

package org.ariane.applet;

import java.awt.*;
import java.net.URL;
import org.ariane.tools.ToolBox;

/**
 * Picture : a component meant to display images.
 * @see Image
 * @version $Id: Picture.java,v 3.0 2000/08/23 13:15:32 lefevre Exp $
 * @author Jean-Paul Le F&egrave;vre
 */

public	class Picture extends Canvas {

  /**
   * @serial The actual image displayed
   */
private	Image image;
  /**
   * @serial The image's id
   */
private	int id;
  /**
   * @serial The object tracking the process
   */
private	MediaTracker tracker;
  /**
   * @serial The border width
   */
private	int border = 1;

//______________________________________________________________________________
/**
 * The object.
 * If the input string is actually an URL creates the image from the net.
 * @param orig	the file or the url storing the Image.
 */
public	Picture(String orig)
  {
      boolean isurl = orig.indexOf("://") >= 0;

      if(isurl) {
	try {
	   loadImage(new URL(orig));
	}
	catch(Exception ex) {
	    tracker = new MediaTracker(this);
	    image   = null;
	}
      }
      else {
	  loadImage(orig);
      }
  }
//______________________________________________________________________________
/**
 * The object.
 * @param url	the URL storing the Image.
 */
public Picture(URL url)
  {
      loadImage(url);
  }
//______________________________________________________________________________
/**
 * The object.
 */
public	Picture()
  {
      image   = null;
      tracker = null;
  }
//______________________________________________________________________________
/**
 * Redraws the window.
 */
final public void refresh()
  {
      Graphics gc = getGraphics();
      if(gc != null) update(gc);
  }
//______________________________________________________________________________
/**
 * Waits for the end of the loading process.
 * @see #isLoaded
 */
final public void doLoad()
    {
      try {
	tracker.waitForID(id);
      }
      catch(InterruptedException ex) {
      }

      setSize(getPreferredSize());
    }
//______________________________________________________________________________
/**
 * Gets the size of the image once it is loaded.
 * @return the size of the image.
 */
final public Dimension imageSize()
  {
      int w = image.getWidth(this);
      int h = image.getHeight(this);

      return new Dimension(w, h);
  }
//______________________________________________________________________________
/**
 * Gets the preferred size of this component.
 * It takes care of the border.
 * @return the size of the image.
 */
public	Dimension getPreferredSize()
  {
      int w = image.getWidth(this)  + 2*border;
      int h = image.getHeight(this) + 2*border;

      return new Dimension(w, h);
  }
//______________________________________________________________________________
/**
 * Checks wether the image is available.
 * @return true when the load is complete.
 */
final public boolean isLoaded()
  {
    return ((tracker.statusID(id, true) & MediaTracker.COMPLETE) != 0);
  }
//______________________________________________________________________________
/**
 * Checks whether the image has failed.
 * @see MediaTracker#isErrorID
 * @return true when the load is complete.
 */
final public boolean hasFailed()
  {
      return tracker.isErrorID(id);
  }
//______________________________________________________________________________
/**
 * Sets the border width
 * @param w the specified width
 */
final public void setBorderWidth(int w)
{
    border = w;
}
//______________________________________________________________________________
/**
 * Draws all the stuffs : the image plus a border
 * @param g the specified Graphics window
 */
public void paint(Graphics gc)
{
    Dimension d = getSize();

    if(border > 0) {
	gc.setColor(getForeground());
	gc.drawRect(0, 0, d.width - 1, d.height - 1);
    }

    if(image == null) return;

    int status = tracker.statusID(id, true);

    if((status & MediaTracker.ERRORED) != 0) {
	ToolBox.warn("Image in error");
	return;
    }
    else if((status & MediaTracker.ABORTED)  != 0) {
	ToolBox.warn("Image aborted");
	return;
    }
    else if((status & MediaTracker.LOADING)  != 0) {
	gc.drawString("Loading ...", d.width/2 - 10, d.height/2);
	return;
    }
    else if((status & MediaTracker.COMPLETE)  != 0) {
	if(ToolBox.debug) ToolBox.warn("Image complete");
    }

    if(ToolBox.debug) System.out.println(imageSize().toString());

    gc.drawImage(image, border, border, this);
}
//______________________________________________________________________________
/**
 * Prepares the image.
 * @param src the source of the image : either a filename or an url.
 */
public void loadImage(Object src)
{
    if(tracker != null)
	return;
    getImage(src);
    id      = 0;
    tracker = new MediaTracker(this);
    tracker.addImage(image, id);
}
//______________________________________________________________________________
/**
 * Gets the image.
 * @parameter the source of the image.
 */
private void getImage(Object src)
{
    if(image != null)
	return;
    else if (src instanceof String) {
	image = getToolkit().getImage((String)src);
    }
    else if (src instanceof URL) {
      ToolBox.warn("Get Image from URL " + ((URL)src).toString());
	image = getToolkit().getImage((URL)src);
    }
}
//______________________________________________________________________________
}



