//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.ariane.applet;

import java.awt.*;

/**
 * A Gadget is a drawing and is not a Component.
 * It is meant to simulate a transparent background.
 * @version $Id: Gadget.java,v 3.0 2000/08/23 13:15:32 lefevre Exp $
 * @author Jean-Paul Le Fevre
 */

public	abstract class Gadget extends Object {

    /**
     * The x position of the gadget in the parent's coordinate system.
     * @see #location
     */
    int x;

    /**
     * The y position of the gadget in the parent's coordinate system.
     * @see #location
     */
    int y;

    /**
     * The x position of the drawing in the gadget's coordinate system.
     * @see #reference
     */
    protected int s;

    /**
     * The y position of the drawing in the gadget's coordinate system.
     * @see #reference
     */
    protected int t;

    /**
     * The width of the gadget.
     * @see #size
     */
    int width;

    /**
     * The height of the gadget.
     * @see #size
     */
    int height;

    /**
     * The width of the border.
     * @see #drawBorder
     */
    int border_width = 0;

    /**
     * The font used to draw strings
     */
    protected Font	font;

    /**
     * The Color used to draw.
     */
    protected Color	foreground = Color.black;

    /**
     * The parent of the gadget.
     * @see #getParent
     */
    protected  Component parent;

//______________________________________________________________________________
/**
 * Initialize the object.
 * @param parent the component holding the Gadget.
 */
public	Gadget(Component parent)
  {
      this.parent = parent;
  }
//______________________________________________________________________________
  /**
   * Display the Gadget.
   * @param g the graphic context.
   */
public	abstract void draw(Graphics g);
//______________________________________________________________________________
/** 
 * Returns the current location of this component.
 * The location will be in the parent's coordinate space.
 * @see #move
 */
public Point location()
{
    return new Point(x, y);
}
//______________________________________________________________________________
/** 
 * Returns the current size of this component.
 * @see #resize
 */
public Dimension size()
{
	return new Dimension(width, height);
}
//______________________________________________________________________________
 /** 
  * Returns the current bounds of this component.
  */
public Rectangle bounds()
{
    return new Rectangle(x, y, width, height);
}
//______________________________________________________________________________
/** 
 * Change the reference position in the Gadget.
 * @param st the new position.
 * @see #move
 */
public void setReference(int s, int t)
{
    this.s = s;
    this.t = t;
}
//______________________________________________________________________________
/** 
 * Change the position in the parent Component.
 * @param xy the new position
 * @see #location
 */
public void move(int x, int y)
{
    this.x = x;
    this.y = y;
}
//______________________________________________________________________________
/** 
 * Changes the position
 * @param p the new position
 * @see #location
 */
public void move(Point p)
{
    this.x = p.x;
    this.y = p.y;
}
//______________________________________________________________________________
/** 
 * Changes the size
 * @param wh the new dimensions
 * @see #size
 */
public void resize(int width, int height)
{
    this.width  = width;
    this.height = height;
}
//______________________________________________________________________________
/**
 * Gets the parent of the component.
 */
public Component getParent()
{
    return parent;
}
//______________________________________________________________________________
/**
 * Gets the current font.
 */
public Font getFont()
{
    return font;
}
//______________________________________________________________________________
/**
 * Returns the total height of the font in pixel.
 * @see FontMetrics#getHeight
 */
public int getFontHeight()
{
    FontMetrics fmt = getFontMetrics(font); 

    return fmt == null ? 10 : fmt.getHeight();
}
//______________________________________________________________________________
/**
 * Returns the total height of the font in pixel.
 * @see FontMetrics#getAscent
 */
public int getFontAscent()
{
    FontMetrics fmt = getFontMetrics(font); 

    return fmt == null ? 5 : fmt.getAscent();
}
//______________________________________________________________________________
/**
 * Returns the width of the specified String in the current font.
 * @see FontMetrics#stringWidth
 */
public int getStringWidth(String str)
{
    FontMetrics fmt = getFontMetrics(font); 

    return fmt == null ? 10 : fmt.stringWidth(str);
}
//______________________________________________________________________________
/**
 * Sets up geometry.
 */
public void realize()
{
}
//______________________________________________________________________________
/**
 * Gets the FontMetrics for the specified font.
 * @param f the font
 * @see Component#getFontMetrics 
 */
public FontMetrics  getFontMetrics(Font f)
{
  try {
	return parent.getFontMetrics(f);
  }
  catch(Exception ex) {
	return null;
  }
}
//______________________________________________________________________________
/**
 * Gets background of the parent.
 * @see Component#getBackground
 */
public Color getBackground()
{
    return parent.getBackground();
}
//______________________________________________________________________________
/**
 * Gets foreground of the gadget.
 */
public Color getForeground()
{
    return foreground;
}
//______________________________________________________________________________
/**
 * Sets foreground of the gadget.
 * @param fg the new color.
 */
public void setForeground(Color fg)
{
   foreground = fg;
}
//______________________________________________________________________________
/**
 * Sets font of the gadget.
 * @param f the new font.
 */
public void setFont(Font f)
{
   font = f;
}
//______________________________________________________________________________
/**
 * Draw a Border around the Gadget.
 * @param g the specified Graphics window
 * @see #setBorderWidth
 * @see Graphics#drawRect
 */
protected void drawBorder(Graphics g)
{
    if(border_width > 0)
	g.drawRect(x, y, width, height);
}
//______________________________________________________________________________
/**
 * Change the current width of the border
 * @param w the requested width.
 * @see #drawBorder
 */
public void setBorderWidth(int w)
{
    border_width = w;
}
//______________________________________________________________________________
/**
 * Draw a Line
 * @param xy the coordinates of the segment in Gadget Coordinates.
 * @param g the specified Graphics window
 * @see Graphics#drawLine
 */
protected void drawLine(Graphics g, int x0, int y0, int x1, int y1)
{
    g.drawLine(x + x0, y + y0, x + x1, y + y1);
}
//______________________________________________________________________________
/**
 * Draw a String
 * @param g the specified Graphics window.
 * @param str the string.
 * @param xy the coordinates of the segment in Gadget Coordinates.
 * @see Graphics#drawString
 */
protected void drawString(Graphics g, String str, int x0, int y0)
{
    g.drawString(str, x + x0, y + y0);
}
//______________________________________________________________________________
/**
 * Return the coordinates of the reference point.
 * @return the reference
 */
public Point reference()
{
    return new Point(s, t);
}
//______________________________________________________________________________
/** 
 * Change the position of the reference point.
 * @param p the new position
 * @see Gadget#move
 */
public void pinAt(Point p)
{
    pinAt(p.x, p.y);
}
//______________________________________________________________________________
/** 
 * Change the position of the reference point.
 * @param p the new position
 * @see Gadget#move
 */
public void pinAt(int x, int y)
{
    move(x - s, y - t);
}
//______________________________________________________________________________
/**
 * Get info
 */
public	String toString()
{
    return "Gadget width : " + width + " height : " + height
	+ " x : " + x + " y : " + y + " s : " + s + " t : " + t;
}
//______________________________________________________________________________
}
