//
// class wlString
//
// FUNCTION:
// Implements a string class that provides somewhat more efficient 
// and powerful string handling abilities than the usual Java and C++
// string classes. In particular, it is efficient at string editing,
// such as search-n-replace of substrings, and concatenation to form
// arbitrary-legth strings. It represents strings internally as a 
// linked list of edit cursors, therby avoiding un-needed string copies
// that might normally occur during a string-edit.
// 
// This class also includes binary data support missing from most
// string classes.  It thereby becomes a handy utility for parsing
// and editing HTTP messages which typically have ascii headers but 
// may have binary bodies.
//
// Since this class provides a cast operator char *() method to cast
// the contents to a contiguous block of memory, it can be used
// directly with standard C lib string and memory handling routines.
// 
// The Memlen() method returns the length of the memory block.
//
// The Substitute() method replaces all occurances of the substring
//    "from" with the string "to".  If string "to" is null, it is 
//    interpreted as if it were a zero-length string.
//
// The SubMulti() method behaves like the Substitute() method, except 
//    that it accepts arrays of substitution pairs.
// 
// The casting operator merely returns a pointer to the internal
//    string.  Any subsequent calls to wlString methods will probably
//    invalidate this pointer.
//
// The PutIn() method will use the storage pointed at directly, 
//    thus avoiding a copy-in.  The passed parameter must have been 
//    alloced with operator new, and must not be deleted. It must be 
//    null-terminated.

#ifndef __WL_SUPER_H__
#define __WL_SUPER_H__

#include <stdlib.h>

class wlString {
   public:
      wlString (void);
      wlString (const char *);
      wlString (wlString &);
      ~wlString ();

      void   PutIn (char *, size_t len);

      void   Strcat (const char * src);
      void   Memcat (const char * src, size_t len);
      void   Memcat (wlString &);

      void   Strcpy (const char * src);
      void   Memcpy (const char * src, size_t len);
      void   Memcpy (wlString &);

      size_t Memlen (void) const;

      int   Substitute (const char *from, const char *to);
      int   SubMulti (char **from, char **to);

      operator char *();
      wlString& operator= (const char *);
      wlString& operator= (const wlString &);

      wlString& operator+= (const char *);
      wlString& operator+= (const wlString &);

   protected:
      class Chunk {
         public:
            Chunk * next;      // ptr to next chunk
            char * root;       // ptr to memory block
            char * start;      // ptr to string within block
            size_t len;        // length of string
      };

      Chunk *head;
      Chunk *tail;


      void gitgone (void);
      void defrag (void);
      int subs (const char *from, const char * to);
      Chunk * subsone (Chunk *, char *, size_t, char *, size_t);
};

#endif /* __WL_SUPER_H__ */
