/*
 * FILE: sslopts.c
 *
 * FUNCTION:
 * Parse SSL related command-line arguments, prints SSL usage
 * info, and reports actual configuration used.
 *
 * DESIGN NOTES:
 * Should be updated with a unified command-line/config-file utility.
 *
 * HISTORY:
 * created by Linas Vepstas April 1998
 */

#include <stdlib.h>
#include <string.h>

#include "errexit.h"
#include "generic.h"
#include "shhopt.h"
#include "socket.h"

void 
print_ssl_usage (void)
{
#ifdef ITS_A_PERFECT_WORLD
    printf(
      "SSL options:\n"
      "\n"
      "      --ssl-server           use SSL to connect to server\n"
      "  -S, --server-cipher=NUM    force use of a specific cipher\n"
      "      --server-timeout=NUM   set SSL session cache timeout (seconds)\n"
      "      --ssl-browser          use SSL to connect to browser\n"
      "      --browser-cipher=NUM   force use of a specific cipher\n"
      "      --browser-timeout=NUM  set SSL session cache timeout (seconds)\n"
      "      --certif=<filename>    specify file containing certificates\n"
      "  -k, --keyring=<filename>   specify file containing keys\n"
      "  -K, --password=<passwd>    specify keyring password\n"
      "  -b, --stashfile=<filename> specify keyring password stashfile\n"
      "  -N, --distinguish=<name>   specify a distinguished name\n"
      "\n"
      "Cipher suites may be specified in the form SSLV2:N with N 1,2,3,..7\n"
      "or SSLV3:NN for NN=00, 01, 02 ... 06 or 0A."
      "\n"
      "\n"
    );
#endif /* ITS_A_PERFECT_WORLD */

    printf(
      "SSL options:\n"
      "\n"
#ifdef USE_SSLEAY
      "      --ssl-server           use SSL to connect to server\n"
#endif /* USE_SSLEAY */
      "  -S, --server-cipher=NUM    force use of a specific cipher\n"
      "      --server-timeout=NUM   set SSL session cache timeout (seconds)\n"
#ifdef USE_SSLEAY
      "      --ssl-browser          use SSL to connect to browser\n"
#endif /* USE_SSLEAY */
      "      --browser-cipher=NUM   force use of a specific cipher\n"
      "      --browser-timeout=NUM  set SSL session cache timeout (seconds)\n"
#ifdef USE_SSLEAY
      "      --certif=<filename>    specify file containing certificates\n"
#endif /* USE_SSLEAY */
      "  -k, --keyring=<filename>   specify file containing keys\n"
      "  -K, --password=<passwd>    specify keyring password\n"
#ifdef USE_SKIT
      "  -b, --stashfile=<filename> specify keyring password stashfile\n"
#endif /* USE_SKIT */
      "  -N, --distinguish=<name>   specify a distinguished name\n"
      "\n"
      "Cipher suites may be specified in the form SSLV2:N with N 1,2,3,..7\n"
      "or SSLV3:NN for NN=00, 01, 02 ... 06 or 0A."
#ifdef USE_SKIT
      "No cipher suite negotiation \n"
      "is performed. Cipher SSLV3:04 (RC4-128 with MD5) is recommended"
#endif /* USE_SKIT */
#ifdef USE_SSLEAY
      "If a cipher suite has been \n"
      "specified, then no cipher suite negotiation is performed. \n"
#endif /* USE_SSLEAY */
      "\n"
    );

}


/* **************************************************************** */
/* parse command line options ..................................... */
/* **************************************************************** */

void
parse_cipher_arg (char **type, char **cipher, char * optarg)
{
    char * start;
    int int_cipher;

    if ((!type) || (!cipher) || (!optarg)) return;

    /* is the user requesting SSLV2 or SSLV3? .................. */
    if ((strstr(optarg,"SSLV2:") != NULL) || 
        (strstr(optarg,"sslv2:") != NULL)) 
    {

        /* user asked for SSLV2 */
        *type = strdup ("SSLV2");
        start = strchr(optarg,':') + 1;
        *cipher = strdup (start);
        /* if the user specified a cipher suite using 0N, 
               *  convert it to what SSLV2 wants. ................... */
        if (strlen(*cipher) > 1) {
            (*cipher)[0] = (*cipher)[1];
            (*cipher)[1] = 0x0;
        }

        int_cipher = strtol (*cipher, 0x0, 16);

        if ((1  > int_cipher) || 
            (5 == int_cipher) || 
            (7  < int_cipher)) {
            PERR ("bad SSLV2 cipher suite: %s\n", *cipher);
            free (*cipher);
            *cipher = strdup ("-1");
        }
        PDBG2("SSL v2 cipher suite=%s\n", *cipher);
    } else 
    if ((strstr(optarg,"SSLV3:") != NULL) || 
        (strstr(optarg,"sslv3:") != NULL)) {

        /* user asked for SSLV3 */
        *type = strdup ("SSLV3");
        start = strchr(optarg,':') + 1;

        *cipher = strdup (start);
        /* check for legality as best we can here */
        if (strlen(*cipher) == 1) {
            (*cipher)[1] = (*cipher)[0];
            (*cipher)[0] = '0';
            (*cipher)[2] = 0x0;
        }
        /* make it upper case, if it is not 
         * (only legal letter is A). ....... */
        if ((*cipher)[1] == 'a') (*cipher)[1] = 'A';

        int_cipher = strtol (*cipher, 0x0, 16);

        if ((0 == int_cipher) ||
            ((6 < int_cipher) && (10 != int_cipher)))
        {
            printf ("Error: parse_cipher_arg(): "
                "bad SSLV3 cipher suite: %s\n", *cipher);
            free (*cipher);
            *cipher = strdup ("-1");
        }
        PDBG("SSL V3 cipher suite=%s\n",  *cipher);
    } else {
        PERR("parameter to -S option must be SSLV2:n or SSLV3:nn\n");
        return;
    }
}

/* **************************************************************** */
/* **************************************************************** */

wlSSLInfo *cli = NULL;
wlSSLInfo *srv = NULL;

void 
parse_server_spec (char * optarg)
{
    if (!srv) return;
    parse_cipher_arg (&(srv->ssl_version), 
                      &(srv->cipher), optarg);
    srv->icipher = strtol (srv->cipher, 0x0, 16);
    srv->use_ssl = 1;
}

void 
parse_client_spec (char * optarg)
{
    if (!cli) return;
    parse_cipher_arg (&(cli->ssl_version), 
                      &(cli->cipher), optarg);
    cli->icipher = strtol (cli->cipher, 0x0, 16);
    cli->use_ssl = 1;
}


/* **************************************************************** */
/* parse command line options ..................................... */
/* **************************************************************** */

#define SOPT(a,b,c,d,e) {            \
    opt[iopt].shortName = (a);       \
    opt[iopt].longName = (b);        \
    opt[iopt].type = (c);            \
    opt[iopt].arg = (void *) (d);    \
    opt[iopt].flags = (e);           \
    opt[iopt].flagDefault = 0x3f3f3f3f;   \
    iopt ++;                         \
};


void
parse_ssl_args (wlSSLInfo &client, wlSSLInfo &server, int *argc, char *argv[])
{

    optStruct opt[30];
    int iopt=0;
    int cli_use_ssl=0;  /* must be an int for OPT_FLAG */
    int srv_use_ssl=0;  /* must be an int for OPT_FLAG */

    /* short long               type        var/func           special    */
    /* reverse server & client: we act as server for client */
    /* reverse client & server: we act as client for server */
    SOPT( 0 , "ssl-server",     OPT_FLAG,   &(cli_use_ssl),         0);
    SOPT( 0 , "ssl-browser",    OPT_FLAG,   &(srv_use_ssl),         0);
    SOPT('S', "server-cipher",  OPT_STRING, parse_client_spec,  OPT_CALLFUNC);
    SOPT( 0 , "browser-cipher", OPT_STRING, parse_server_spec,  OPT_CALLFUNC);
    SOPT( 0 , "server-timeout", OPT_INT,    &(client.sess_timeout), 0);
    SOPT( 0 , "browser-timeout",OPT_INT,    &(server.sess_timeout), 0);

    SOPT('b', "stashfile",      OPT_STRING, &(client.keyring_stashfile),  0);
    SOPT('k', "keyring",        OPT_STRING, &(client.keyring_file),       0);
    SOPT('K', "password",       OPT_STRING, &(client.keyring_file_passwd),0);
    SOPT('N', "distinguish",    OPT_STRING, &(client.distinguished_name), 0);
    SOPT( 0 , "certif",         OPT_STRING, &(client.certificate_file),   0);
#ifdef IN_THE_FUTURE
    SOPT( 0 , "ldap-server",    OPT_STRING, &dummy,               0 );
    SOPT( 0 , "ldap-user",      OPT_STRING, &dummy,               0 );
    SOPT( 0 , "ldap-password",  OPT_STRING, &dummy,               0 );
    SOPT( 0 , "ldap-port",      OPT_INT,    &intdummy,            0 );
#endif /* IN_THE_FUTURE */

    SOPT(0, 0, OPT_END, 0, 0 );  /* no more options */
 
    /* parse all options */
    srv = &server;
    cli = &client;
    optParseOptions(argc, argv, opt, 0);
    srv = cli = NULL;

    /* be careful mixing pointers to shorts & longs on big endian machines */
    if (srv_use_ssl) server.use_ssl = srv_use_ssl;
    if (cli_use_ssl) client.use_ssl = cli_use_ssl;

    /* some info is duplicated */
    server.keyring_stashfile   = client.keyring_stashfile;
    server.keyring_file        = client.keyring_file;
    server.keyring_file_passwd = client.keyring_file_passwd;
    server.distinguished_name  = client.distinguished_name;
    server.certificate_file    = client.certificate_file;
}

/* **************************************************************** */
/* Make sure that all of the needed arguments have been specified   */
/* **************************************************************** */

void
validate_ssl_opts (wlSSLInfo &self)
{
#ifdef USE_SKIT
    /* The skit toolkit requires that a keyring be specified.
     * its a fatal error if one is not specified.
     * It also requires that a cipher suite be specified; its 
     * its a fatal error if one is not.
     */
    if (self.use_ssl) 
    {
        if (!(self.keyring_file)) 
        {
            PFATAL ("no keyring specified.\n"
                "\tThe use of the SKIT SSL toolkit requires that a \n"
                "\tkeyring containing CA Certifying Authority \n"
                "\tcertifiicates be specified. \n"
                "\tUse the -k flag to specify a keyring. \n");
            exit (2);
        }
        if ((!(self.keyring_stashfile)) && (!(self.keyring_file_passwd)))
        {
            PFATAL ("no keyring password specified.\n"
                "\tThe use of the SKIT SSL toolkit requires that a \n"
                "\tpassword or a password stash file be specified \n"
                "\tin order to open the certificate keyring. \n"
                "\tUse the -K flag to specify a keyring password, \n"
                "\tor use the -b flag to specify a password stash. \n");
            exit (3);
        }

        if ((!(self.ssl_version)) || (!(self.cipher)))
        {
            PFATAL ("no cipher suite specified.\n"
                "\tThe use of the SKIT SSL toolkit requires that a \n"
                "\tcipher suite be specified \n"
                "\tUse the -S flag to specify a server cipher. \n");
            exit (4);
            
        }
    }

#endif /* USE_SKIT */
}

/* ********************************************************************** */
/* print options summary on file outfile. ............................... */
/* ********************************************************************** */

#define REPORT(ciph,int_ciph,vers,suite,who) {				\
    if (ciph){								\
        if (0 > int_ciph) {						\
            fprintf(outfile,"Error: Illegal or unimplemented %s cipher"	\
                " suite requested:%s\n", vers, ciph);			\
            return;							\
        }								\
        fprintf(outfile,"Using %s connection to %s with "		\
            "cipher suite %s (%s)\n",vers, who,ciph, suite[int_ciph]);	\
    } else {								\
        fprintf (outfile, "Will negotiate for cipher suite with %s\n",who);\
    }									\
}


void 
print_ssl_options_summary (FILE *outfile, wlSSLInfo &client, wlSSLInfo &server) 
{
    if ((0 == client.use_ssl) && 
        (0 == server.use_ssl)) 
    {
        fprintf(outfile,"Encryption is not being used.\n");
        return;
    }

    if (0 == client.use_ssl)
    {
        fprintf(outfile,"Not using encrypted webserver connection.\n");
    } else {
        fprintf(outfile,"Using encrypted connection to webserver.\n");

        /* print info about server connection */
        if (strcmp(client.ssl_version,"SSLV2") == 0) {
            REPORT (client.cipher, client.icipher,
                "SSLV2", client.v2_cipher_suite, "webserver");
        } else {
            REPORT (client.cipher, client.icipher,
                "SSLV3", client.v3_cipher_suite, "webserver");
        }
    }

    if (0 == server.use_ssl)
    {
        /* don't bother printing this, as it only applies to webmon anyway */
        /* fprintf(outfile,"Not using encrypted browser connection.\n"); */
    } else {

        fprintf(outfile,"Using encrypted connection to browser.\n");
        /* print info about client connection */
        if (strcmp(server.ssl_version,"SSLV2") == 0) {
            REPORT (server.cipher, server.icipher,
                "SSLV2", server.v2_cipher_suite, "browser");
        } else {
            REPORT (server.cipher, server.icipher,
                "SSLV3", server.v3_cipher_suite, "browser");
        }
    }


    if (0x0 == client.keyring_file) 
        fprintf(outfile,"    No keyring file specified.\n");
    else
        fprintf(outfile,"    keyring file is:%s\n",client.keyring_file);

    if (0x0 == client.keyring_file_passwd) 
        fprintf(outfile,"    No keyring file password specified.\n");
    else
        fprintf(outfile,"    keyring file password is:%s\n",
            client.keyring_file_passwd);

    if (0x0 == client.keyring_stashfile) 
        fprintf(outfile,"    No keyring password stash file specified.\n");
    else
        fprintf(outfile,"    keyring password stash file is:%s\n",
            client.keyring_stashfile);

    if (0x0 == client.distinguished_name) 
        fprintf(outfile,"    No distinguished name specified.\n");
    else
        fprintf(outfile,"    distinguished name is:%s\n",
            client.distinguished_name);
}

