
/**************************************************************************
 *                                                                        *
 *               Copyright (C) 1995 Silicon Graphics, Inc.                *
 *                                                                        *
 *  These coded instructions, statements, and computer programs were      *
 *  developed by SGI for public use.  If any changes are made to this code*
 *  please try to get the changes back to the author.  Feel free to make  *
 *  modifications and changes to the code and release it.                 *
 *                                                                        *
 **************************************************************************/

#ifndef __WEBLOAD_SOCKET_H_
#define __WEBLOAD_SOCKET_H_

#include <stdio.h>

#ifdef WIN32
#include <winsock.h>
#include <windows.h>
#include <process.h>
#include <io.h>
#endif /* WIN32 */

#ifndef WIN32
#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/in.h>
#include <pthread.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <unistd.h>
#endif /* WIN32 */


#include "stats.h"
#include "sysdep.h"
#include "timefunc.h"


/* ================================================================== */

// class wlHost is used as a utility to resolve the host addresses.
// The ResolveAddr() method will resolve the indicated hostname.
//   If this method returns zero, then the various public members
//   will contain valid data for the indicated hostname.  If this
//   method returns a negative value, then the members do NOT 
//   contain valid data.  The returned negative value will be
//   minus an errno code.
// 
class wlHost 
{
   public:
      wlHost (void);
      ~wlHost ();
      int ResolveAddr (const char *hostname, const char * protocol);

      char             *hostname;
      char             *ipnum;
      struct hostent    s_hostent;
      struct protoent   s_protent;
      struct in_addr    s_addr;
      short             sock_type;    /* socket type ....... */
   private:
      void killname (void);
      size_t            hlen;
};

// class wlHostCache provides a cache to bypass DNS/YP resolution
//    for every request.  It maintains its cache in a thread-safe 
//    fashion and is suitable for use as a global.  The pointers
//    returned by ResolveAddr() will remain valid as long as this 
//    object continues to exist.  That is, the cache only grows,
//    never shrinks, and cache slots are never reused.
//    
// N.B. I would think that its the responsibility of the OS DNS/YP
// subssystem to maintain a valid cache of recently resolved hostnames.
// But some of aour favorite Windozian OS's fail to do so, so we do it
// ourselves.

class wlHostCache 
{
   public:
      wlHostCache (void);
      ~wlHostCache ();

      wlHost * ResolveAddr (const char *hostname, const char * protocol);

   private:
      unsigned long cache_size;
      unsigned long cache_entries;
      wlHost **cache;
      pthread_mutex_t pool_lock;

};


/* ================================================================== */
/* ================================================================== */
/* ================================================================== */
/* ================================================================== */

/* maximum number of times to retry an socket I/O operation */
#define MAX_IO_RETRIES   150

/* ================================================================== */
/* ================================================================== */
/* ================================================================== */
/* ================================================================== */

#ifdef USE_SKIT
#include "skit.h"
#endif /* USE_SKIT */

#ifdef USE_SSLEAY
#define PERL5
#include <openssl/rsa.h>
#include <openssl/crypto.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#undef PERL5
#endif /* USE_SSLEAY */

// class wlSSLInfo ...
// contains information that can be set to determine how SSL will be 
// used/configured.  Also contains process-wide state information, 
// such as global context and thread locks.
//
// The Init() method takes a flag, indicating if initialization 
//    should be performed so that ssl acts as a server.  This routine
//    initializes the global context state and locks.
//
// The FlushContext() method will delete all SSL sessions.  This call 
//    should be used only after all sockets have been closed.  It is
//    useful to reset the SSL state back to the begining, so that sessions
//    from prior connections do not get reused.
//
// Note about the size of flags: e.g. use_ssl: these should be ints
// and not changed to short or char even though short or char can
// hold the desired value.  This is because the command line option
// parsing code expects (pointers to) ints, and on a big-endian 
// machine, the LSB's don't match: *((short *)ptr) != *((long *) ptr)

class wlSSLInfo {
   public:
      wlSSLInfo (void);			// constructor
      ~wlSSLInfo ();			// destructor
      void Init (int act_as_server);
      void FlushContext (void);
	
   public:

      // Public configuration and initialization data
      int     use_ssl;
      int     be_server;

      
      char    *ssl_version;    /* type SSLV2 or SSLv3 */
      char    *cipher;         /* restriction on cipher to use  */
      int     icipher;
      int     sess_timeout;    /* session cache timeout */
      
      
      char    *keyring_file;
      char    *keyring_stashfile;
      char    *keyring_file_passwd;
      char    *certificate_file;
      char    *distinguished_name;

      // Global context and lock structures
#ifdef USE_SKIT
      // skitInitData is used globally
      static skit_init_data *skitInitData;
#endif /* USE_SKIT */

#ifdef  USE_SSLEAY
      static short   is_inited;
      SSL_CTX       *context;  
      // SSL_SESSION   *session;

   private:
      // lock state
      static pthread_mutex_t lock_cs[CRYPTO_NUM_LOCKS];
      static unsigned long thread_id (void);
      static void locking_callback (int mode, int type, const char *file, int line);

#endif /* USE_SSLEAY */


      // strings
   public:
#define MAX_V2_CIPHER_SPEC 7
      static char *v2_cipher_suite[MAX_V2_CIPHER_SPEC+2];

#define MAX_V3_CIPHER_SPEC 30
      static char *v3_cipher_suite[MAX_V3_CIPHER_SPEC+2];
};


// some subsidiary routines for parsing command-line options,
// and printing them out.  Implemented in sslopts.C

extern void parse_ssl_args (wlSSLInfo &cli, wlSSLInfo &serv, int *, char **);
extern void print_ssl_options_summary (FILE *, wlSSLInfo &cli, wlSSLInfo &srv);
extern void print_ssl_usage (void);
extern void validate_ssl_opts (wlSSLInfo &);


/* ================================================================== */
/* ================================================================== */
/* ================================================================== */
/* ================================================================== */
//
// The Connect() method establishes a connection to the host 'host'
//    on port 'port' using protocol 'protocol'.  If the 'reuse' flag is
//    set to 1, and this socket is *already* connected to this server/port,
//    then nothing is done and ths routine returns immediately with EISCONN.
//    Otherise, if the 'reuse' flag is not set, and the socket is already
//    connected, that connection is torn down and rebuilt.
//    Returns 0 if successful, otherwise -errno.
//
// The StartSSL() method is used to initialize the socket so that it 
//    acts as an SSL server.  This method can only be called on a 
//    connected socket; otherwise its a no-op.
//
// The SSLConnect() method performs an SSL handshake on the socket,
//    setting up the socket to be an SSL client.  The socket must 
//    already be connected before this method is called, otherwise its 
//    a no-op.
//
// The ResetNetTimes() method resests the timing statistics collected 
//    for this socket back to zero.
//
// The ResetSessionCounts() method resets the connect, read and write 
//    counts back to zero. 
// 
// The member struct sockaddr sin contains the sin of the last connect
// The member sock_fd contains the file descriptor for the last connect
//
// Timing and Instrumentation Info:
// This class has a number of members used to cather timing and count
// information.  These members are public and can be read at any time.
// The timing data can be reset with the ResetNetTimes() method.  
// The count data can be reset with the ResetSessionCounts() method.  
//
// int num_connects: counts the number of times that a tcpip connect() 
//    was performed.  
//
// int num_kept_alive: counts the number of times that the Connect() 
//    method was called, but wasn't needed because the socket was 
//    in keep-alive mode.  This value, plus num_connects equals the 
//    total number of times that Connect() was called.
//
// int num_reads: counts the number of times the Read() method was called.
// int num_writes: counts the number of times the Write() method was called.
//
// ConnectTime: records the elapsed time from just before the unix 
//    socket() calls (to create a socket), to just after the return
//    from the connect() call (by which time the tcpip connection
//    to the remote server has been firmly established).
//
// SSLConnectTime: records the elapsed time waiting for socket reads
//    and writes do complete while performing an SSL connection handshake.
//    This time includes tiem spent in low-level socket read/write, 
//    but does NOT include time elapsed on the client side performing
//    the potentially CPU-intensive encryption calculations.
//
// SSLConnectOvhd: records the elapsed time spent in client-side 
//    encryption algorithms.  This equals the total elapsed time spent
//    on the SSL connect handshake, minus the SSLConnectTime above.
//
// ReadTime: records the elapsed time between issueing a low-level
//    socket read() call, and the time that read() returns with data.
//    Note that the system read() call blocks until there is data
//    to be read; thus the ReadTime can be quite large when the remote 
//    end is slow in responding.
//
// WriteTime: records the elapsed time between the issueing of the
//    low level write() call, and its return.  Because the write()
//    system call usually does not block, the WriteTime will usually
//    be quite small.  Note that WriteTime is *not* a measure of how 
//    long it took data to get to the remote end.  It is rather a 
//    measure of how long it takes to stuff it into the low-level OS
//    layers.
//  
// Note that ReadTime and WriteTime do *NOT* include network read and 
// write times accrued during SSL connection negotiation.  These times
// are reported seperately in SSLConnectTime.

class wlSocket {
   public:
      wlSocket (void);      // constructor
      ~wlSocket ();

      int  Connect(char *host, NETPORT portnum, char *protocol, int reuse);

      int  StartSSL (wlSSLInfo &);
      void SSLConnect (wlSSLInfo &);

      int  Read (void *buffer, size_t readlen);
      int  Write (void *buffer, size_t writelen);
      void ResetNetTimes (void);
      void ResetSessionCounts (void);

      int  Close (void);

   public:
      struct sockaddr_in sin;
      SOCKET        sock_fd;
      int           use_ssl;
      int           keep_afloat;  /* keep_alive */
   
      int           num_connects;
      int           num_kept_alive;
      int           num_reads;
      int           num_writes;

      time_struct   ConnectTime;
      time_struct   SSLConnectTime;
      time_struct   SSLConnectOvhd;
      time_struct   ReadTime;
      time_struct   WriteTime;

#if !defined(USE_SKIT) && !defined(USE_SSLEAY)
      void * sslHandle;
#endif /* !defined(USE_SKIT) && !defined(USE_SSLEAY) */

#ifdef USE_SKIT
      skit_soc_init_data skitSocInitData;
      sec_soc_data  *sslHandle;
#endif /* USE_SKIT */

#ifdef USE_SSLEAY
      SSL           *sslHandle;

   private:
      static long raw_tcp_timing_callback (BIO*, int, const char*, int, long, long);
      time_struct   rd_start;
      time_struct   rd_stop;    
      time_struct   wr_start;
      time_struct   wr_stop;

#endif /* USE_SSLEAY */


};
   

#endif /* __WEBLOAD_SOCKET_H_ */

