
/**************************************************************************
 *                                                                        *
 *               Copyright (C) 1995 Silicon Graphics, Inc.                *
 *                                                                        *
 *  These coded instructions, statements, and computer programs were      *
 *  developed by SGI for public use.  If any changes are made to this code*
 *  please try to get the changes back to the author.  Feel free to make  *
 *  modifications and changes to the code and release it.                 *
 *                                                                        *
 **************************************************************************/

/* socks redefinition must preceed system headers */
#include "sockify.h" 

#include <ctype.h>
#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>

#ifdef WIN32
#include <winsock.h>
#include <windows.h>
#include <process.h>
#include <io.h>
#endif /* WIN32 */

#ifndef WIN32
#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/in.h>
#include <sys/errno.h>
#include <sys/file.h>
#include <sys/ipc.h>
#include <sys/param.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <unistd.h>
#endif /* WIN32 */


#include "errexit.h"
#include "generic.h"
#include "pool.h"
#include "socket.h"
#include "sysdep.h"
#include "timefunc.h"

#ifdef USE_SKIT
#include "skit.h"
#include "skit_error.h"
#endif /* USE_SKIT */

#ifdef USE_SSLEAY
#include <openssl/rsa.h>       /* SSLeay stuff */
#include <openssl/crypto.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#endif /* USE_SSLEAY */


/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* forward declarations */

#ifdef USE_SKIT

#ifdef WIN32
#define LINKAGE  __cdecl
#else
#define LINKAGE  
#endif /* WIN32 */

int LINKAGE skREAD(int fd, void * data_buff, int num_bytes_to_read);
int LINKAGE skWRITE(int fd, void * data_buff, int num_bytes_to_write);

#endif /* USE_SKIT */

#ifdef USE_SSLEAY

long
raw_tcp_timing_callback (BIO *bio, int mode, char *buf, int buflen, 
            long junk, long retval);

#endif /* USE_SSLEAY */

/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* ================================================================= */


wlHost :: wlHost (void)
{
   hostname = NULL;
   hlen = 0;
   ipnum = new char[32];
}

wlHost :: ~wlHost ()
{
   if (hostname) delete [] hostname;
   hostname = 0;
   delete [] ipnum;
}

void
wlHost :: killname (void)
{
   if (hostname) delete [] hostname;
   hostname = 0;
   hlen = 0;
}

/*
 * look up the host name and protocol
 * called once by main() since all threads
 * use the same protocol and address
 */

int 
wlHost :: ResolveAddr (const char *whosename, const char *protocol)
{
   struct hostent *phe;
   struct protoent *ppe;
   size_t whlen;

   if (!whosename) return -ENOENT;

   whlen = strlen (whosename);
   if (whlen > hlen) 
   {
      if (hostname) delete [] hostname;
      hostname = new char [whlen+1];
      hlen = whlen;
   }
   strcpy (hostname, whosename);

   /* if numeric IP address given, convert to internal form */
   if (isdigit (whosename[0]))
   {
      s_addr.S_ADDR = inet_addr(whosename);
      if (INADDR_NONE == s_addr.S_ADDR) 
      {
         PERR("Invalid IP address %s\n", whosename);
         killname ();
         return(-EADDRNOTAVAIL);
      }
   } else {
      /* look up by name */
      phe = gethostbyname(whosename);                        
      if (!phe) 
      {
         int norr = errno;
         PERR("Can't get %s host entry\n", whosename);
         perr ("\terrno=%d %s\n", norr, strerror (norr));
         killname ();
            return(-errno);
      }
      memcpy(&s_hostent, phe, sizeof(struct hostent));
      memcpy(&(s_addr.S_ADDR), phe->h_addr, sizeof(unsigned long));
   }
   strcpy (ipnum, inet_ntoa (s_addr));

   /* Map protocol name to protocol number */
   ppe = getprotobyname(protocol);

   if (!ppe) {
      PERR("Can't get %s protocol entry\n",protocol);
      killname ();
      return(-ENOPROTOOPT);
   }
   memcpy(&s_protent, ppe, sizeof(struct protoent));

   PDBG2("Protocol number %d\n", ppe->p_proto );

   /* Use protocol to choose a socket type */
   if (0 == strcmp(protocol,"udp")) {
      sock_type = SOCK_DGRAM;
      PDBG2("Choosing SOCK_DGRAM (type %d) \n", sock_type);
   } else {
      sock_type = SOCK_STREAM;
      PDBG2("Choosing SOCK_STREAM (type %d) \n", sock_type);
   }

   return 0;
}

/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* ================================================================= */

#define CACHE_GROW_SIZE 16

wlHostCache :: wlHostCache (void)
{
   cache_entries = 0;
   cache_size = CACHE_GROW_SIZE;
   cache = (wlHost **) malloc (CACHE_GROW_SIZE * sizeof (wlHost *));

   pthread_mutexattr_t matter;
   pthread_mutexattr_init (&matter);
   pthread_mutex_init (&pool_lock, &matter);
   pthread_mutexattr_destroy (&matter);
}

wlHostCache :: ~wlHostCache ()
{
   int i;
   for (i=0; i<cache_entries; i++)
   {
      delete cache[i];
      cache[i] = NULL;
   }
   cache_entries = 0;
   free (cache);

   pthread_mutex_destroy (&pool_lock);
}

wlHost *
wlHostCache :: ResolveAddr (const char *whosename, const char *protocol)
{
   int i;
   if (!whosename) return NULL;

   // lets see if we've already got this one
   for (i=0; i<cache_entries; i++)
   {
      if (0 == strcmp(whosename, cache[i]->hostname))
      {
         return cache[i];
      }
   }

   pthread_mutex_lock (&pool_lock);

   // we didn't find the host in our cache.
   // do we need to increase the size of the cache?
   if (cache_entries >= cache_size) {
      cache_size += CACHE_GROW_SIZE;
      cache = (wlHost **) realloc (cache, cache_size * sizeof (char *));
   }

   // look up the hostname ...
   wlHost *newhost = new wlHost;
   int rc = newhost->ResolveAddr (whosename, protocol);
   if (rc)
   {
      // oops !
      delete newhost;
      return 0x0;
   }

   cache[cache_entries] = newhost;
   cache_entries ++;

   pthread_mutex_unlock (&pool_lock);

   return newhost;
}


/* ================================================================= */

/* *********************************************************** */
/* is the given socket in a connected state? ................. */
/* *********************************************************** */
int
is_connected(SOCKET sock) 
{
   struct sockaddr Name;
   size_t NameLength;
   unsigned int    rc;
   int    norr;

   NameLength = sizeof(Name);
   rc = getpeername(sock,&Name,&NameLength);
   if (0 == rc) return 1;
#ifdef WIN32
   norr = WSAGetLastError();
#else
    norr = errno;
#endif /* WIN32 */
   PDBG("getpeername returned %d for socket %d; errno=%d (%s)\n",
      rc,sock, norr, strerror (norr));
#ifndef WIN32
   if ((ENOTCONN == norr) || (EBADF == norr)) return 0;
#else /* WIN32 */
   if ((WSAENOTCONN == norr) || (WSAENOTSOCK == norr)) return 0;
#endif /* WIN32 */
   PERR("errno=%d (%s)\n",errno, strerror (errno));
   return (-1);
}

/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* ================================================================= */

#ifdef USE_SKIT
   
char *wlSSLInfo::v2_cipher_suite[MAX_V2_CIPHER_SPEC+2]=
   {
   "Not Supported",
   "SSL2_RC4_128_WITH_MD5",
   "SSL2_RC4_128_EXPORT_40_WITH_MD5",
   "SSL2_RC2_128_CBC_WITH_MD5",
   "SSL2_RC2_128_CBC_EXPORT40_WITH_MD5",
   "Not Supported",
   "SSL2_DES_64_CBC_WITH_MD5",
   "SSL2_DES_192_EDE3_CBC_WITH_MD5",
   0
   };
   
char *wlSSLInfo::v3_cipher_suite[MAX_V3_CIPHER_SPEC+2]= 
   {
   "SSL_NULL_WITH_NULL_NULL", 
   "SSL_RSA_WITH_NULL_MD5",
   "SSL_RSA_WITH_NULL_SHA", 
   "SSL_RSA_EXPORT_WITH_RC4_40_MD5",
   "SSL_RSA_WITH_RC4_128_MD5",
   "SSL_RSA_WITH_RC4_128_SHA",
   "SSL_RSA_EXPORT_WITH_RC2_CBC_40_MD5",
   "Not Supported",
   "Not Supported",
   "Not Supported",
   "SSL_RSA_WITH_3DES_EDE_CBC_SHA",
   0
   };
#endif /* USE_SKIT */

#ifdef USE_SSLEAY

char *wlSSLInfo::v2_cipher_suite[MAX_V2_CIPHER_SPEC+2]=
   {
   "NULL",
   "RC4-MD5",            /* SSL2_RC4_128_WITH_MD5 */
   "EXP-RC4-MD5",         /* SSL2_RC4_128_EXPORT_40_WITH_MD5 */
   "RC2-CBC-MD5",         /* SSL2_RC2_128_CBC_WITH_MD5 */
   "EXP-RC2-CBC-MD5",      /* SSL2_RC2_128_CBC_EXPORT40_WITH_MD5 */
   "IDEA-CBC-MD5",         /* SSL2_IDEA_128_CBC_WITH_MD5 */
   "DES-CBC-MD5",         /* SSL2_DES_64_CBC_WITH_MD5 */
   "DES-CBC3-MD5",         /* SSL2_DES_192_EDE3_CBC_WITH_MD5 */
   0
   };
   
char *wlSSLInfo::v3_cipher_suite[MAX_V3_CIPHER_SPEC+2]= 
   {
   "NULL",                     /* SSL_NULL_WITH_NULL_NULL */
   "NULL-MD5",                  /* SSL_RSA_WITH_NULL_MD5 */
   "NULL-SHA",                  /* SSL_RSA_WITH_NULL_SHA */
   "EXP-RC4-MD5",               /* SSL_RSA_EXPORT_WITH_RC4_40_MD5 */
   "RC4-MD5",                  /* SSL_RSA_WITH_RC4_128_MD5 */
   "RC4-SHA",                  /* SSL_RSA_WITH_RC4_128_SHA */
   "EXP-RC2-CBC-MD5",            /* SSL_RSA_EXPORT_WITH_RC2_CBC_40_MD5 */
   "IDEA-CBC-SHA",               /* SSL_RSA_WITH_IDEA_CBC_SHA */
   "EXP-DES-SHA",               /* SSL_RSA_EXPORT_WITH_DES40_SHA */
   "DES-DBD-SHA",               /* SSL_RSA_WITH_DES_CBC_SHA */
   "DES-CBC3-SHA",               /* SSL_RSA_WITH_3DES_EDE_CBC_SHA */
   0
   /* hack alert -- SSLEAY supports a whole bunch more, 
    * I am too lazy to type them in */
   };

#else 

char *wlSSLInfo::v2_cipher_suite[MAX_V2_CIPHER_SPEC+2]=
   {
   "NULL",
   "NULL",
   "NULL",
   "NULL",
   "NULL",
   "NULL",
   "NULL",
   "NULL",
   0
   };
   
char *wlSSLInfo::v3_cipher_suite[MAX_V3_CIPHER_SPEC+2]= 
   {
   "NULL",
   "NULL",
   "NULL",
   "NULL",
   "NULL",
   "NULL",
   "NULL",
   "NULL",
   "NULL",
   "NULL",
   "NULL",
   0
   };

#endif /* USE_SSLEAY */


/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* ================================================================= */

#ifdef USE_SKIT
skit_init_data * wlSSLInfo :: skitInitData = NULL;
#endif /* USE_SKIT */

#ifdef USE_SSLEAY
short wlSSLInfo :: is_inited = 0;
#endif /* USE_SSLEAY */

   
/* **************************************************************** */
/* init skit ...................................................... */
/* **************************************************************** */
wlSSLInfo :: wlSSLInfo (void)
{

   use_ssl = 0;
   be_server = 0;
   ssl_version = strdup ("SSLV3");
   cipher  = 0x0;
   icipher = -1;
   sess_timeout = 1000;      /*  default V3 timeout */

   keyring_file = NULL;
   keyring_file_passwd = NULL;
   keyring_stashfile = NULL;      
   certificate_file = NULL;
   distinguished_name = NULL;

#ifdef USE_SSLEAY
   context = NULL;
   // session = NULL;
#endif /* USE_SSLEAY */

}

wlSSLInfo :: ~wlSSLInfo ()
{
#ifdef USE_SSLEAY
   for (int i=0; i<CRYPTO_NUM_LOCKS; i++)
   {
      pthread_mutex_destroy(&(lock_cs[i]));
   }
#endif /* USE_SSLEAY */
}

void
wlSSLInfo :: Init (int act_as_server)
{
   be_server = act_as_server;
   if (!use_ssl) return;

#ifdef USE_SKIT

   // a part of skit needs only be initialized once, and
   // so we set that up as a static (global) for this class.
   if (!skitInitData) 
   {
      skitInitData = 
         (skit_init_data *) malloc(sizeof(skit_init_data));

      /* **************************************************** */
      /* following code shamelessly stolen from skitsamp.c .. */
      /* **************************************************** */

      skitInitData->keyring = keyring_file;      
      skitInitData->keyring_pw = keyring_file_passwd;  
      skitInitData->keyring_stash = keyring_stashfile;      
      skitInitData->sec_types =  client_ssl_version;  
      skitInitData->V2_session_timeout = client_sess_timeout;
      skitInitData->V3_session_timeout = client_sess_timeout;
   
      skitInitData->LDAP_server = NULL;
      skitInitData->LDAP_user = NULL;
      skitInitData->LDAP_password = NULL;
      skitInitData->LDAP_port = 0;
      skitInitData->LDAP_CA_roots = 0;
      skitInitData->auth_type = CLIENT_AUTH_LOCAL;
   
      PDBG("calling skit_initialize\n");
      int rc = skit_initialize(skitInitData);
      PDBG("skit_initialize returned %d\n", rc);
      if (rc != 0) {
         switch (rc) {
           case SKIT_ERR_INIT_PARM_NOT_VALID:
             PERR("Error occurred initializing cipher specs\n");
             break;
           case SKIT_INIT_HARD_RT:
             PWARN("Neither the Keyring file nor password were supplied. "
                   "Hardcoded roots used. \n");
             rc = 0;
             break;
           case SKIT_INIT_SEC_TYPE_NOT_VALID:
             PERR ("Invalid handshake type.\n");
             break;
           case SKIT_INIT_V2_TIMEOUT_NOT_VALID:
             PERR("Invalid V2 timeout value.\n");
             break;
           case SKIT_INIT_V3_TIMEOUT_NOT_VALID:
             PERR("Invalid V3 timeout value.\n");
             break;
           case SKIT_KEYFILE_IO_ERROR:
             PERR("I/O error for keyring file.  "
                  "Default hardcoded roots used.\n");
             rc = 0;
             break;
           case SKIT_KEYFILE_OPEN_FAILED:
             PERR("Unable to open keyring file.  "
                  "Default hardcoded roots used.\n");
             rc = 0;
             break;
           case SKIT_KEYFILE_BAD_FORMAT:
             PERR("Keyring file has invalid format.  "
                  "Default hardcoded roots used.\n");
             rc = 0;
             break;
           case SKIT_KEYFILE_BAD_PASSWORD:
             PERR("Password for keyfile is invalid.  "
                  "Default hardcoded roots used.\n");
             rc = 0;
             break;
           case SKIT_KEYFILE_BAD_MALLOC:
             PERR("Malloc was not successful for "
                  "keyring file processing.\n");
             break;
           case SKIT_KEYFILE_BAD_DNAME:
             PERR("Invalid distinguished name specified for keyfile.\n");
             break;
           case SKIT_KEYFILE_CERT_EXPIRED:
             PERR("The certificate within the keyfile has expired.\n");
             break;
           default:
             PERR("(skit toolkit): "
               "An error occurred during SSL initialization. rc=%d\n", rc);
             break;
         }
      }
   }


#endif /* USE_SKIT */

#ifdef USE_SSLEAY
   PDBG("initializing SSLeay\n");

   // parts of the ssleay initialization need be done only once.
   if (!is_inited) 
   {
      is_inited = 1;

      ERR_load_SSL_strings();
      SSL_load_error_strings();

      SSLeay_add_ssl_algorithms();

      // Initilize the thread lock structures that SSLeay will need.
      for (int i=0; i<CRYPTO_NUM_LOCKS; i++)
      {
         pthread_mutex_init(&(lock_cs[i]),NULL);
      }
      CRYPTO_set_id_callback(thread_id); 
      CRYPTO_set_locking_callback(locking_callback);
   }

   if (be_server) {
      context = SSL_CTX_new (SSLv23_server_method()); 
   } else {
      context = SSL_CTX_new (SSLv23_client_method()); 
   }
   if (!context) {
      PERR("can't initialize SSL\n");
      ERR_print_errors_fp (stderr);
   }

   /* set the default session caching behaviour to resemble that
    * which we would expect from a typical client or server. */
   SSL_CTX_set_session_cache_mode (context, 
      SSL_SESS_CACHE_BOTH | SSL_SESS_CACHE_NO_AUTO_CLEAR);

#ifdef NOT_NEEDED
   /* for example, if we wanted to verify the server ... */
   SSL_CTX_set_default_verify_paths (context);
#endif /* NOT_NEEDED */

   if (!certificate_file) {
      certificate_file = keyring_file;
   }

   if (certificate_file) {
      int rc = SSL_CTX_use_certificate_file (context, 
                                         certificate_file,
                                         SSL_FILETYPE_PEM);
      if (0 >= rc) {
         PERR("Can't load certificate file %s\n", 
            certificate_file);
      }
   }

   if (!keyring_file) {
      // the private key is probably jammed into the same file as the certif. 
      keyring_file = certificate_file;
   }

   if (keyring_file) {
      int rc = SSL_CTX_use_RSAPrivateKey_file (context, 
                                        keyring_file,
                                        SSL_FILETYPE_PEM);
      if (0 >= rc) {
         PERR("Can't load keyring file %s\n", keyring_file);
      }
   }

   /* OK, if the user is demanding that we use a certain cipher,
    * lets use it. */

   if (cipher) {
      char *ciph;
      if (0 == strcmp ("SSLV2", ssl_version)) {
         ciph = v2_cipher_suite[icipher];
      } else {
         ciph = v3_cipher_suite[icipher];
      }
      SSL_CTX_set_cipher_list (context, ciph);
   }
  
   /* enable compatibility flags for Microsoft & Netscape server 
    * and browser bugs.  This will cause SSLeay to be have in 
    * the most compatible possible way, rather than according 
    * to what the SSL spec says.
    */
   SSL_CTX_set_options (context,SSL_OP_ALL);
 
#endif /* USE_SSLEAY */

}

/* ============================================================= */
/* locking structures needed by SSLeay */

#ifdef USE_SSLEAY
pthread_mutex_t wlSSLInfo::lock_cs[CRYPTO_NUM_LOCKS];


unsigned long 
wlSSLInfo::thread_id(void)
{
   return (unsigned long) pthread_self();
}

void
wlSSLInfo::locking_callback (int mode, int type, const char *file, int line)
{
   if (mode & CRYPTO_LOCK)
   {
      PDBG ("locking type=%d\n");
      pthread_mutex_lock(&(lock_cs[type]));
   }
   else
   {
      PDBG ("unlocking type=%d\n");
      pthread_mutex_unlock(&(lock_cs[type]));
   }
}

#endif /* USE_SSLEAY */
/* ============================================================= */

void 
wlSSLInfo :: FlushContext (void)
{
#ifdef USE_SKIT
    /* hack alert!!!
     * The following subroutine call is *supposed* to
     * clear the ssl session cache, and force the SSL code 
     * to renegotiate the session keys.  However, it does
     * not actually doe this, as is horribly clear from
     * the socket traffic traces. Bummer.  As of skit-act3
     * this continues to remain to be a bug.
     */
   skit_secure_soc_reset(clientHandle);
#endif /* USE_SKIT */

#ifdef USE_SSLEAY
   /* Print some statistics about the session */
   int hits = SSL_CTX_sess_hits (context);
   int misses = SSL_CTX_sess_misses (context);
   int tkos = SSL_CTX_sess_timeouts (context);
   PDBG(" previous session had %d hits, %d misses and %d timeouts \n",
      hits, misses, tkos);

   /* Its enough to null this out to force a renegotiation */
   // session = 0x0;

   /* Flush the session cache, forcing subsequent SSL 
    * connections to renegotiate & exchange private keys. */
   SSL_CTX_flush_sessions (context, 0);
#endif /* USE_SSLEAY */

}

/* =============================================================== */
/* =============================================================== */
/* =============================================================== */

wlHostCache host_cache;

/* =============================================================== */
/* =============================================================== */
/* =============================================================== */

wlSocket :: wlSocket (void) 
{
   sock_fd = -1;
   use_ssl = 0;

   sslHandle = NULL;
   ResetNetTimes ();
   ResetSessionCounts ();

#ifdef USE_SKIT
   skitSocInitData.fd = -1;
#endif /* USE_SKIT */
}

wlSocket :: ~wlSocket () 
{
   Close ();
}

/* =============================================================== */
/* =============================================================== */
/* =============================================================== */

int
wlSocket :: Connect (char *host, NETPORT portnum, 
                     char *protocol, int recycle)
{
   PDBG1("will connect to host %s on port %d\n", 
      host, portnum);

   /* Don't bother opening TCP/IP connection if the socket is still 
    * connected to the correct host and port.  Still have to do 
    * skitSecureSocInit since last message skit got was a skit
    * bad msg.  If we don't do this, then skit will simply abort 
    * the program (silently!).........  ???? Huhh ???
    */

   wlHost *cached_host = host_cache.ResolveAddr (host, protocol);
   if (!cached_host) 
   {
      PERR("Can't resolve hostname %s\n", host);
      return -EINVAL;
   }
   
   if (recycle &&
       !(BADSOCKET(sock_fd)) &&
       is_connected(sock_fd) &&
       (htons(portnum) == sin.sin_port) &&
       (sin.sin_addr.s_addr == cached_host->s_addr.s_addr))
   {
      PDBG1("reusing keepalive socket fd=%d for host=%s\n", 
         sock_fd, host);
      num_kept_alive ++;
      return EISCONN;
   }
   num_connects ++;

   // make sure we suffer no damage from keep-alive
   Close ();

   /* set up the internet endpoint address */
   memset((char *)&sin, 0, sizeof(sin));

   sin.sin_port = htons(portnum);
   PDBG2("Set port number %d\n", portnum );

   sin.sin_addr = cached_host->s_addr;
   sin.sin_family = PF_INET;
   short proto = cached_host->s_protent.p_proto;
   int type = cached_host->sock_type;

   /* don't start the connect clock running until after 
    * the name resolution has happened.
    */
   time_struct start;
   TIMESTAMP(&start);

   /* Allocate a socket */
   SOCKET s = socket(PF_INET, type, proto);
   PDBG2("Socket %d returned sockfd=%d\n", type, s);
   
   if (BADSOCKET(s))
   {
      int norr = errno;
      PERR("Can't create socket: \n"
         "\t%d %s\n",
         norr, strerror(norr));
      return -norr;
   }
   
   /* Connect the socket */
   PDBG2("Trying to connect sockfd=%d \n", s);
   PDBG2("Address is family %d, port %d, addr %s\n", 
      sin.sin_family, ntohs(sin.sin_port), inet_ntoa(sin.sin_addr) );
   
   /* if interrupted during negotiation, retry.
    * by interrupting the client, we can stress-test the server... */
   int ntries = 0;
   int rc = 0;
    do {
      retry_io = 0;
      close_io = 0;

      /* lets be very specific-- we sockify this and *only* this routine. */
#ifdef USE_SOCKS
      rc = Rconnect(s, (struct sockaddr *)&sin, sizeof(sin));
#else
      rc = connect(s, (struct sockaddr *)&sin, sizeof(sin));
#endif 
      /* if we got a EADDRINUSE, then a prvious connect may have partly
       * gone through... just to be sure, close and try again. 
       */
      if (0>rc && 0 < ntries)
      {
         if (EADDRINUSE == errno)
         {
            NETCLOSE(s);
            s = socket(PF_INET, type, proto);
            retry_io = 1;
         } else
         if (EISCONN == errno)
         {
            rc = 0;
         }
      }
      ntries ++;
   } while ((0>rc) && retry_io && MAX_IO_RETRIES>ntries);


   time_struct stop; 
   TIMESTAMP(&stop);
   if (0 > rc)
   {
      int norr = errno;
      if (!close_io) {
         PERR("Can't connect: %d %s\n", 
            norr, strerror (norr));
      }
      NETCLOSE(s);
      return -norr;
   }

   CONVERTTIME (&start);
   CONVERTTIME (&stop);
   wl_difftime(&stop,&start,&stop);
   addtime(&ConnectTime,&stop);
   DT_PP("Info: wlSocket::Connect(): ConnectTime this connect :",stop);

   PDBG2("Connect returned %d\n", rc);

   sock_fd = s;
   
   /* all done, returning socket descriptor */
   PDBG2("Returning sockfd=%d\n", s );
   return 0;

} /* END wlSocket::Connect() */

/* ---------------------------------------------------------------- */

void
wlSocket::ResetSessionCounts (void)
{
   num_connects = 0;
   num_kept_alive = 0;
   num_reads = 0;
   num_writes = 0;
}

void
wlSocket::ResetNetTimes (void)
{
   memset(&ConnectTime,    0, sizeof(time_struct));
   memset(&SSLConnectTime, 0, sizeof(time_struct));
   memset(&SSLConnectOvhd, 0, sizeof(time_struct));
   memset(&ReadTime,       0, sizeof(time_struct));
   memset(&WriteTime,      0, sizeof(time_struct));
}

/* ---------------------------------------------------------------- */
/* SSLConnect performs an SSL handshake on the socket */

void
wlSocket::SSLConnect (wlSSLInfo &sinfo)
{
   // If not authorized to use ssl, don't do anything.
   if (0 == sinfo.use_ssl) return;

   // We can connect only if we are set up to be a client.
   if (0 != sinfo.be_server) return;

   // If we've already got a handle, then assume that we're a keep-alive
   // socket, i.e. that we are already initialized and connected.
   if (sslHandle) return;

   PDBG1("Begin SSL connection negotiation on sock=%d\n", sock_fd);

   // Measure network times separately for the connects ...
   time_struct SaveReadTime = ReadTime;
   time_struct SaveWriteTime = WriteTime;
   memset(&ReadTime, 0, sizeof(time_struct));
   memset(&WriteTime, 0, sizeof(time_struct));

   time_struct start;
   TIMESTAMP(&start);
   use_ssl = 1;   

#ifdef USE_SKIT
   skitSocInitData.fd= sock_fd;  /* file descriptor to use */
   skitSocInitData.hs_type = AS_CLIENT;  /* connect as client */
   if (strcmp(sinfo.ssl_version,"SSLV2") == 0) {
      skitSocInitData.cipher_specs = sinfo.cipher; 
      skitSocInitData.v3cipher_specs = 0x0;
   } else {
      skitSocInitData.cipher_specs = 0x0;
      skitSocInitData.v3cipher_specs = sinfo.cipher; 
   }
   skitSocInitData.skread = skREAD;     /* read callback         */
   skitSocInitData.skwrite = skWRITE;   /* write callback        */
   skitSocInitData.sec_type = NULL;     /* this will be returned */
   skitSocInitData.DName = sinfo.distinguished_name; 
   skitSocInitData.skit_data = sinfo.skitInitData;

   sslHandle = skit_secure_soc_init(&skitSocInitData);
#endif /* USE_SKIT */


#ifdef USE_SSLEAY
   // Create a new connection handle.
   sslHandle = SSL_new (sinfo.context); 
   /* SSL_clear (sslHandle); not needed, done by SSL_new */
   SSL_set_fd (sslHandle, sock_fd);

#if 0
   // Copy in the existing session if there is one.
   if (sinfo.session) {
      PDBG2 ("reuse client session %p\n", 
         sinfo.session);
      SSL_set_session (sslHandle, sinfo.session);
   }
#endif

   // Get the basic-io structure so that we can install 
   // callbacks for timing.
   BIO *bio = SSL_get_rbio (sslHandle);
   BIO_set_callback (bio, raw_tcp_timing_callback);
   BIO_set_callback_arg (bio, this);
   // If interrupted during negotiation, retry.
   // By interrupting the client, we can stress-test the server... 
   int ntries = 0;
   int       rc;   /* temporary return value */
   do {
      retry_io = 0;
      close_io = 0;
      caught_sigpipe = 0;
      rc = SSL_connect (sslHandle);
      ntries ++;
   } while ((0>rc) && (retry_io || caught_sigpipe) && MAX_IO_RETRIES>ntries);

#if 0
   // If first time, then save the session handle. 
   // The session is created only *after* the connect.
   if (!(sinfo.session)) {
      sinfo.session = SSL_get_session (sslHandle);
      PDBG2("new client session %p\n", sinfo.session);
   }
#endif

   memset(&rd_start, 0, sizeof(time_struct));
   memset(&rd_stop,  0, sizeof(time_struct));
   memset(&wr_start, 0, sizeof(time_struct));
   memset(&wr_stop,  0, sizeof(time_struct));
#endif /* USE_SSLEAY */

   time_struct stop;
   TIMESTAMP(&stop);
   CONVERTTIME (&start);
   CONVERTTIME (&stop);
   /* ************************************************************* */
   /* SSLConnectOvhd is total time for SSL connection handshaking,  */
   /* minus time spent in NETREAD + NETWRITE. ......................*/
   /* ************************************************************* */
   wl_difftime(&stop,&start,&stop);
   time_struct tmptime = ReadTime;
   addtime (&tmptime, &WriteTime);
   wl_difftime(&stop,&tmptime,&stop);
   DT_PP("Info: wlSocket::SSLConnect(): SSLConnectTime this connect :",tmptime);
   DT_PP("Info: wlSocket::SSLConnect(): SSLConnectOvhd this connect :",stop);
   addtime(&SSLConnectTime,&tmptime);
   addtime(&SSLConnectOvhd,&stop);
   DT_PP("wlSocket::SSLConnect(): SSLConnectOvhd (running sum):",SSLConnectOvhd);

   // restore the read and write time accumulators
   ReadTime = SaveReadTime;
   WriteTime = SaveWriteTime;

#ifdef USE_SKIT
   if (skitSocInitData.failureReasonCode == 0) {
      if (strcmp(sinfo.skitInitData->sec_types, "ALL") == 0)
      PDBG1("The SSL Security type used was: %s\n",
          skitSocInitData.sec_type);

      if (strcmp(skitSocInitData.sec_type, "SSLV2") == 0) {
      PDBG1("The Cipher Specs used are: %d %d %d\n",
          skitSocInitData.cipherSelected[0],
          skitSocInitData.cipherSelected[1],
          skitSocInitData.cipherSelected[2]);
      }
      else {
      PDBG1("The Cipher Spec used was: %c%c\n",
             skitSocInitData.v3cipherSelected[0],
             skitSocInitData.v3cipherSelected[1]);
      }

      /* all done, returning SSL secure socket handle */
      PDBG2("returning 0x%x from skit_soc_init call\n",
         sslHandle);
      return;
   } else {
      PERR("skit_secure_soc_init() returned %d \n",
         skitSocInitData.failureReasonCode);
      perr ("\tCan't SSL connect: %s\n",
         skit_error_str[skit_error(skitSocInitData.failureReasonCode)]);
      perr ("\tCheck that the cipher suite was correctly specified. \n"
            "\tCheck that the keyring contains a CA Certifying Authority \n"
            "\tcertificate for the site you are contacting. \n"
            "\tCheck the password or stashfile presented for the keyring. \n");
      NETCLOSE(sock_fd);
      sock_fd = -1;
      return;
   }
#endif /* USE_SKIT */

#ifdef USE_SSLEAY
   if (0 > rc) {
      if (!close_io) {
         PERR("unable to SSL connect \n");
         ERR_print_errors_fp(stderr);
      }
      SSL_shutdown (sslHandle);
      SSL_free  (sslHandle);
      sslHandle = NULL;
      NETCLOSE(sock_fd);
      sock_fd = -1;
      return;
   }

   char * cipher_used = (char *) SSL_get_cipher (sslHandle);
   PDBG("Using cipher: %s \n", cipher_used);

   /* all done, returning SSL secure socket handle */
   PDBG2("SSL handle is %p\n", sslHandle);

   /* do some final config -- set session timeouts */
#if 0
   SSL_set_timeout (sinfo.session, sinfo.sess_timeout);
   SSL_CTX_set_cipher_list (sinfo.context, cipher_used);
#endif

   // Get and print info about the certificate used at 
   // the far side of this connection. 
   if (debug) {
      X509 *certif;

      certif = SSL_get_peer_certificate (sslHandle);
      if (certif) {
         char * str;
         PDBG1("X509 Certificate used by the server has: \n");
  
         str = X509_NAME_oneline (X509_get_subject_name (certif), NULL, 0);
         PDBG1("Certificate Subject: %s\n", str);
         Free (str);

         str = X509_NAME_oneline (X509_get_issuer_name  (certif), NULL, 0);
         PDBG1("Certificate Issuer: %s \n", str);
         Free (str);

         /* We could do all sorts of certificate verification stuff 
          * here before deallocating the certificate ... But we have 
          * no need for this right now */
         X509_free (certif);
      }
   }

#endif /* USE_SSLEAY */

   return;

} /* END wlSocket::SSLConnect() */

/* ---------------------------------------------------------------- */
/* StartSSL needs to be called just once, to initialize the session. */

int
wlSocket::StartSSL (wlSSLInfo &sinfo)
{
   // don't do ssl if we aren't authorized to do that.
   if (0 == sinfo.use_ssl) return -1;

   // we can act as server only if we are initialized to do that
   if (0 == sinfo.be_server) return -1;

   use_ssl = 1;

    // XXX shouldn't we be checking for a non-null handle ???
   sslHandle = NULL;

#ifdef USE_SKIT
   skit_soc_init_data *spoofSocket;
   spoofSocket = (skit_soc_init_data *) malloc (sizeof (skit_soc_init_data));

   spoofSocket->fd = sock_fd;      /* socket file descriptor */

   spoofSocket->hs_type = AS_SERVER; 

   if (strcmp(sinfo.server_ssl_version,"SSLV2") == 0) {
      spoofSocket->cipher_specs = sinfo.server_cipher; 
      spoofSocket->v3cipher_specs = 0x0;
   } else {
      spoofSocket->cipher_specs = 0x0;
      spoofSocket->v3cipher_specs = sinfo.server_cipher; 
   }
   spoofSocket->DName = sinfo.distinguished_name;

   spoofSocket->skread = skREAD;              /* callback */
   spoofSocket->skwrite = skWRITE;            /* callback */
   spoofSocket->sec_type = NULL;              /* this will be returned */
   spoofSocket->skit_data = sinfo.skitInitData;

   PDBG2("about to init sockfd=%d\n", sock_fd);  
   sec_soc_data       *spoofHandle; /* ssl handle returned if using SSL */
   spoofHandle = skit_secure_soc_init (spoofSocket);

   sslHandle = spoofHandle;

   if (spoofSocket->failureReasonCode == 0) {
      if (strcmp(sinfo.skitInitData->sec_types, "ALL") == 0)
         PDBG1("SSL Security type used was: %s\n",
            spoofSocket->sec_type);

      if (strcmp(spoofSocket->sec_type, "SSLV2") == 0) {
         PDBG1("V2 Cipher Specs used are: %d %d %d\n",
            spoofSocket->cipherSelected[0],
            spoofSocket->cipherSelected[1],
            spoofSocket->cipherSelected[2]);
      } else {
         PDBG1("V3 Cipher Spec used was: %c%c\n",
            spoofSocket->v3cipherSelected[0],
            spoofSocket->v3cipherSelected[1]);
      }

      /* all done, returning SSL secure socket handle */
      PDBG2("sockfd=%d handle=%p\n", 
         sock_fd, spoofHandle);
      return(sock_fd);
   } else {
      PERR("skit_secure_soc_init() returned %d\n",
         spoofSocket->failureReasonCode);
      perr ("\tCan't SSL connect:%s\n",
         skit_error_str[skit_error(spoofSocket->failureReasonCode)]);
      perr ("\tCheck that the cipher suite was correctly specified. \n"
            "\tCheck that the keyring contains a CA Certifying Authority \n"
            "\tcertificate for the site you are contacting. \n"
            "\tCheck the password or stashfile presented for the keyring. \n");

      NETCLOSE(sock_fd);
      sock_fd = -1;
      return GET_SSL_CONNECT_ERROR;
   }
#endif /* USE_SKIT */

#ifdef USE_SSLEAY
   if (!sinfo.certificate_file) {
      PERR("a certificate is required in order to act as a server\n"
         "\tSpecify a certificate with the -k flag\n");
   }

   sslHandle = SSL_new (sinfo.context);
   /* SSL_clear (sslHandle);  not needed, done by SSL_new */
   SSL_set_accept_state(sslHandle);
   SSL_set_fd (sslHandle, sock_fd);
   int rc = SSL_accept (sslHandle);
   if (0 >= rc) 
   {
      int serr;
      long verr;
      serr = SSL_get_error(sslHandle, rc);

      PERR("SSL_accept() returned rc=%d err=%d\n", rc, serr);
      switch (serr) {
         case SSL_ERROR_WANT_WRITE:
         case SSL_ERROR_WANT_READ:
         case SSL_ERROR_WANT_X509_LOOKUP:
            perr ("\tblocked\n");
            break;
         case SSL_ERROR_SYSCALL:
         case SSL_ERROR_SSL:
            perr ("\tunable to negotiate\n");
            break;
         case SSL_ERROR_ZERO_RETURN:
         case SSL_ERROR_NONE:
         default:
            perr ("\tUnknown error\n");
      }
      perr ("\tPerhaps you are using an export-grade browser (40-bit)\n"
          "\twhereas you have specified export-controlled RSA key\n"
          "\t(>512 bits) to webmon?\n");


      verr = SSL_get_verify_result(sslHandle);
      if (X509_V_OK != verr)
      {
         PERR("X509 certif verify error: %s \n",
            X509_verify_cert_error_string(verr));
      }
      else
      {
         ERR_print_errors_fp(stderr);
      }
      NETCLOSE(sock_fd);
      sock_fd = -1;
      return GET_SSL_CONNECT_ERROR;
   }

   /* do some final config -- set session timeouts */
   {
      SSL_SESSION *sess = SSL_get_session (sslHandle);
      SSL_set_timeout (sess, sinfo.sess_timeout);
   }

   if (debug) {
      char tmpbuff[1000];

      char *str = SSL_get_shared_ciphers(sslHandle,tmpbuff, 999);
      if (str) PDBG("Shared ciphers:%s\n",tmpbuff);

      SSL_CIPHER *ciph = SSL_get_current_cipher(sslHandle);
      const char *cstr = SSL_CIPHER_get_name(ciph);
      if (cstr) 
      {
         PDBG("Using cipher:%s %s\n", 
            SSL_CIPHER_get_version (ciph), cstr);
      }
      else 
      {
         PDBG("** encryption is not on ** \n");
      }

      if (sslHandle->hit) {
         PDBG("reusing session-id\n");
      }
   }

#endif /* USE_SSLEAY */
   return sock_fd;

} /* END wlSocket::StartSSL () */

/* ---------------------------------------------------------------- */
/* ---------------------------------------------------------------- */
/* ---------------------------------------------------------------- */
/* ---------------------------------------------------------------- */
/* ---------------------------------------------------------------- */

#ifdef USE_SKIT
/* **************************************************************** */
/* these routines are called by skit toolkit to send or receive     */
/* data ........................................................... */
/* **************************************************************** */

int LINKAGE
skREAD(int fd, void * data_buff, int num_bytes_to_read)
{
   int rc = 0;
   time_struct start, stop;
   /* we keep a running total of time spent in NETREAD in NETREADtime */
   TIMESTAMP(&start);
   rc = NETREAD(fd, data_buff, num_bytes_to_read);
   TIMESTAMP(&stop);
   CONVERTTIME (&start);
   CONVERTTIME (&stop);
   wl_difftime(&stop,&start,&stop);
   addtime(&NETREADtime,&stop);
   PDBG2("fd=%d read %d of %d bytes\n",
             fd, rc, num_bytes_to_read);
   return(rc);
}

int LINKAGE
skWRITE(int fd, void * data_buff, int num_bytes_to_write)
{
   int rc = 0;
   time_struct start, stop;
   /* we keep a running total of time spent in NETWRITE in NETWRITEtime */
   TIMESTAMP(&start);
   rc = NETWRITE(fd,data_buff,num_bytes_to_write);
   TIMESTAMP(&stop);
   CONVERTTIME (&start);
   CONVERTTIME (&stop);
   wl_difftime(&stop,&start,&stop);
   addtime(&NETWRITEtime,&stop);
   PDBG2("fd=%d wrote %d of %d bytes\n",
              fd, rc, num_bytes_to_write);
   return(rc);
}

#endif /* USE_SKIT */


#ifdef USE_SSLEAY

long
wlSocket::
raw_tcp_timing_callback (BIO *bio, int mode, const char *buf, int buflen, 
            long junk, long retval)
{
   wlSocket *sock = (wlSocket *) BIO_get_callback_arg (bio);

   // wlThreadPool::ChkStk();
   /* we keep a running total of time spent in NETREAD in ReadTime */
   if (BIO_CB_READ == ((~BIO_CB_RETURN) & mode)) {
      if (BIO_CB_RETURN & mode) {
         TIMESTAMP(&(sock->rd_stop));
         CONVERTTIME (&(sock->rd_start));
         CONVERTTIME (&(sock->rd_stop));
         wl_difftime(&(sock->rd_stop), &(sock->rd_start),&(sock->rd_stop));
         addtime(&(sock->ReadTime),&(sock->rd_stop));
         PDBG2("read %d of %d bytes\n", retval, buflen);
         DT_PP1 ("Info: skREAD(): NETREADtime: ", sock->ReadTime);
      } else {
         TIMESTAMP(&(sock->rd_start));
      }
   }

   if (BIO_CB_WRITE == ((~BIO_CB_RETURN) & mode)) {
      if (BIO_CB_RETURN & mode) {
         TIMESTAMP(&(sock->wr_stop));
         CONVERTTIME (&(sock->wr_start));
         CONVERTTIME (&(sock->wr_stop));
         wl_difftime(&(sock->wr_stop), &(sock->wr_start), &(sock->wr_stop));
         addtime (&(sock->WriteTime), &(sock->wr_stop));
         PDBG2("write %d of %d bytes\n", retval, buflen);
         DT_PP1 ("Info: skWRITE(): NETWRITEtime: ", sock->WriteTime);
      } else {
         TIMESTAMP(&(sock->wr_start));
      }
   }

   if (2<debug) {
      char * tmp;
      tmp = SSL_state_string_long (sock->sslHandle);
      PDBG2("SSL Socket Connect state is: %s \n", tmp);
      tmp = SSL_rstate_string_long (sock->sslHandle);
      PDBG2("SSL Socket Read state is: %s \n", tmp);
   }

   return retval;

}
#endif /* USE_SSLEAY */

/* ************************************************************** */
/* interface routines to hide use of or non use of SSL from caller*/
/* ************************************************************** */
int
wlSocket::Write (void * buffer, size_t writelen) 
{
   int status = 0;

   if (BADSOCKET(sock_fd)) return -ENOTCONN;
   num_writes ++;

   close_io = 0;
   if (use_ssl) {
      PDBG("will write %d bytes "
         "to SSL sock fd=%d, handle=%p\n", 
         writelen, sock_fd, sslHandle);

#ifdef USE_SKIT
        status = skit_secure_soc_write(sslHandle, buffer, writelen);
#endif /* USE_SKIT */

#ifdef USE_SSLEAY
        status = SSL_write(sslHandle, (char *) buffer, writelen);
#endif /* USE_SSLEAY */

      if (0 > status && !close_io && !retry_io) 
      {
         PDBG("SSL socket write on fd=%d returned %d\n", sock_fd, status);

#ifdef USE_SKIT
         PDBG("skit error name is %s\n", skit_error_str[skit_error(status)]);
#endif /* USE_SKIT */

#ifdef USE_SSLEAY
         ERR_print_errors_fp(stderr);
#endif /* USE_SSLEAY */

      }
   } else {
      PDBG("will write %d bytes to sock fd=%d \n", writelen, sock_fd);

      status = NETWRITE(sock_fd, buffer, writelen);
      if (0 > status) {
         int norr;
         norr = errno;
         PDBG("unencrypted socket write on fd=%d\n"
              "\tErrno=%d %s\n",
              sock_fd, norr, strerror (norr));
         status = -norr;
      }
      if (alarm_expired || retry_io || close_io)  return(-EINTR);
   }
   return(status);
}

/* --------------------------------------------- */

int
wlSocket::Read (void * buffer, size_t readlen) 
{
   int bytes_read=0;

   if (BADSOCKET (sock_fd)) return -ENOTCONN;
   num_reads ++;

   close_io = 0;
   if (use_ssl) 
   {
#ifdef USE_SKIT
      bytes_read = skit_secure_soc_read(sslHandle, buffer, readlen);
#endif /* USE_SKIT */

#ifdef USE_SSLEAY
      bytes_read = SSL_read(sslHandle, (char *) buffer, readlen);
#endif /* USE_SSLEAY */

      if ((0 > bytes_read) && !close_io && !retry_io
#ifdef USE_SKIT
         /* skit thinks its acceptable to return an error 
          * during normal operations. So just ignore this error.
          */
         && (SKIT_ERROR_BAD_MESSAGE != bytes_read)
#endif /* USE_SKIT */
      ) 
      {
         PERR("SSL socket read returned %d \n", bytes_read);

#ifdef USE_SKIT
         PERR("skit error name is %s\n",
               skit_error_str[skit_error(bytes_read)]);
#endif /* USE_SKIT */

#ifdef USE_SSLEAY
         ERR_print_errors_fp(stderr);
#endif /* USE_SSLEAY */
      }

   } else {

      bytes_read = NETREAD(sock_fd, buffer, readlen);
   }

   if ((0 > bytes_read) && (alarm_expired || close_io || retry_io))
   {
      errno = EINTR;
      return(-EINTR);
   }
   return(bytes_read);
}

/* --------------------------------------------- */

int
wlSocket::Close (void)
{
    if (BADSOCKET(sock_fd)) return -ENOTCONN;

   if (use_ssl && sslHandle) 
   {
#ifdef USE_SKIT
      /* don't reset the socket, since that will 
       * wipe out the session cache !!  */
      /* skit_secure_soc_reset(sslHandle); */
      skit_secure_soc_close(sslHandle);
#endif /* USE_SKIT */

#ifdef USE_SSLEAY
      SSL_shutdown (sslHandle);
      SSL_free  (sslHandle);
#endif /* USE_SSLEAY */

      sslHandle = NULL;
   } 
   PDBG2("about to close fd=%d\n", sock_fd);

   int status = NETCLOSE(sock_fd);
   sock_fd = -1;
   if (status) {
      int norr = errno;
      PERR("non-zero rc=%d\n", status);
      perr ("\terrno=%d (%s)\n", norr, strerror (norr));
      return -norr;
   }

   return 0;
}

/* --------------------------------------------- */

#ifdef WIN32
/* close socket library at exit() time */
void sock_cleanup(void) 
{
    WSACleanup();
}
#endif /* WIN32 */

