//
// class wlRequest
//
// FUNCTION:
// implement utilities to assembling and manipulating HTTP requests
//

#include <stdio.h>
#include <string.h>

#include "generic.h"
#include "header.h"
#include "request.h"
#include "super.h"

// =====================================================================

wlMessage :: wlMessage (void)
{
}

// =====================================================================

size_t
wlMessage :: HeaderLength (void)
{
   // XXX this design results in excessive copying of data,
   // because the (char *) cast causes a defrag to run.
   char *header_start = (char *) message;
   char *header_end = strstr (header_start, "\n\n");
   char *header_crend = strstr (header_start, "\n\r\n");

   if (!header_end && !header_crend) return 0;

   if (!header_end) return (header_crend-header_start +3); 
   if (!header_crend) return (header_end-header_start +2);

   if (header_end < header_crend) return (header_end-header_start +2);
   return (header_crend-header_start +3);
}

// =====================================================================

void
wlMessage :: Split (void)
{

   // XXX this design causes excessive copying of data, because the 
   // header object makes a copy of the reply buffer, rather than being 
   // merely a pointer into it.

   char *message_start = (char *) message;
   char *start_start = message_start;
   if (!start_start) return;
   start_start += strspn (start_start, "\r\n\v");   // skip over leading garbage
   size_t startlen = strcspn (start_start, "\r\n\v");
   startline.Memcpy (start_start, startlen);        // copy the start line

   char * msg_end = message_start + message.Memlen(); // total msg len so far
   char *header_start = start_start + startlen +1;  // find start of header
   if (header_start > msg_end) header_start = msg_end; // oops ran over the end
   if ('\n' == *header_start) header_start ++;      // skip over LF if needed

   size_t header_length = HeaderLength ();          // header length is zero if
   if (0 == header_length) {                        // crlf not yet arrived 
      header.Strcpy(header_start);                  // copy the buffer into header
      body = NULL;
      return;                                       // error !!
   }

   size_t slen = header_start - message_start;      // total len of startline
   size_t hlen = header_length - slen;              // header minus status line
   size_t striplen = hlen;
   if ('\n' == header_start [striplen-1]) striplen --; // remove "blank line"
   if ('\r' == header_start [striplen-1]) striplen --;
   header.Memcpy (header_start, striplen);

   char * body_start = header_start + hlen;
   size_t bodylen = message.Memlen();
   bodylen -= (body_start - message_start);          // body may be binary
   body.Memcpy (body_start, bodylen);                // copy remainder into body
}

void
wlMessage :: Join (void)
{
   // although this should work ... it uses ambiguous signatures ...
   // message = startline;
   // message += "\r\n";
   // message += header;
   // message += "\r\n";
   // message += body;

   // this is a bit safer since it is less ambiguous in method signature
   message.Memcpy (startline);
   message.Strcat ("\r\n");
   message.Memcat (header);
   message.Strcat ("\r\n");
   message.Memcat (body);
}

// =====================================================================

#define HTTP_VERSION "HTTP/1.1"

wlRequest :: wlRequest (void)
{
   version = HTTP_VERSION;
   bug_compat = 1;
}

void
wlRequest :: Assemble (void)
{
   // the body contains post data or other HTTP data
   if ((char *) body) 
   {
      size_t bodylen = body.Memlen();
      if (0 != bodylen)
      {
         header.AddIfAbsent ("Content-type", 
                             "application/x-www-form-urlencoded");

         // check for netscape bug compatibility
         // if the request is POST and the body is \r\n terminated
         // then report the length as being short by two.
         if (bug_compat && !strcmp ("POST", (char *)method))
         {
            char * booty = (char *) body;
            size_t orig_len = bodylen;
            if ('\n' == booty[bodylen-1]) bodylen--; 
            if ('\r' == booty[bodylen-1]) bodylen--; 
#if 0
            if (orig_len != bodylen) {
               perr ("Warning: wlRequest::Assemble(): "
                     " emulating Netscape Content-Length bug\n"
                     "\treported length=%d actual length=%d\n",
                     bodylen, orig_len);
            }
#endif
         }

         char tmpbuff[16];
         // snprintf (tmpbuff, 16, "%u", bodylen);
         sprintf (tmpbuff, "%u", bodylen);
         header.ReplaceOrAddField ("Content-Length", tmpbuff);
      }
   }

   startline.Memcpy (method);
   startline.Strcat (" ");
   startline.Memcat (url);
   startline.Strcat (" ");
   startline.Memcat (version);

   Join ();
} 

void
wlRequest :: Analyze (void)
{

   // XXX this design causes excessive copying of data, because the 
   // header object makes a copy of the reply buffer, rather than being 
   // merely a pointer into it.

   Split();
   char *request_line = (char *) startline;

   char *method_start = request_line;
   size_t methlen = strcspn (method_start, " \t\v");
   method.Memcpy (method_start, methlen);

   char *url_start = method_start + methlen;
   url_start += strspn (url_start, " \t\v");
   size_t urllen = strcspn (url_start, " \t\v");
   url.Memcpy (url_start, urllen);

   char *version_start = url_start + urllen;
   version_start += strspn (version_start, " \t\v");
   size_t verslen = strcspn (version_start, " \t\v");
   version.Memcpy (version_start, verslen);
}


// =====================================================================

wlResponse :: wlResponse (void)
{
   status_code = 0;
}

void
wlResponse :: AnalyzeHeader (void)
{

   // XXX this design causes excessive copying of data, because the 
   // header object makes a copy of the reply buffer, rather than being 
   // merely a pointer into it.

   Split();
   char *status_line = (char *) startline;

   if (strncmp (status_line, "HTTP/1.", 7)) {
      status_code = -1;
      return;                                       // error !!
   }

   version.Memcpy (status_line, 8);                 // copy version field

   char *stet = status_line + 8;                    // skip over HTTP/1.x
   stet += strspn (stet, " \t");                    // skip whitespace
   size_t stetlen = strcspn (stet, "\r\n\v");
   status.Memcpy (stet, stetlen);                   // copy the response code

   status_code = atoi (stet);                       // numeric response code
}

// =====================================================================
