// 
// FILE:
// pool.h
//
// FUNCTION:
// Implements a thread pool.  This class helps manage a pool of 
// scarce resources that need a separate thread to make use of 
// each resource.  This is accomplished with a pair of classes. 
// 
// The wlThread class is and abstract virtual base class that 
// will serve as the "worker".  The real worker needs to derive 
// from this class and provide the RunInThread() method.
//
// The wlThreadPool class is the thread/resource manager.  
// To use it, create an array of "worker" classes and pass
// that array to the wlThreadPool constructor.  Next, whenever
// some work item needs to be done, you can 'check out' or
// 'reserve' a worker out of the pool with the GetThread() method.
// Tell the worker whatever you need, and then say Go().
// This will cause the worker to go off in a separate thread 
// and call the RunInThread() method.  When RunInThead() returns,
// the worker will automatically check itself back into the pool.
// 
// It there are no free workers, the GetThread() method will block 
// until a worker finishes a previous task and returns to the pool.
// The Join(), method will block until all of the workers have 
// returned to the pool.
//
// The JoinAll() method will block until all of the workers have
//    returned to the pool, i.e. until all threads are idle.
//
// The destructor will shut down all of the threads.  It will
// block until all of the threads are destroyed.  Note that
// a thread must be idle before it can be destroyed; thus,
// the desctructor might hang as long as there are hung
// (non-idle) threads.  Note that most system calls are 
// cancellation points.
//
// The NeverMind() method is a handy alternative to the Go() method.
//    If a thread was obtained with the GetThread() function, but 
//    its then determined that the thread is no longer needed,
//    then the NeverMind() method can be called to return it to the 
//    free pool. Whenever a free thread has been checked out with
//    GetThread(), it *must* be checked back in sooner or later
//    by calling either Go() (to do something) or NeverMind() to
//    do nothing and return to the pool.
//
// The get_tid() subroutine is a quick hack to get a thread id 
// for debug-printing reasons.
//
// HISTORY:
// Created by Linas Vepstas October 1999
//
#ifndef __WL_POOL_H__
#define __WL_POOL_H__

#include <errno.h>
#include <pthread.h>

class wlThreadPool;

class wlThread
{
   public:
      void Go(void);
      void NeverMind(void);
      int GetId(void);

      void ChkStk (void);       // stack size monitoring utility
      
   protected:
      virtual void RunInThread (void) = 0;

   private:
      void Dispatch (void);
      wlThreadPool *home;
      pthread_t thread;
      pthread_mutex_t lock;
      pthread_cond_t got_work;
      int id;
      int busy;
      unsigned long stack_base;  // approximate location of stack base
      size_t        stack_used;  // approximate max stack size

   friend class wlThreadPool;
};

extern "C" {
extern int get_tid (void);
};

class wlThreadPool
{
   public:
      wlThreadPool(int nthreads, wlThread **array);
      ~wlThreadPool ();

      wlThread * GetThread (void);
      void JoinAll (void);
      static void ChkStk(void);   // stack size debugging utility

   private:
      int free_threads;           // number of idle threads
      int total_threads;          // total number of threads
      int pool_entries;           // total thread slots
      pthread_mutex_t pool_lock;  // pool resource lock
      pthread_cond_t pool_empty;  // no free threads
      pthread_cond_t pool_full;   // no busy threads
      static pthread_key_t tid_key;   // thread_id key
      static pthread_key_t self_key;  // thread wlThread 'this' pointer
      static int tid_key_is_init; // thread_id key is initialized

      wlThread **pool;

      static void * dispatch (void *arg);
   friend class wlThread;
   friend int get_tid (void);
};


#endif /* __WL_POOL_H__ */

