//
// class wlHeader
//
// FUNCTION:
// Implements an HTTP header
// See the header.h file for additional documentation
//

#include <string.h>
#include "generic.h"
#include "header.h"

/* ============================================================= */

wlKeyPairs::~wlKeyPairs()
{
}

int
wlKeyPairs :: Remove (const char * field)
{
   defrag();
   return replace (field, NULL, 0, 0);
}

int
wlKeyPairs :: AddIfAbsent (const char * field, const char * newvalue)
{
   defrag();
   return replace (field, newvalue, 1, 0);
}

int
wlKeyPairs :: ReplaceField (const char * field, const char * newvalue)
{
   defrag();
   return replace (field, newvalue, 0, 1);
}

int
wlKeyPairs :: ReplaceOrAddField (const char * field, const char * newvalue)
{
   defrag();
   return replace (field, newvalue, 1, 1);
}

/* ============================================================= */

int 
wlKeyPairs :: ReplaceFields (char ** fields, char ** newvalues)
{
   if (!fields || !newvalues) return 0;
   defrag();

   int i=0;
   int num_replacements = 0;
   const char * field = fields[0];
   const char * newvalue = newvalues[0];
   while (field) {
      num_replacements += replace (field, newvalue, 0, 1);
      i++;
      field = fields[i];
      newvalue = newvalues[i];

      // hack alert -- the semantics would be ever so slightly more correct
      // for ugly - nested - malformed headers if we did a defrag(); inside 
      // the loop. However, we don't as I don't expect headers to get this 
      // convoluted and I want to save some cpu time.  That is, unless 
      // someone is intentionally trying to create icky nested headers ...
      // defrag ();
   }
   return num_replacements;
}

int 
wlKeyPairs :: ReplaceFields (wlKeyPairs &pairs)
{
   wlString newpairs(pairs);

   int num_replacements = 0;
   char * field = (char *) newpairs;
   char * value = val_start (field);
   char * next_field = fld_start (value);
   while (field) {
      num_replacements += replace (field, value, 0, 1);
      field = next_field;
      value = val_start (field);
      next_field = fld_start (value);

      // hack alert -- the semantics would be ever so slightly more correct
      // for ugly - nested - malformed headers if we did a defrag(); inside 
      // the loop. However, we don't as I don't expect headers to get this 
      // convoluted and I want to save some cpu time.  That is, unless 
      // someone is intentionally trying to create icky nested headers ...
      // defrag ();
   }
   return num_replacements;
}

int
wlKeyPairs :: ReplaceOrAddFields (char ** fields, char ** newvalues)
{
   if (!fields || !newvalues) return 0;
   defrag();

   int i=0;
   int num_replacements = 0;
   const char * field = fields[0];
   const char * newvalue = newvalues[0];
   while (field) {
      num_replacements += replace (field, newvalue, 1, 1);
      i++;
      field = fields[i];
      newvalue = newvalues[i];

      // hack alert -- the semantics would be ever so slightly more correct
      // for ugly - nested - malformed headers if we did a defrag(); inside 
      // the loop. However, we don't as I don't expect headers to get this 
      // convoluted and I want to save some cpu time.  That is, unless 
      // someone is intentionally trying to create icky nested headers ...
      // defrag ();
   }
   return num_replacements;
}

int 
wlKeyPairs :: ReplaceOrAddFields (wlKeyPairs &pairs)
{
   wlString newpairs(pairs);

   int num_replacements = 0;
   char * field = (char *) newpairs;
   char * value = val_start (field);
   char * next_field = fld_start (value);
   while (field) {
      num_replacements += replace (field, value, 1, 1);
      field = next_field;
      value = val_start (field);
      next_field = fld_start (value);

      // hack alert -- the semantics would be ever so slightly more correct
      // for ugly - nested - malformed headers if we did a defrag(); inside 
      // the loop. However, we don't as I don't expect headers to get this 
      // convoluted and I want to save some cpu time.  That is, unless 
      // someone is intentionally trying to create icky nested headers ...
      // defrag ();
   }
   return num_replacements;
}

/* ============================================================= */
/* ============================================================= */
/* ============================================================= */
/* ============================================================= */

wlHeader::~wlHeader()
{
}

wlHeader &
wlHeader :: operator= (const wlString& ref)
{
   Memcpy ((wlString&) ref);
   return *this;
}

/* ============================================================= */

char *
wlHeader :: GetValue (const char *field)
{
   defrag ();

   if (!field || !head) return NULL;
   size_t flen = strlen (field);
   char *field_start = head->start;
   while (1) {
      field_start = strcasestr (field_start, field);
      if (!field_start) return NULL;

      // to be a valid field start, it must start on a new line
      if (field_start == head->start) break;
      if ('\n' == *(field_start-1)) break;
      field_start += flen;
   }

   // we found the field; find the start of the value
   char *field_end = strchr (field_start, ':');
   char *value_start = field_end +1;
   value_start += strspn (value_start, " \t");

   return value_start;
}

/* ============================================================= */

char *
wlHeader :: val_start (char *feeled_start)
{
   if (!feeled_start) return NULL;

   char *feeled_end = strchr (feeled_start, ':');
   if (!feeled_end) return NULL;
   *feeled_end = 0;

   char *valou_start = feeled_end +1;
   valou_start += strspn (valou_start, " \t");

   return valou_start;
}

/* ============================================================= */

char *
wlHeader :: fld_start (char *valou_start)
{
   if (!valou_start) return NULL;

   char *valou_end, *feeled_start, *next = valou_start;
   do {
      valou_end = next + strcspn (next, "\r\n");
      if (0 == *valou_end) return NULL;

      feeled_start = valou_end +1;
      if ('\r' == *feeled_start) feeled_start ++;
      if ('\n' == *feeled_start) feeled_start ++;
      if (0 == *feeled_start) 
      {
         *valou_end = 0;
         return NULL;
      }
      next = feeled_start;
   } while ('\t' == *next);


   *valou_end = 0;
   return feeled_start;
}

/* ============================================================= */

int
wlHeader :: replace (const char * field, const char * newvalue, 
                     int do_add, int do_replace)
{
   int do_remove = 0;
   if (!field) return 0;
   if (!newvalue) { do_remove = 1; do_replace = 0; do_add = 0; }

   // if empty and don't add, do nothing
   if (!head && !do_add) return 0;

   // if empty and append, then append
   if (!head) {
      Strcpy (field);
      Strcat (": ");
      Strcat (newvalue);
      Strcat ("\r\n");
      return 1;
   }

   Chunk *cursor = head;

   // search for the field in this header
   size_t flen = strlen (field);
   char *field_start = NULL;
   while (cursor) {
      field_start = strcasestr (cursor->start, field);
      if (field_start == cursor->start) goto gotone;
      while (field_start) {
         // to be a valid field start, it must start on a new line
         if ('\n' == *(field_start-1)) goto gotone;
         field_start += flen;
         field_start = strcasestr (field_start, field);
      }
      cursor = cursor->next;
   }

gotone:
   if (field_start && (do_replace || do_remove)) 
   {
      // we found the field; find the start and end of the old value
      char *field_end = strchr (field_start+flen, ':');
      char *value_start = field_end + 1;
      if (' ' == *value_start) value_start ++;
      char *value_end = strpbrk (value_start, "\r\n");
      if (!value_end) value_end = value_start + strlen (value_start);
   
      if (0 != *value_end)
      {
         // handle multi-line fields as per HTTP/1.1 spec
         char *nextline = value_end +1;
         if ('\n' == *nextline) nextline ++;
         while ((' ' == *nextline) || ('\t' == *nextline)) 
         {
            value_end = strpbrk (nextline, "\r\n");
            if (!value_end) 
            {
               value_end = nextline + strlen (nextline);
               break;
            }
            nextline = value_end +1;
            if ('\n' == *nextline) nextline ++;
         }
      }

      if (do_replace) 
      {
         // replace old value with new value
         size_t nvlen = strlen (newvalue);
         char * nv = new char [nvlen+1];
         strcpy (nv, newvalue);
         subsone (cursor, value_start, value_end-value_start, nv, nvlen);
         return 1;
      }
      else if (do_remove)
      {
         // be sure to remove only one CRLF at the end as well.
         if ('\r' == *value_end) value_end ++;
         if ('\n' == *value_end) value_end ++;
         subsone (cursor, field_start, value_end-field_start, NULL, 0);
         return 1;
      }
      
   } 
   else
   if (!field_start && do_add) 
   {
      // field was NOT found, add the value ... 
      Strcat (field);
      Strcat (": ");
      Strcat (newvalue);
      Strcat ("\r\n");
      return 1;
   }
   return 0;
}

/* ============================================================= */
/* ============================================================= */
/* ============================================================= */
/* ============================================================= */

wlURLEncoding::~wlURLEncoding()
{
}

wlURLEncoding &
wlURLEncoding :: operator= (const wlString& ref)
{
   Memcpy ((wlString&) ref);
   return *this;
}

/* ============================================================= */

char *
wlURLEncoding :: GetValue (const char *field)
{
   defrag ();

   if (!field || !head) return NULL;
   size_t flen = strlen (field);
   char *field_start = head->start;
   while (1) {
      field_start = strcasestr (field_start, field);
      if (!field_start) return NULL;

      // to be a valid field, it must start after an ampersand
      // and end with an equals sign. No embedded whitespace allowed.
      if ('=' != field_start[flen]) {
         field_start += flen;
         continue;
      }
      if (field_start == head->start) break;
      if ('&' == *(field_start-1)) break;
      field_start += flen;
   }

   // we found the field; find the start of the value
   char *value_start = field_start + flen + 1;

   return value_start;
}

/* ============================================================= */

char *
wlURLEncoding :: val_start (char *feeled_start)
{
   if (!feeled_start) return NULL;

   char *feeled_end = strchr (feeled_start, '=');
   if (!feeled_end) return NULL;
   *feeled_end = 0;

   char *valou_start = feeled_end +1;
   return valou_start;
}

/* ============================================================= */

char *
wlURLEncoding :: fld_start (char *valou_start)
{
   if (!valou_start) return NULL;

   char * valou_end = strchr (valou_start, '&');
   if (!valou_end) return NULL;
   *valou_end = 0;

   char * feeled_start = valou_end +1;
   return feeled_start;
}

/* ============================================================= */

int
wlURLEncoding :: replace (const char * field, const char * newvalue, 
                     int do_add, int do_replace)
{
   int do_remove = 0;
   if (!field) return 0;
   if (!newvalue) { do_remove = 1; do_replace = 0; do_add = 0; }

   // if empty and don't add, do nothing
   if (!head && !do_add) return 0;

   // if empty and append, then append
   if (!head) {
      Strcpy (field);
      Strcat ("=");
      Strcat (newvalue);
      return 1;
   }

   Chunk *cursor = head;

   // search for the field in this header
   char *field_start = NULL;
   size_t flen = strlen (field);
   while (cursor) {
      field_start = strcasestr (cursor->start, field);
      while (field_start) {
         // to be a valid field start, it must start after an ampersand,
         // and end with an equals sign
         if ('=' != field_start[flen]) {
            field_start += flen;
            field_start = strcasestr (field_start, field);
            continue;
         } 
         if (field_start == cursor->start) goto gotone;
         if ('&' == *(field_start-1)) goto gotone;

         field_start += flen;
         field_start = strcasestr (field_start, field);
      }
      cursor = cursor->next;
   }

gotone:

   if (field_start && (do_replace || do_remove)) {

      // we found the field; find the start and end of the old value
      char *value_start = field_start + flen +1;
      char *value_end = strchr (value_start, '&');
      if (!value_end) value_end = value_start + strlen (value_start);
   
      if (do_replace)
      {
         // replace old value with new value
         size_t nvlen = strlen (newvalue);
         char * nv = new char [nvlen+1];
         strcpy (nv, newvalue);
         subsone (cursor, value_start, value_end-value_start, nv, nvlen);
         return 1;
      } 
      else if (do_remove)
      {
         // be sure to remove a trailing ampersand as well 
         if ('&' == *value_end) value_end ++;
         subsone (cursor, field_start, value_end-field_start, NULL, 0);
         return 1;
      }
      
   } else
   if (!field_start && do_add) 
   {
      // field was NOT found, add the value ... 
      Strcat ("&");
      Strcat (field);
      Strcat ("=");
      Strcat (newvalue);
      return 1;
   }
   return 0;
}

/* ============================================================= */
/* ============================================================= */
/* ============================================================= */
/* ============================================================= */

wlCookie &
wlCookie :: operator= (const wlString& ref)
{
   Memcpy ((wlString&) ref);
   return *this;
}

/* ============================================================= */

char *
wlCookie :: GetValue (const char *field)
{
   defrag ();

   if (!field || !head) return NULL;
   size_t flen = strlen (field);
   char *field_start = head->start;
   char *field_end = NULL;
   while (1) {
      field_start = strcasestr (field_start, field);
      if (!field_start) return NULL;

      // To be a valid field, it must start after a semicolon 
      // possibly followed by whitespace, and end in an equals sign,
      // possibly preceeded by whitespace.

      field_end = field_start + flen;
      field_end += strspn (field_end, "\t\v ");

      if ('=' != *field_end) 
      {
         // check for an important exception: 'secure'
         // 'secure' has no value, its eaither present or not present.
         // Do donote its absence return null, else return non-null.
         if ((('\n' == *field_end) || (';' == *field_end) ||
              ('\r' == *field_end) || (0x0 == *field_end)) &&
              (0 == strncasecmp (field_start, "secure", 6)))
         {
            return field_start;
         }
         field_start = field_end;
         continue;
      }

      if (field_start == head->start) break;
      field_start --;
      while ((field_start >= head->start) &&
             ((' '  == *field_start) ||
              ('\t' == *field_start) ||
              ('\v' == *field_start)) ) field_start-- ;


      if (field_start == head->start) break;
      if (';' == *field_start) break;
      field_start = field_end;
   }

   // we found the field; find the start of the value
   char *value_start = field_end + 1;
   value_start += strspn (value_start, "\t\v ");

   return value_start;
}

/* ============================================================= */

char *
wlCookie :: val_start (char *feeled_start)
{
   if (!feeled_start) return NULL;

   char *feeled_end = strchr (feeled_start, '=');
   if (!feeled_end) return NULL;
   char *valou_start = feeled_end +1;

   feeled_end --;
   while (((' ' == *feeled_end) || ('\t' == *feeled_end)) && 
          (feeled_end > feeled_start)) feeled_end --;
   feeled_end ++;
   *feeled_end = 0;

   valou_start += strspn (valou_start, "\t\v ");
   return valou_start;
}

/* ============================================================= */

char *
wlCookie :: fld_start (char *valou_start)
{
   if (!valou_start) return NULL;

   char * valou_end = strchr (valou_start, ';');
   if (!valou_end) return NULL;
   *valou_end = 0;

   char * feeled_start = valou_end +1;
   feeled_start += strspn (feeled_start, "\t\v ");
   return feeled_start;
}

/* ============================================================= */

int
wlCookie :: replace (const char * field, const char * newvalue, 
                     int do_add, int do_replace)
{
   int do_remove = 0;
   if (!field) return 0;
   if (!newvalue) { do_remove = 1; do_replace = 0; do_add = 0; }

   // if empty and don't add, do nothing
   if (!head && !do_add) return 0;

   // if empty and append, then append
   if (!head) {
      Strcpy (field);
      Strcat ("=");
      Strcat (newvalue);
      Strcat (";");
      return 1;
   }

   Chunk *cursor = head;

   // search for the field in this header
   char *field_start = NULL;
   char *field_end = NULL;
   size_t flen = strlen (field);
   while (cursor) {
      field_start = strcasestr (cursor->start, field);
      while (field_start) {
         // To be a valid field, it must start after a semicolon 
         // possibly followed by whitespace, and end in an equals sign,
         // possibly preceeded by whitespace.
         field_end = field_start + flen;
         field_end += strspn (field_end, "\t\v ");
         if ('=' != *field_end) {

            // check for an important exception: 'secure'
            // 'secure' has no value, its eaither present or not present.
            // Do donote its absence return null, else return non-null.
            if ((('\n' == *field_end) || (';' == *field_end) ||
                 ('\r' == *field_end) || (0x0 == *field_end)) &&
                 (0 == strncasecmp (field_start, "secure", 6)))
            {
               field_end --;
            } else {
               field_start = field_end;
               field_start = strcasestr (field_start, field);
               continue;
            }
         }
         while ((field_start >= head->start) &&
                ((' '  == *field_start) ||
                 ('\t' == *field_start) ||
                 ('\v' == *field_start)) ) field_start-- ;

         if (field_start == cursor->start) goto gotone;
         if (';' == *(field_start-1)) goto gotone;

         field_start = field_end;
         field_start = strcasestr (field_start, field);
      }
      cursor = cursor->next;
   }

gotone:
   if (field_start && (do_replace || do_remove))
   {

      // we found the field; find the start and end of the old value
      char *value_start = field_end +1;
      char *value_end = strchr (value_start, ';');
      if (!value_end) value_end = value_start + strlen (value_start);
   
      if (do_replace)
      {
         // replace old value with new value
         size_t nvlen = strlen (newvalue);
         char * nv = new char [nvlen+1];
         strcpy (nv, newvalue);
         subsone (cursor, value_start, value_end-value_start, nv, nvlen);
         return 1;
      }
      else if (do_remove)
      {
         // be sure to remove a trailing semicolon as well
         if (';' == *value_end) value_end ++;
         subsone (cursor, field_start, value_end-field_start, NULL, 0);
         return 1;
      }
      
   } else
   if (!field_start && do_add) 
   {
      // field was NOT found, add the value ... 
      Strcat (field);
      Strcat ("=");
      Strcat (newvalue);
      Strcat (";");
      return 1;
   }
   return 0;
}

/* ============================================================= */
