
/**************************************************************************
 *                                                                        *
 *               Copyright (C) 1995 Silicon Graphics, Inc.                *
 *                                                                        *
 *  These coded instructions, statements, and computer programs were      *
 *  developed by SGI for public use.  If any changes are made to this code*
 *  please try to get the changes back to the author.  Feel free to make  *
 *  modifications and changes to the code and release it.                 *
 *                                                                        *
 **************************************************************************/

/* socks redefinition must preceed system headers */
#include "sockify.h" 

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <math.h>
#include <pthread.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>

#ifdef WIN32
#include <winsock.h>
#include <windows.h>
#include <process.h>
#include <io.h>
#endif /* WIN32 */

#ifndef WIN32
#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/in.h>
#include <strings.h>
#include <sys/errno.h>
#include <sys/file.h>
#include <sys/ipc.h>
#include <sys/param.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <unistd.h>
#endif /* WIN32 */

#include "errexit.h"
#include "generic.h"
#include "sysdep.h"
#include "timefunc.h"

/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* assorted string utilites */

#ifdef WIN32
/* these utilities are not defined for some operating systems. */

/* compare two strings with max length, ignoring case */
int 
strncasecmp(const char *str1, const char *str2, size_t len) 
{
    int diff;

    while (*str1 && *str2 && len--) 
   {
        diff = UPPER(*str1) - UPPER(*str2);
        if (diff)
            return diff < 0 ? -1 : 1;
        str1++;
        str2++;
    }

    /* its only a match if *all* of the chars match, and
     * we haven't prematurely reached end of string 
     */ 
    if (0 == len) return 0;
    if (*str1) return 1;
    return -1;
}

int 
strcasecmp(const char *str1, const char *str2) 
{
   size_t len = strlen (str1);
   int retval = strncasecmp (str1, str2, len);
   return retval;
}
#endif /* WIN32 */

#if defined(WIN32) || defined (AIX)
char *
stpcpy (char * dest, const char * src)
{
   strcpy (dest, src);
   return (dest+strlen (src));
}

char *
stpncpy (char * dest, const char * src, size_t len)
{
   size_t mlen;
   strncpy (dest, src, len);
   mlen = strlen (src);
   if (mlen > len) mlen = len;
   return (dest+mlen);
}

#endif /* defined(WIN32) || defined (AIX) */

/* search for str2 in first nchar chars of str1, ignore case.. */ 
/* return pointer to first match, or null. ................... */
char *
strncasestr(const char *str1, const char *str2, size_t len) 
{
    while (*str1 && len--) 
   {
      if (UPPER(*str1) == UPPER(*str2)) 
      {
         if (strncasecmp(str1,str2,strlen(str2)) == 0) 
         {
            return (char *) str1;
         }
      }
      str1++;
   }
   return NULL;
}

/* search for str2 in str1, ignore case.. */ 
/* return pointer to first match, or null. ................... */
char *
strcasestr(const char *str1, const char *str2) 
{
   size_t len = strlen (str1);
   char * retval = strncasestr (str1, str2, len);
   return retval;
}


/* reversed strstr -- search for a needle in the haystack,
 * from the far end */
char *
rstrstr (const char *haystack, const char * needle)
{
    int haylen = strlen (haystack);
    int neelen = strlen (needle);

    const char * hp = haystack + haylen - 1;
    const char * np = needle + neelen - 1;

    if ((0 == neelen) || (0 == haylen)) return 0x0;

    while (hp >= haystack+neelen) {
        if (*hp == *np) {
            --np;
            if (np < needle) return (char *) hp;
        } else {
            np = needle + neelen - 1;
        }
        --hp;
    }

    return 0x0;
}


/* The strpskip() function locates the first occurrence in the
 * string s that does not match any of the characters in "reject".
 * This is the opposite of strpbrk()
 */
char * 
strpskip (const char * s, const char *reject)
{
   size_t i, rlen;
   char * retval;

   if (!s) return NULL;
   if (!reject) return (char *) s;

   rlen = sizeof (reject);
   retval = (char *) s;

   while (*retval) {
      int match = 0;
      for (i=0; i<rlen; i++) {
         if (reject[i] == *retval) {match=1; break; }
      }
      if (!match) return retval;
      retval ++;
   }
   return NULL;
}


/* **************************************************************** */
/* change every occurrence of change_from to change_to in string    */
/* append the result to "result" .................................. */
/* (set result to "" before calling if no append desired) ......... */
/* return the number of times change_from was found. .............. */
/* **************************************************************** */
int 
substitute(const char *string, 
         const char *change_from, 
         const char *change_to, char *result) 
{
   const char *start;
    const char *end;
   int  fromlen = strlen(change_from);
   int  tolen = strlen (change_to);
   int  nfound=0;

   start=string;

   end=strstr(start,change_from);
   while (end) 
   {
      strncpy (result, start, end-start);
      result += end-start;
      strcpy (result, change_to);
      result += tolen;
      start = end + fromlen;
      nfound++;
      end=strstr(start,change_from);
   }
   strcpy (result,start);
   return (nfound);
}

/* ================================================================ */
/* ================================================================ */
/* ================================================================ */
/* ================================================================ */

/* prettify() cleans up subroutine names. 
 * AIX/xlC has the habit of printing signatures not names; clean this up.
 * On other operating systems, truncate name to 30 chars.
 * Note this routine is not thread safe. Note we wouldn't need this
 * routine if AIX did something more reasonable.  Hope thread safety
 * doesn't poke us in eye.
 */
char *
prettify (const char *name)
{
   static char bf[35];
   char *p;
   strncpy (bf, name, 29); bf[28] = 0;
   p = strchr (bf, '(');
   if (p) 
   {
      *(p+1) = ')';
      *(p+2) = 0x0;
   } 
   else
   {
      strcpy (&bf[26], "...()");
   }
   return bf;
}

#ifdef HAVE_VPRINTF
#define   VFPRINTF(file,format,args)   vfprintf((file), (format), (args))
#else
#ifdef HAVE_DOPRNT
#define VFPRINTF(file,format,args)   _doprnt((format), (args), (file))
#endif /* HAVE_DOPRNT */
#endif /* HAVE_VPRINTF */

FILE * report_file = NULL;
int lock_count = 0;
int lock_holder = -1;
pthread_mutex_t mutt = PTHREAD_MUTEX_INITIALIZER;

extern int get_tid(void);

void plck (int lock_incr)
{
   pthread_mutex_lock (&mutt);
   lock_count += lock_incr;
   lock_holder = get_tid();
}

void maybe_plck(void)
{
   /* if lock already held in this thread, no-op */
   if (get_tid() == lock_holder) return;

   pthread_mutex_lock (&mutt);
   lock_count ++;
   lock_holder = get_tid();
}

void maybe_unplck(void)
{
   lock_count --;
   if (0 == lock_count) 
   {
      lock_holder = -1;
      pthread_mutex_unlock (&mutt);
   }
}

/* print to stdout, report file and serialized logfile */
int
pfatal(char * fmt, ...)
{
    va_list args;

    maybe_plck();
    if (error_log_file) 
    {
        errlck ();
        va_start (args, fmt);
        VFPRINTF (error_log_file, fmt, args);
        va_end (args);
        errunlck ();
    }

    va_start (args, fmt);
    VFPRINTF (stdout, fmt, args);
    va_end (args);

    if (report_file) 
    {
        va_start (args, fmt);
        VFPRINTF (report_file, fmt, args);
        va_end (args);
    }
    maybe_unplck();
    return 0;
}

/* print to stdout and to report file */
int
perr(char * fmt, ...)
{
    va_list args;

    maybe_plck();
    va_start (args, fmt);
    VFPRINTF (stdout, fmt, args);
    va_end (args);

    if (report_file) 
    {
        va_start (args, fmt);
        VFPRINTF (report_file, fmt, args);
        va_end (args);
    }
    maybe_unplck();
    return 0;
}

/* print to stdout and to report file */
int
prt(char * fmt, ...)
{
    va_list args;

    maybe_plck();
    if (!quiet_stdout) 
    {
       va_start (args, fmt);
       VFPRINTF (stdout, fmt, args);
       va_end (args);
    }

    if (report_file) 
    {
        va_start (args, fmt);
        VFPRINTF (report_file, fmt, args);
        va_end (args);
    }
    maybe_unplck();
    return 0;
}


/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* ================================================================= */

FILE   *error_log_file = NULL;
int     error_log = -1;
int     error_log_lock_count = 0;

/* print to error log file */
void
errprt(char * fmt, ...)
{
    va_list args;

    if (error_log_file) 
    {
        errlck();
        va_start (args, fmt);
        VFPRINTF (error_log_file, fmt, args);
        va_end (args);
        errunlck();
    }
}

void
errlck (void)
{
    int rc;
#ifndef WIN32
    if (0 < error_log) 
    {
        error_log_lock_count ++;
        if (1 < error_log_lock_count) return;
        /* better lock the error log file in case other copies 
         * of webbot/webclient are running here */
        rc = flock(error_log,LOCK_EX);
        if (rc) 
        {
            int norr = errno;
            PERR("lockf(LOCK_EX) %d %s\n", norr, strerror(norr));
        }
    }
#endif /* WIN32 */
}

void
errunlck (void)
{
    int rc;
#ifndef WIN32
    if (0 < error_log) 
    {
        error_log_lock_count --;
        if (0 < error_log_lock_count) return;

        /* flush any buffered output before unlocking */
        fflush (error_log_file);
        rc = flock(error_log,LOCK_UN);
        if (rc) 
        {
            int norr = errno;
            PERR("lockf(LOCK_UN) %d %s\n", norr, strerror(norr));
        }
    }
#endif /* WIN32 */
}

/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* error message infrastructure */

/* global variables */
char    *webload_error_msg[ERROR_MESSAGE_ARRAY_SIZE];

void init_webload_errors (void) 
{
    /* ****************************************************************** */
    /* initialize the error message array. ...............................*/
    /* ****************************************************************** */
    webload_error_msg[0]                       ="Unknown error message encountered";
    webload_error_msg[-GET_CONNECT_ERROR]      ="Could not connect to server.";
    webload_error_msg[-GET_METHOD_ERROR]       ="Unknown method encountered.";
    webload_error_msg[-GET_SEND_ERROR]         ="Encountered a send error.";
    webload_error_msg[-GET_READ_ERROR]         ="Can't read data from webserver.";
    webload_error_msg[-GET_HEADER2_ERROR]      ="Encountered header error 2.";
    webload_error_msg[-GET_STATUS_ERROR]       ="Could not find status in HTTP header.";
    webload_error_msg[-GET_ALARM_CONNECT_ERROR]="Alarm expired while waiting for connect.";
    webload_error_msg[-GET_ALARM_ERROR]        ="Alarm expired while waiting for data";

    webload_error_msg[-GET_SSL_CONNECT_ERROR]  ="SSL socket initialization failed.  Check server certificate and cipher suite specification.";
    webload_error_msg[-GET_NETCLOSE_ERROR]     ="Internal error: NETCLOSE() failed.";
    webload_error_msg[-GET_EINTR_ERROR]        ="Internal error: EINTR loop trap.";
    webload_error_msg[-GET_HEADER_TRIES_ERROR] ="Page contains no data (header read retry loop exceeded)";
    webload_error_msg[-GET_SIGINT_ERROR]       ="Interrupted by SIGINT, will exit cleanly." ;
    webload_error_msg[-GET_SIGHUP_ERROR]       ="Interrupted by SIGHUP, will reload page." ;
    webload_error_msg[-GET_SIGFIN_ERROR]       ="Interrupted by SIGHUP, will quit page." ;
    webload_error_msg[-GET_EOF_ERROR]          ="Page contains no data (unexpected EOF while reading header)";
    webload_error_msg[-GET_THREAD_ERROR]       ="Internal error: no free threads";
    webload_error_msg[-GET_URL_ERROR]          ="Can't parse request URL";

    webload_error_msg[-GET_LOCATION_ERROR]     ="Could not find 'location' in a redirect header";
    webload_error_msg[-GET_LOCATION_ERROR2]    ="Could not parse new location in redirect";
    webload_error_msg[-GET_URL_LENGTH_ERROR]   ="Encountered url length error 1.";
    webload_error_msg[-GET_MISSING_COOKIE]     ="Want a cookie and didn't get it.";

}

char *webload_error(int error) 
{
   /* error numbers are negative numbers */
   if ((error < MIN_ERROR_MESSAGE) || (error > MAX_ERROR_MESSAGE)) {
      return(webload_error_msg[0]);
   } else {
      return(webload_error_msg[-error]);
   }
}

/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* debug-related stuff */

int     quiet_stdout = 0;
int     debug = 0;
int     tdebug= 0;

/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* alarm stuff, signal stuff */

volatile int alarm_expired = 0;
volatile int have_been_interrupted = 0;
volatile int retry_io = 0;
volatile int close_io = 0;
volatile int caught_sigpipe = 0;

#ifndef WIN32
/* ***************************************************************** */
/* signal_handler .................................................. */
/* ***************************************************************** */
#define REPORT_SIG(SIGGY)                                 \
   if (SIGGY == siggy)                                    \
   {                                                      \
      prt("\nWarning: signal_handler(): received %s will exit\n", #SIGGY);\
      have_been_interrupted ++;                           \
   } else                                                 \

#define RETRY_SIG(SIGGY)                                  \
   if (SIGGY == siggy)                                    \
   {                                                      \
      prt("\nInfo: signal_handler(): received %s, will retry I/O operation\n", #SIGGY);\
      retry_io ++;                                        \
   } else                                                 \

#define CLOSE_SIG(SIGGY)                                  \
   if (SIGGY == siggy)                                    \
   {                                                      \
      prt("\nInfo: signal_handler(): received %s, will close I/O socket\n", #SIGGY);\
      close_io ++;                                        \
   } else                                                 \

#define IGNORE_SIG(SIGGY)                                 \
   if (SIGGY == siggy)                                    \
   {                                                      \
      caught_sigpipe ++;                                  \
      if (100 < caught_sigpipe)                           \
      {                                                   \
         have_been_interrupted ++;                        \
         /* we're probably spinning, so block */          \
         sigignore (SIGGY);                               \
         prt("\nInfo: signal_handler(): received %s %d times, will exit\n", #SIGGY, caught_sigpipe);\
      }                                                   \
      else                                                \
      {                                                   \
         prt("\nWarning: signal_handler(): received %s %d times, will ignore\n", #SIGGY, caught_sigpipe);\
      }                                                   \
   } else                                                 \

void signal_handler(int siggy) 
{
   if (SIGALRM == siggy) 
   {
      prt("\nWarning: signal_handler(): received SIGALRM\n");
      alarm_expired ++;
   } else
   RETRY_SIG (SIGURG)
   CLOSE_SIG (SIGHUP)
   REPORT_SIG (SIGINT)
   REPORT_SIG (SIGQUIT)
   IGNORE_SIG (SIGPIPE)
   REPORT_SIG (SIGTERM)
   REPORT_SIG (SIGUSR1)
   REPORT_SIG (SIGUSR2)
   {
      prt("\nWarning: signal_handler(): received  unknown siggy %d\n", siggy);
   };
   return;
}
#else  /* WIN32 */
/* ***************************************************************** */
/* signal_handler for win32......................................... */
/* Unlike UNIX, signals do not interrupt win32 out of system calls.  */
/* Instead the signal_handler runs under another thread.  This means */
/* that have_been_interrupted flags will not be seen until the next  */
/* accept .......................................................... */
/* ***************************************************************** */
void signal_handler(int signal) {
   if (signal == SIGINT) {
      have_been_interrupted = 1;
   };
}
#endif /* WIN32 */

#ifdef WIN32
/* 
 * Because win32 signal and alarm handling is either broken or 
 * non-existant,  we'll hack up a bogus pretend timeout function, 
 * based on the select () system call.  This hurts performance, 
 * in a very nasty way, but oh well.
 */
int
expire_alarm (int fd, int numsecs) 
{
   struct fd_set fds;
   struct timeval timeout;
   int retval;

   /* don't bother messing with this unless we have alarms enabled */
   if (0 >= numsecs) return 0;

   timeout.tv_sec = numsecs;
   timeout.tv_usec = 0;

   PDBG2("going to wait in select for %d secs\n", numsecs);

   if (FD_SETSIZE <= fd) {
      perr ("Error: expire_alarm(): "
                "socket descriptor too large: "
                "was %d max %d \n", fd, FD_SETSIZE);
      errno = EBADF;
      return -1;
    }
   FD_ZERO (&fds);
   FD_SET (fd, &fds);
   retval = select (FD_SETSIZE, &fds, NULL, NULL, &timeout);

    if (0 > retval) {
      return -1;
   } else 
   if (0 == retval) {
      PDBG ("timed out on select waiting for data\n");
        /* pretend that we were interrupted by a timer */
      errno = EINTR;
      alarm_expired = 1;
      return -1;
   } 
   return 0;

}
#endif /* WIN32 */

/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* ================================================================= */

#ifdef WIN32
/* ********************************************************* */
/* find out if the cpu we are running on has the necessary   */
/* features to run this program.  We require a time stamp    */
/* counter to work properly (pentium or higher). ........... */
/* ********************************************************* */

int is_good_cpu(void) 
{
   int hv, eax1, edx1, cpu, time_stamp;

#ifdef COMPILER_SUPPORTS_INLINE_ASM
   /* execute the CPUID instruction to find out if we have  */
   /* what is needed.  Note: this instruction should trap   */
   /* or return undefined stuff on non-pentium processor    */
   __asm
   {
      mov eax, 0       
     ;CPUID
     _emit 0x0F
     _emit 0xA2
     mov hv,eax  
     mov eax, 1
     ;CPUID
     _emit 0x0F
     _emit 0xA2
     mov eax1, eax
     mov edx1, edx
   }
   /* first, we test what comes back for CPUID with EAX=0....*/
   /* we expect the hv value returned in EAX to the first of */
   /* the CPUID instructions above to be at least 1 (pentium)*/
   /* or 2 (pentium pro) ....................................*/
   if (hv < 1) return(0);
   /* the following test results of CPUID with EAX=1. .......*/
   /* we expect the processor type in EAX[11:8] to be 5 or   */
   /* more (5=Pentium, 6=Pentium Pro. ...................... */
   cpu = (eax1 & 0xF00) >> 8;
   if (cpu < 5) return(0);
   /* we expect the time stamp counter to be suported.  This */
   /* flag is returned in EDX[4:4]. ........................ */
   time_stamp = (edx1 & 0x10) >> 4;
   if (time_stamp != 1) return(0);
   return(1);
#else /* COMPILER_SUPPORTS_INLINE_ASM */
   return (1);
#endif /* COMPILER_SUPPORTS_INLINE_ASM */
}
#endif /* WIN32 */


/* ================================================================= */
/* ================================================================= */
/* ================================================================= */
/* ================================================================= */

