//
// FILE:
// cookie.h
//
// FUNCTION:
// This file defines a couple of handy utility container classes.
// I chose to create this rather than using e.g. STL or ICLUI because
// STL (standard template library) is syntactically obtuse, and 
// I wanted a novice programmer to be able to understand and 
// maintain this code.  I avoided ICLUI because of its proprietary
// nature and it's lack of widespread availability on multiple
// platforms.  I'm not aware of any other commonly available 
// collection classes; at any rate, it seems like a good idea to
// avoid pre-req'ing any package that is rare, hard to obtain, hard 
// to build/install, poorly supported. etc.
//
// These classes are meant to be dirt-simple in order to avoid
// complexity problems with memory allocation & fragmentation, 
// copy-constructors, etc. and to avoid the performance penalties
// of more complex schemes. 
// 
// HISTORY:
// Linas Vepstas Summer 1999
//
#ifndef __WL_COOKIE_H__
#define __WL_COOKIE_H__

#include "header.h"
#include "sysdep.h"
#include "super.h"


/* ============================================= */
//
// The wlCache class implements a very simple string
// cache.  That is, it is a container (an unsorted bag)
// of strings.  It is used primarily to emulate a browser's
// image cache; it also stores cookie paths.  
//
// The AddToCache() method adds (a copy of) the string
//    to the cache, only if the string is not already in 
//    the cache.  It returns a 1 if its added, else it returns 
//    a zero.
// The PutInCache() method adds the string to the cache.
//    The arg must point to malloc memory.  If the string
//    is already in the cache, it frees the string and returns 0.
//    Otherwise it returns 1.  Note that the argument  passed in
//    will be freed when this class is destroyed.
// The IsInCache() method returns TRUE if an identical
//    string is in the cache.
// The ClearCache() method empties the cache.
// The NumEntries() returns the number of elements.
// The GetEntry() method returns the i'th entry on the 
//    cache. Useful for printing the contents.
// The StartsWith() method returns a pointer to an element
//    in the cache if that element starts with the indicated 
//    string.
// The OccursIn() method returns a pointer to an element
//    in the cache if that element occurs as a substring
//    in the argument 'haystack'.

class wlCache 
{
   public:
      wlCache (void);
      ~wlCache ();
      int IsInCache (const char *);
      char * StartsWith (const char *);
      char * OccursIn (const char *haystack);
      void ClearCache (void);
      int AddToCache (const char *);
      int AddToCache (const char *, size_t len);
      int PutInCache (char *);
      int NumEntries (void);
      char * GetEntry (int);

   private:
      unsigned long cache_size;
      unsigned long cache_entries;
      char **cache;

};

/* ============================================= */
//
// The GetCookie() method returns a pointer to a 
//   (quasi-statically-allocated) string containing cookie 
//   key-value pairs for the indicated path.  The string is 
//   HTTP/1.1 spec format: that is, it is in the form
//   key1=value1;key2=value2;\n
//
//   Note that this storage is associated with this class instance,
//   so take care when using in a multi-threaded environment.
//
// The Print() method does a lowbrow debug print of the jar contents.
//
// The AddCookie() method adds a cookie to the cookie jar.
//    If a cookie by that name already exists in the jar, 
//    its value will be replaced by new value.
//    It has three signatures:
//    AddCookie (char *, char *, char *): path, key, value.
//    AddCookie (char *path, wlString& list): the list should be
//       a semicolon-separated string of cookie names and values.
//       If 'path=' appears in the list, it will override the arg.
//    AddCookie (wlString& list): the list should be a semicolon-
//       separated string of cookie names and values. The string
//       must include a 'path=' compnenet or else the cookie won't 
//       be added.
//    AddCookie(wlHeader &, char * url): The header is searched for 
//       Set-Cookie: directives and those cokies are put into the jar.
//       If the Set-Cookie: directive did not indicate the path,
//       then the argument 'url' is used for the path. 'url' may be 
//       NULL, in which case if there is no path directive, the 
//       cookies aren't added.
//
// Note that this cookie-jar does not handle 'expires' (there is no 
// need for this in this application).  Note that as of now, it doesn't
// implement domain seraches.  It does not implement the correct use of 
// 'secure'.


class wlCookieJar
{
   public:
      wlCookieJar (void);
      ~wlCookieJar ();
      void ClearCache (void);
      void AddCookie (const char *path, const char *key, const char * val);
      void AddCookie (const char *path, wlCookie &cookie);
      void AddCookie (wlCookie &cookie);
      void AddCookie (wlHeader &, const char *path);
      wlCookie & GetCookie (const char * path);


   private:
      void Enlarge(void);
      void Print (void);

      unsigned long jar_size;
      unsigned long num_entries;
      wlString **domains;
      wlString **paths;
      wlCookie **cookies;
      time_t   *expires;
      char     *secures;
      wlCookie last;
};

#endif /* __WL_COOKIE_H__ */

