//
// FILE:
// cookie.C
//
// FUNCTION:
// See the header file for documentation
//
// HISTORY:
// Linas Vepstas Summer-Fall 1999

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cookie.h"
#include "generic.h"
#include "sysdep.h"


#define CACHE_GROW_SIZE 32
/* ================================================================ */

wlCookieJar :: wlCookieJar (void)
{
   jar_size = 0;
   num_entries = 0;
   cookies = 0x0;
   domains = 0;
   expires = 0;
   secures = 0;
   paths = 0x0;
   Enlarge();
}

wlCookieJar :: ~wlCookieJar ()
{
   ClearCache();
   if (paths) free (paths); paths = 0x0;
   if (domains) free (domains); domains = 0x0;
   if (expires) free (expires); expires = 0x0;
   if (secures) free (secures); secures = 0x0;
   if (cookies) free (cookies); cookies = 0x0;
   jar_size = 0;
}

void
wlCookieJar :: ClearCache (void)
{
   for (int i=0; i<num_entries; i++)
   {
      if (paths[i]) delete paths[i]; paths[i] = 0x0;
      if (domains[i]) delete domains[i]; domains[i] = 0x0;
      if (cookies[i]) delete cookies[i]; cookies[i] = 0x0;
      if (expires[i]) expires[i] = 0x0;
      if (secures[i]) secures[i] = 0x0;
   }
   num_entries = 0;
}

void
wlCookieJar :: AddCookie (const char * path, 
                          const char * key, 
                          const char * value)
{
   if (!path || !key || !value) return;

   // search for the path in the cookie jar
   int i;
   for (i=0; i<num_entries; i++)
   {
      if (0 == strcmp (path, (char *) *(paths[i]))) break;
   }

   // path not found, enlarge the cookie jar
   if (i == num_entries) 
   {
      Enlarge();
      paths[i] = new wlString (path);
      cookies[i] = new wlCookie;
      num_entries ++;
   }
   
   // Add the cookie if its absent, replace the field val if present
   cookies[i] -> ReplaceOrAddField (key, value);
}

void
wlCookieJar :: AddCookie (const char * pith, wlCookie &ookie)
{
   char * path = ookie.GetValue ("path");
   if (!path) path = (char *) pith;
   if (!path) return;

   // make sure the path doesn't have any trailing garbage
   char * fixup = 0x0;
   fixup = strchr (path, ';');
   if (fixup) *fixup = 0;

   // search for the path in the cookie jar
   int i;
   for (i=0; i<num_entries; i++)
   {
      if (0 == strcmp (path, (char *) *(paths[i]))) break;
   }

   // path not found, enlarge the cookie jar
   if (i == num_entries) 
   {
      Enlarge();
      paths[i] = new wlString (path);
      cookies[i] = new wlCookie;
      num_entries ++;
   }

   // yank info out of the cookie
   if (ookie.GetValue ("secure")) secures[i] = 1;

   // Add the cookie
   cookies[i] -> ReplaceOrAddFields (ookie);

   // remove the 'path=whatever' if it is present
   cookies[i] -> Remove ("path");
   cookies[i] -> Remove ("domain");
   cookies[i] -> Remove ("expires");
   cookies[i] -> Remove ("secure");

   if (fixup) *fixup = ';';
}

void
wlCookieJar :: AddCookie (wlCookie &ookie)
{
   AddCookie (NULL, ookie);
}

void
wlCookieJar :: Enlarge (void)
{
   // we enlarge the jar only if we cannot fit one more cookie into it.
   if (jar_size > num_entries +1) return;

   int new_jar_size = jar_size + CACHE_GROW_SIZE;
   paths = (wlString **) realloc (paths, new_jar_size * sizeof (wlString *));
   domains = (wlString **) realloc (domains, new_jar_size * sizeof (wlString *));
   cookies = (wlCookie **) realloc (cookies, new_jar_size * sizeof (wlCookie *));
   expires = (time_t *) realloc (expires, new_jar_size * sizeof (time_t));
   secures = (char *) realloc (secures, new_jar_size * sizeof (char));
   for (int i=jar_size; i<new_jar_size; i++) 
   {
      paths[i] = 0x0;
      domains[i] = 0x0;
      cookies[i] = 0x0;
      expires[i] = 0;
      secures[i] = 0;
   }
   jar_size = new_jar_size;
}

/* ================================================================ */

void
wlCookieJar :: Print (void)
{
   PDBG("==========cookie jar contents===========\n");
   if (num_entries) 
   {
      PDBG("path count=%d\n", num_entries);
      for(int i=0; i<num_entries; i++) 
      {
         PDBG("domain[%d]=%s path[%d]=%s\n", i, (char *) *(domains[i]), i, (char *) *(paths[i]));
         PDBG("\t%s\n", (char *) *(cookies[i]));
      }
   } else {
      PDBG("Cookie jar is empty\n");
   }
   PDBG("=========bottom of cookie jar==========\n");
}

/* ================================================================ */

wlCookie &
wlCookieJar :: GetCookie (const char * url)
{
   PDBG2 ("url=%s\n",url);
   if (debug > 1) Print();

   last = "";

   // find *all* paths that are roots of the current url,
   // and add thier cookies to the list.
   for (int i=0; i<num_entries; i++) {
      if (0 == strncmp (url, (char *) *(paths[i]), paths[i]->Memlen()))
      {
         last.ReplaceOrAddFields (*(cookies[i]));
      }
   }
   return last;
}

/* ================================================================ */

void 
wlCookieJar :: AddCookie (wlHeader &header, const char *url)
{
   // search the header for 'Set-Cookie: tokens
   char *cursor = (char *) header;
   cursor = strcasestr (cursor, SET_COOKIE);
   if (debug > 1) Print();
   while (cursor) 
   {
      // Skip over the "Set-cookie:" string and any whitespace
      cursor += strlen(SET_COOKIE);
      cursor += strspn (cursor, " \t");

      // To avoid confusion when there are multiple Set-cookie
      // tokens in the header, we split out this line here,
      // and copy it into tmp working memory
      char * end = cursor + strcspn (cursor, "\r\n");

      wlCookie ookie;
      ookie.Memcpy (cursor, end-cursor);
      AddCookie (url, ookie);


      if (debug > 1) Print();
      cursor = end + 1;
      cursor = strcasestr (cursor, SET_COOKIE);
   }
}

/* ================================================================ */
/* ================================================================ */
/* ================================================================ */


wlCache :: wlCache (void)
{
   cache_entries = 0;
   cache_size = CACHE_GROW_SIZE;
   cache = (char **) malloc (CACHE_GROW_SIZE * sizeof (char *));
}

wlCache :: ~wlCache ()
{
   ClearCache();
   cache_size = 0;
   free(cache);
   cache = 0;
}

int 
wlCache :: IsInCache (const char *image_name) 
{
   int i;
   if (!image_name) return (1);     /* null images are always cached! */
   for (i=0; i<cache_entries; i++) 
   {
      if (0 == strcmp(image_name, cache[i]))
      {
         return(1);
      }
   }
   return(0);
}

char *
wlCache :: GetEntry (int i)
{
   if (0>i || i>=cache_entries) return 0x0;
   return cache[i];
}

char *
wlCache :: StartsWith (const char *path_frag) 
{
   int i, len;
   if (!path_frag) return (0x0);
   len = strlen (path_frag);
   for (i=0; i<cache_entries; i++) 
   {
      if (0 == strncmp(path_frag, cache[i], len))
      {
         return(cache[i]);
      }
   }
   return(0x0);
}

char *
wlCache :: OccursIn (const char *haystack) 
{
   int i;
   if (!haystack) return (0x0);
   for (i=0; i<cache_entries; i++) 
   {
      if (strstr(haystack, cache[i]))
      {
         return(cache[i]);
      }
   }
   return(0x0);
}

void 
wlCache :: ClearCache (void)
{
   int i;
   for (i=0; i<cache_entries; i++) 
   {
      free (cache[i]);
      cache[i] = NULL;
   }
   cache_entries = 0;
}

int
wlCache :: AddToCache (const char *image_name) 
{
   if (!image_name) return 0;

   if (IsInCache (image_name)) return 0;
   if (cache_entries >= cache_size) {
      cache_size += CACHE_GROW_SIZE;
      cache = (char **) realloc (cache, cache_size * sizeof (char *));
   }

   cache[cache_entries] = strdup (image_name);
   cache_entries ++;
   return 1;
}

int
wlCache :: AddToCache (const char *image_name, size_t len) 
{
   if (!image_name) return 0;

   char *term = (char *) malloc (len+1);
   strncpy (term, image_name, len);
   term[len] = 0;

   if (IsInCache (term)) {free (term); return 0; }
   if (cache_entries >= cache_size) {
      cache_size += CACHE_GROW_SIZE;
      cache = (char **) realloc (cache, cache_size * sizeof (char *));
   }

   cache[cache_entries] = term;
   cache_entries ++;
   return 1;
}

int
wlCache :: PutInCache (char *image_name) 
{
   if (!image_name) return 0;

   if (IsInCache (image_name)) {
      free (image_name);
      return 0;
   }

   if (cache_entries >= cache_size) {
      cache_size += CACHE_GROW_SIZE;
      cache = (char **) realloc (cache, cache_size * sizeof (char *));
   }

   cache[cache_entries] = image_name;
   cache_entries ++;
   return 1;
}

int
wlCache :: NumEntries (void)
{
   return cache_entries;
}

/* ======================== END OF FILE ======================= */
