# Mapster: a graphical tool for creating client-side imagemaps for Web pages
# Copyright (c) 1998-1999  Matthew C. Gushee
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.

#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.

#  You should have received a copy of the GNU General Public License
#  along with this program; see the file COPYING.  If not, write to
#  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
#  Boston, MA 02111-1307, USA.

# please send comments, praise, bug reports to mgushee@havenrock.com

package provide Mapster 0.2
set mapster_version 0.2
set config(mapster) [file dirname [info script]]
lappend auto_path [file join $config(mapster)]

proc init_unix {} {
    global config mapInfo 
    # maxw maxh

    # directory for icons and so on
    set config(images) [file join $config(mapster) images]

    # command to convert files to GIF format
    # '%i' and '%o' are placeholders for the input and output
    # filenames. You can add options or arguments at any point
    # in the command string, as long as %i and %o are in the
    # right places.
    set config(converter) "convert %i %o"

    # location for saving image map files and temporary GIFs
    # make sure you have write permission for this directory
    set config(tempdir) "/tmp"

    # set this to 0 if you don't want to be prompted for a name every time
    # you create an area on the map
    set config(aname) 1

    # colors
    set config(light) white
    set config(light-act) orange
    set config(light-bg) white
    set config(dark) black
    set config(dark-act) blue
    set config(dark-bg) black
    set config(colordefault) dark
    set mapInfo(colordefault) $config(colordefault)

    # For that slim look. If you prefer the default clunky
    # appearance, just delete these lines
    option add *Button.borderWidth 1 startupFile
    option add *Entry.borderWidth 1 startupFile
    option add *Entry.background white startupFile
    option add *Scrollbar.borderWidth 1 startupFile
    option add *Scrollbar.width 13 startupFile
    option add *Menu.borderWidth 1 startupFile
    option add *MenuButton.borderWidth 1 startupFile

    # don't mess with this
    if {[catch {package require Img}] == 0} {
	set config(useimg) 1
    } else {
	set config(useimg) 0
    }

    foreach item {mapInfo(defalt) mapInfo(defactn) \
		      mapInfo(htmlname) mapInfo(output)} {
	set $item ""
    }

    # set maxw [winfo screenwidth .]
    # set maxh [winfo screenheight .]
}

proc init_windows {} {
    error "Sorry. Mapster doesn't run\non Windows yet. It will someday\n\
---maybe soon if you help."
}

proc init_mac {} {
    error "Sorry. Mapster doesn't run\non MacOS yet. It will someday\n\
---maybe soon if you help."
}


proc parse_args {argv} {
    global mapInfo

    set imgfile ""

    for {set i 0} {$i < [llength $argv]} {incr i} {

	set item [lindex $argv $i]
	set next [lindex $argv [expr {$i + 1}]]

	if {[string match "-*" $item]} {
	    switch -- [string index $item 1] {
		"A" {
		    set mapInfo(defalt) [arg_value $item $next]
		}
		"D" {
		    set mapInfo(defactn) [arg_value $item $next]
		}
		"N" {
		    set mapInfo(htmlname) [arg_value $item $next]
		}
		"O" {
		    set mapInfo(output) [arg_value $item $next]
		}
		default {
		    puts "\nInvalid options detected!\n\nUSAGE\
:\n-d default URL\n-n HTML name\n-o output file\n"
		    exit
		}
	    }
	} else {
	    set imgfile $item
	}

    }
    
    return $imgfile

}


proc arg_value {arg1 arg2} {

    if {[set result [string range $arg1 2 end]] == ""} {
 	if {[string match "-*" $arg2] || [string length $arg2] == 0} {
	    puts "Error: a value must be supplied for the $arg1 option."
	    exit
 	} else {
 	    set result $arg2
	    uplevel {incr i}
 	}
    }

    return $result

}


proc main {imgfile} {
    global config toolInfo mapInfo

    # create images for toolbar buttons
    image create bitmap rect -file [file join $config(images) "rectangl.xbm"]
    image create bitmap circ -file [file join $config(images) "circle.xbm"]
    image create bitmap poly -file [file join $config(images) "polygon.xbm"]
    image create bitmap mvobj -file [file join $config(images) "mvobj.xbm"]
    image create bitmap iload -file [file join $config(images) "load.xbm"]
    image create bitmap msave -file [file join $config(images) "save.xbm"]
    image create bitmap adelete -file [file join $config(images) "delete.xbm"]
    image create bitmap posneg -file [file join $config(images) "posneg.xbm"]
    image create bitmap qexit -file [file join $config(images) "exit.xbm"]

    ## set up the toolbar ====================
    # first the toolbar frame and the selectable tools
    # add the tag 'ToolButton' to each button
    pack [frame .tbar] -side left -fill y -padx 1 -pady 3
    foreach tool {rect circ poly adelete mvobj posneg} {
	pack [label .tbar.$tool -image $tool -relief raised -borderwidth 1] \
	    -side top -padx 1 -pady 1
	bindtags .tbar.$tool {ToolButton [bindtags .tbar.$tool]}
    }

    # a separator
    pack [frame .tbar.sep -height 12] -side top

    # buttons that perform operations
    pack [button .tbar.iload -image iload -command \
	      {gui_image_load .disp.c} -highlightthickness 0] \
	-side top -padx 1 -pady 1
    pack [button .tbar.msave -image msave -command \
	      {mapfile_save .disp.c} -highlightthickness 0] \
	-side top -padx 1 -pady 1
    pack [button .tbar.qexit -image qexit -command \
	      {query_exit .disp.c} -highlightthickness 0] \
	-side top -padx 1 -pady 1

    # set up the display area
    pack [frame .disp] -side left -expand yes -fill both
    canvas .disp.c -background black -yscrollcommand {.disp.ybar set} \
	-xscrollcommand {.disp.xbar set}
    scrollbar .disp.ybar -orient vertical -command {.disp.c yview}
    scrollbar .disp.xbar -orient horizontal -command {.disp.c xview}
    grid .disp.c -row 0 -column 0 -sticky nsew
    grid .disp.ybar -row 0 -column 1 -sticky ns
    grid .disp.xbar -row 1 -column 0 -sticky ew
    grid columnconfigure .disp 0 -weight 1
    grid rowconfigure .disp 0 -weight 1

    # set up mouse bindings
    bind ToolButton <Enter> {%W configure -background "#ececec"}
    bind ToolButton <Leave> {%W configure -background "#d9d9d9"}
    bind ToolButton <ButtonPress-1> {tool_select .disp.c %W}
    .disp.c bind mapobj <Enter> {obj_activate .disp.c}
    .disp.c bind mapobj <Leave> {obj_deactivate .disp.c}

    # and some key bindings
    bind . <KeyPress-r> {tool_select .disp.c .tbar.rect}
    bind . <KeyPress-c> {tool_select .disp.c .tbar.circ}
    bind . <KeyPress-p> {tool_select .disp.c .tbar.poly}
    bind . <KeyPress-x> {tool_select .disp.c .tbar.adelete}
    bind . <KeyPress-m> {tool_select .disp.c .tbar.mvobj}
    bind . <KeyPress-t> {tool_select .disp.c .tbar.posneg}
    bind . <Control-l> {gui_image_load .disp.c}
    bind . <Control-s> {mapfile_save .disp.c}
    bind . <Control-q> {query_exit .disp.c}

    # rectangle tool is initially selected
    tool_select .disp.c .tbar.rect

    set mapInfo(modified) 0

    if {$imgfile != ""} {
	image_load $imgfile
    }
}

proc tool_select {window tool} {
    global toolInfo

    if {[info exists toolInfo(selected)]} {
	if {$toolInfo(selected) == $tool} {
	    return
	} else {
	    $toolInfo(selected) configure -relief raised
	}
    }

    set toolInfo(selected) "$tool"
    $tool configure -relief sunken

    # clear all existing bindings before activating new ones
    # this probably reduces performance just a hair but 
    # simplifies the code significantly
    foreach item {mapimage mapobj} {
	foreach event {<ButtonPress-1> <B1-Motion>\
			   <ButtonRelease-1> <ButtonPress-3>} {
	    $window bind $item $event {}
	}
    }
    $window bind newobj <Double-Button-1> {}
    bind .disp.c <ButtonPress-1> {}

    switch -- $tool {
	.tbar.rect {
	    $window bind mapimage <ButtonPress-1> {rect_start %x %y}
	    $window bind mapimage <B1-Motion> {rect_draw .disp.c %x %y}
	    $window bind mapimage <ButtonRelease-1> {rect_set .disp.c %x %y}
	}
	.tbar.circ {
	    $window bind mapimage <ButtonPress-1> {circle_start %x %y}
	    $window bind mapimage <B1-Motion> {circle_draw .disp.c %x %y}
	    $window bind mapimage <ButtonRelease-1> {circle_set .disp.c %x %y}
	}
	.tbar.poly {
	    $window bind mapimage <ButtonPress-1> \
		{polygon_start_or_draw .disp.c %x %y}
	    $window bind mapimage <B1-Motion> \
		{polygon_draw .disp.c %x %y}
	    $window bind mapimage <ButtonRelease-1> \
		{polygon_set_point .disp.c %x %y}
	    $window bind mapimage <ButtonPress-3> \
		{polygon_cancel_point .disp.c}
	    $window bind mapimage <Double-ButtonPress-1> \
		{polygon_set .disp.c}
	    $window bind newobj <Double-ButtonPress-1> {polygon_set .disp.c}
	}
	.tbar.adelete {
	    $window bind mapobj <ButtonPress-1> {obj_delete .disp.c}
	}
	.tbar.mvobj {
	    $window bind mapobj <ButtonPress-1> {obj_move_start %x %y}
	    $window bind mapobj <B1-Motion> {obj_move .disp.c %x %y}
	    $window bind mapobj <ButtonRelease-1> {obj_move_end}
	}
	.tbar.posneg {
	    bind .disp.c <ButtonPress-1> {pos_neg_toggle .disp.c}
	}
	default {
	    # do nothing
	}
    }
}

##################################################
####  FILE  HANDLING  ###########################
##################################################

proc gui_image_load {win} {
    global mapInfo

    if {$mapInfo(modified)} {
	set whattodo \
	    [tk_dialog .conf "Confirmation" "Save changes\n\
before loading new image?" "question" 0 "Save" "Discard" "Cancel"]
	switch -- $whattodo {
	    0 {
		mapfile_save $win
	    }
	    1 {}
	    2 {
		return
	    }
	}
	
    }


    set imgfile [tk_getOpenFile -filetypes \
		     {"GIF {.gif}" "JPEG {.jpg .jpeg .jfif}" \
			  "PNG {.png}" "TIFF {.tif .tiff}"}]
    if {$imgfile != ""} {
	image_load $imgfile
    }
}

proc image_load {imgfile} {
    global mapInfo config 

    set basename [file rootname [file tail $imgfile]]
    set mapInfo(basename) $basename
    if {![string match *gif $imgfile] && \
	    ![string match *GIF $imgfile] && \
	    !$config(useimg)} {
	set newfile [file join $config(tempdir) "$basename.gif"]
	regsub %i $config(converter) $imgfile command
	regsub %o $command $newfile command
	eval exec $command
	set imgfile $newfile
    }

    .disp.c delete mapimage
    # maybe this should be put into a variable and done generically
    image create photo MapImage -file $imgfile
    set dispwidth [expr {[image width MapImage] + 8}]
    set mapheight [image height MapImage]
    set dispheight [expr {[winfo height .tbar] - 10}]
    if {$mapheight >= [expr {$dispheight - 8}]} {
	set dispheight [expr {$mapheight + 8}]
    }
    set xctr [expr {$dispwidth/2}]
    set yctr [expr {$dispheight/2}]

    .disp.c configure -width $dispwidth -height $dispheight

    .disp.c create image $xctr $yctr -anchor center \
	-image MapImage -tags mapimage

    set mapInfo(imgcoords) [.disp.c bbox mapimage]

    if {$mapInfo(htmlname) == "" || $mapInfo(defactn) == "" \
	    || $mapInfo(defalt) == ""} {
	get_mapname
    }

    set mapInfo(modified) 0

    # Just in case the image is bigger than the screen, the following
    # code keeps the main window within bounds. The -16 and -32 are to 
    # allow space for the window's title bar & border, since there
    # doesn't seem to be any way to calculate them automatically.
    set maxw [expr {[winfo screenwidth .] - 16}]
    set maxh [expr {[winfo screenheight .] - 32}]

    set w [winfo reqwidth .]
    set h [winfo reqheight .]
    set resize 0

    if {$w > $maxw} {
	set w $maxw
	set resize 1
    }
    if {$h > $maxh} {
	set h $maxh
	set resize 1
    }
    if {$resize} {
	set geom $w
	append geom "x" $h
	wm geometry . $geom
    }

    update idletasks
    .disp.c configure -scrollregion [.disp.c bbox all]
}

proc mapfile_save {win} {
    global mapInfo config mapster_version

    if {![info exists mapInfo(filename)]} {
	set mapInfo(filename) "$mapInfo(basename).map"
    }

    set filename \
	[tk_getSaveFile -defaultextension .map -filetypes \
	     {{"Image Maps" {.map}} {"All Files" {*}}} -initialdir \
	     $config(tempdir) -initialfile $mapInfo(filename)]

    if {$filename == ""} {
	return
    }

    set mapInfo(filename) $filename
    mapdata_write $win $filename
}

proc mapdata_write {win {output ""}} {
    global config mapInfo mapster_version

    if {[info exists mapInfo(objects)]} {
	if {$output == ""} {
	    set output $mapInfo(output)
	}
	
	set outputlist \
	    [list "<!-- Image map generated by mapster v$mapster_version -->" \
		 "<map name=\"$mapInfo(htmlname)\">"]
	
	foreach objID $mapInfo(objects) {
	    set type [$win type $objID]
	    switch -- $type {
		rectangle {
		    set type rect
		}
		polygon {
		    set type poly
		}
		oval {
		    set type circle
		}
	    }
	    set coords ""
	    set xdelta [lindex $mapInfo(imgcoords) 0]
	    set ydelta [lindex $mapInfo(imgcoords) 1]
	    set bounds [$win coords $objID]
	    if {$type == "circle"} {
		set x [expr {round(([lindex $bounds 0] + \
					[lindex $bounds 2]) / 2) - $xdelta}]
		set y [expr {round(([lindex $bounds 1] + \
					[lindex $bounds 3]) / 2) - $ydelta}]
		set r [expr {round([lindex $bounds 2] - $x)}]
		set coords "$x,$y,$r"
	    } else {
		foreach {xval yval} $bounds {
		    if {$coords != ""} {
			append coords ","
		    }
		    append coords [expr round($xval) - $xdelta]
		    append coords ","
		    append coords [expr round($yval) - $ydelta]
		}
	    }
	    set alttext $mapInfo(aname-$objID)
	    lappend outputlist \
		"<area shape=\"$type\" coords=\"$coords\" alt=\"$alttext\">"
	}
	
	set deftext "<area shape=\"default\" alt=\"$mapInfo(defalt)\" "
	if {$mapInfo(defactn) == "nohref"} {
	    append deftext "nohref>"
	} else {
	    append deftext "href=\"$mapInfo(defactn)\">"
	}
	lappend outputlist $deftext
	lappend outputlist "</map>"
	
	if {$output == "-"} {
	    set channel stdout
	} else {
	    if {[catch {set channel [open $output "w"]}] != 0} {
		error "Unable to save file!\nYou should make sure you have write permission\nfor $config(tempdir). If that's not the problem,\nyou may have found a bug."
	    }
	}
	foreach line $outputlist {
	    puts $channel $line
	}
	if {$channel != "stdout"} {
	    close $channel
	}
	set mapInfo(modified) 0
    }
}


proc pos_neg_toggle {win} {
    global config mapInfo
    set objID [$win find withtag current]
    set taglist [$win gettags $objID]
    if {[lsearch $taglist mapobj] >= 0} {
	if {$mapInfo(color-$objID) == "dark"} {
	    set mapInfo(color-$objID) "light"
	    $win itemconfigure current -fill $config(light-act) \
		-outline $config(light-act)
	} else {
	    set mapInfo(color-$objID) "dark"
	    $win itemconfigure current -fill $config(dark-act) \
		-outline $config(dark-act)
	}
    } else {
	if {$mapInfo(colordefault) == "dark"} {
	    set mapInfo(colordefault) "light"
	    $win configure -background $config(light-bg)
	} else {
	    set mapInfo(colordefault) "dark"
	    $win configure -background $config(dark-bg)
	}
    }
}
	    

proc area_delete {} {
    global mapInfo
    set mapInfo(modified) 1
}

proc query_exit {win} {
    global mapInfo

    if {[info exists mapInfo(modified)] && $mapInfo(modified)} {
	if {$mapInfo(output) != ""} {
	    mapdata_write $win
	} else {
	    set whattodo \
		[tk_dialog .conf "Confirmation" "Save changes\n\
before exiting?" "question" 0 "Save" "Exit" "Cancel"]
	    switch -- $whattodo {

		"0" {
		    mapfile_save $win
		}

		"1" {}

		"2" {
		    return
		}
	    }
	}
    }
    exit
}


#################################################
###  DIALOG BOXES AND SUCHLIKE  #####################
#################################################

proc get_mapname {} {
    global mapInfo

    set done 0

    toplevel .mn
    wm withdraw .mn

    pack [frame .mn.name -relief groove -borderwidth 2] -padx 8 -pady 4 -fill x
    pack [label .mn.name.text -text "HTML name for this image map:"] -pady 4
    pack [entry .mn.name.ent] -pady 4

    pack [frame .mn.defreg -relief groove -borderwidth 2] \
	-padx 8 -pady 4 -fill x
    pack [label .mn.defreg.text1 -text "Default action:"] -pady 4
    pack [frame .mn.defreg.rad] -fill x -padx 4
    pack [radiobutton .mn.defreg.rad.b1 -variable choice -text \
	      "href (enter below)" -value "href" -command \
	      {focus .mn.defreg.ent1}]\
	-side left -padx 3 -pady 4
    pack [radiobutton .mn.defreg.rad.b2 -variable choice -text \
	      "nohref" -value "nohref" -command \
	      {focus .mn.defreg.rad.b2}] \
	-side right -padx 3 -pady 4
    pack [entry .mn.defreg.ent1]\
	-fill x -padx 4 -pady 4
    pack [label .mn.defreg.text2 -text "ALT text for default region:"] \
	-pady 4
    pack [entry .mn.defreg.ent2] -pady 4

    .mn.name.ent insert end $mapInfo(htmlname)
    if {$mapInfo(defactn) == "" || $mapInfo(defactn) == "nohref"} {
	.mn.defreg.rad.b2 select
    } else {
	.mn.defreg.ent1 insert end $mapInfo(defactn)
	.mn.defreg.rad.b1 select
    }
    .mn.defreg.ent2 insert end $mapInfo(defalt)

    pack [frame .mn.controls]
    pack [button .mn.controls.ok -text "OK" -command \
	      {set mapInfo(htmlname) [.mn.name.ent get]
		  if {$choice == "nohref"} {
		      set mapInfo(defactn) "nohref"
		  } else {
		      set mapInfo(defactn) [.mn.defreg.ent1 get]
		  }
		  set mapInfo(defalt) [.mn.defreg.ent2 get]
		  set done 1}] -side left -padx 12 -pady 4

     pack [button .mn.controls.cancel -text "Cancel" -command \
 	      {set done 1}] -side right -padx 12 -pady 4

    bind .mn <KeyPress-Return> {.mn.controls.ok invoke}
    bind .mn <KeyPress-Escape> {.mn.controls.cancel invoke}
    bind .mn.defreg.rad.b1 <KeyPress-Right> {.mn.defreg.rad.b2 invoke}
    bind .mn.defreg.rad.b2 <KeyPress-Left> {.mn.defreg.rad.b1 invoke}
    bind .mn.defreg.ent1 <KeyPress-Right> {.mn.defreg.rad.b2 invoke}
    foreach entry {.mn.name.ent .mn.defreg.ent1 .mn.defreg.ent2} {
	bind $entry <FocusIn> [list $entry select range 0 end]
	bind $entry <FocusOut> [list $entry selection clear]
    }

    wm title .mn "map info"
    bind .mn <ButtonPress> {
	wm deiconify .mn
	raise .mn
	focus .mn.name.ent
    }

    wm protocol .mn WM_DELETE_WINDOW ".mn.controls.cancel invoke"
    wm group .mn .

    wm deiconify .mn
    focus .mn.name.ent
    raise .mn
    grab set .mn
    vwait done
    grab release .mn
    destroy .mn
}
 

proc get_string {wintitle msg1 {msg2 ""}} {
    global result

    catch {unset result}

    toplevel .getstring
    pack [frame .getstring.main] -side top -expand yes \
	-fill both -padx 4 -pady 4
    pack [frame .getstring.controls] -side top -expand yes -fill x
    pack [label .getstring.main.msg1 -text $msg1] -side top \
	-padx 8 -pady 4
    pack [entry .getstring.main.enter -borderwidth 1] \
	-side top -padx 16 -pady 4 -fill x
    if {$msg2 != ""} {
	pack [label .getstring.main.msg2 -text $msg2] -side top \
	    -padx 8 -pady 4
    }
    set okbttn \
	[button .getstring.controls.ok -text "OK" \
 	     -command {
 		 set result [.getstring.main.enter get]
 	     }
	 ]
    set cancelbttn \
	[button .getstring.controls.cancel -text "Cancel" \
	     -command {
		 set result ""
	     }
	 ]
    pack $cancelbttn -side right -padx 28 -pady 4
    pack $okbttn -side left -padx 28 -pady 4

    wm title .getstring $wintitle
    bind .getstring <KeyPress-Return> {.getstring.controls.ok invoke}
    bind .getstring <KeyPress-Escape> {.getstring.controls.cancel invoke}
    bind .getstring <ButtonPress> {
 	wm deiconify .getstring
 	raise .getstring
	focus .getstring.main.enter
    }
    wm protocol .getstring WM_DELETE_WINDOW ".getstring.controls.cancel invoke"
    focus .getstring.main.enter
    raise .getstring
    grab set .getstring
    vwait result
    grab release .getstring
    destroy .getstring
    return $result
}



######################################################
######   MANIPULATING OBJECTS   ##########################
######################################################

proc obj_set_aname {objID} {
    global mapInfo
    set mapInfo(aname-$objID) \
	[get_string "Area Name" \
	     "Enter a brief descriptive name for this area:" \
	     "This will be the alternate text for the HTML <area> tag."]
}

proc obj_activate {win} {
    global mapInfo config

    set objID [$win find withtag current]
    if {$mapInfo(color-$objID) == "dark"} {
	$win itemconfigure $objID -fill $config(dark-act) \
	    -outline $config(dark-act)
    } else {
	$win itemconfigure $objID -fill $config(light-act) \
	    -outline $config(light-act)
    }
#    $win addtag $objID active
    $win addtag active withtag current
}

proc obj_deactivate {win} {
    global mapInfo config

    set objID [$win find withtag current]
    if {$mapInfo(color-$objID) == "dark"} {
	$win itemconfigure $objID -fill $config(dark) \
	    -outline $config(dark)
    } else {
	$win itemconfigure $objID -fill $config(light) \
	    -outline $config(light)
    }
    $win dtag $objID active
}

proc obj_delete {win} {
    global mapInfo
    if {[info exists mapInfo(objects)]} {
	set objID [$win find withtag current]
	set pos [lsearch $mapInfo(objects) $objID]
	if {$pos >= 0} {
	    set mapInfo(objects) [lreplace $mapInfo(objects) $pos $pos]
	    $win delete current
	}
    }
    set mapInfo(modified) 1
}

proc obj_move_start {x y} {
    global prevx prevy
    set prevx $x
    set prevy $y
}

proc obj_move {win x y} {
    global prevx prevy
    set xmove [expr {$x - $prevx}]
    set ymove [expr {$y - $prevy}]
    $win move current $xmove $ymove
    set prevx $x
    set prevy $y
}

proc obj_move_end {} {
    global prevx prevy
    unset prevx prevy
    set mapInfo(modified) 1
}

######################################################
######   DRAWING FUNCTIONS  #############################
######################################################

proc circle_start {x y} {
    global startpoint
    set startpoint [list $x $y]
}

proc circle_draw {win x y} {
    global mapInfo startpoint config

    set colorspec $mapInfo(colordefault)
    set color $config($colorspec-act)

    set x0 [lindex $startpoint 0]
    set y0 [lindex $startpoint 1]
    set xlen [expr {$x - $x0}]
    set ylen [expr {$y - $y0}]
    set radius [expr {hypot($xlen,$ylen)}]

    set x1 [expr {$x0 - $radius}]
    set y1 [expr {$y0 - $radius}]
    set x2 [expr {$x0 + $radius}]
    set y2 [expr {$y0 + $radius}]

    $win delete newobj
    $win create oval $x1 $y1 $x2 $y2 -outline $color -tags newobj
}

proc circle_set {win x y} {
    global config startpoint mapInfo

    set start $startpoint
    unset startpoint

    foreach {startx starty} $start {
	if {$x == $startx || $y == $starty} {
	    return
	}
    }

    set colorspec $mapInfo(colordefault)
    set color $config($colorspec-act)

    set objID [$win find withtag newobj]
    lappend mapInfo(objects) $objID
    set mapInfo(color-$objID) $colorspec

    $win itemconfigure newobj -fill $color -stipple gray25
    $win addtag mapobj withtag newobj
    $win dtag newobj

    set mapInfo(modified) 1

    if {$config(aname)} {
	obj_set_aname $objID
    }
}

 proc polygon_start_or_draw {win x y} {
     global coords waiting startpoint

     if {[info exists coords]} {
     } else {
	 set coords [list $x $y]
	 set startpoint 1
     }
 }

 proc polygon_draw {win x y} {
     global mapInfo coords config startpoint

     if {[info exists startpoint]} {
	 unset startpoint
     }

     set colorspec $mapInfo(colordefault)
     set color $config($colorspec-act)

     set len [llength $coords]
     set prevx [lindex $coords [expr {$len - 2}]]
     set prevy [lindex $coords [expr {$len - 1}]]
    
     $win delete templine
     $win dtag newline
     $win create line $prevx $prevy $x $y -fill $color -tags templine
     
 }

 proc polygon_set_point {win x y} {
     global mapInfo coords config startpoint

     if {[info exists startpoint]} {
	 unset startpoint
     } else {

	 set colorspec $mapInfo(colordefault)
	 set color $config($colorspec-act)
	 
	 if {[info exists coords]} {
	     set len [llength $coords]
	     set prevx [lindex $coords [expr {$len - 2}]]
	     set prevy [lindex $coords [expr {$len - 1}]]
	     
	     set coords [lappend coords $x $y]
	     
	     $win delete templine
	     $win dtag newline
	     $win create line $prevx $prevy $x $y -fill $color \
		 -tags {newline newobj}
	 }
     }
 }

 proc polygon_cancel_point {win} {
     global coords
     if {[info exists coords]} {
 	set coords [lreplace [lreplace $coords end end] end end]
 	$win delete newline
     }
 }

 proc polygon_set {win} {
     global coords waiting config mapInfo

     if {[info exists waiting]} {
	 after cancel $waiting
     }

     if {[info exists coords]} {
	 set colorspec $mapInfo(colordefault)
	 set color $config($colorspec-act)
 	$win delete newobj
	 set objID [eval $win create polygon $coords -outline $color \
			-fill $color -stipple gray25 -tags mapobj]
	 lappend mapInfo(objects) $objID
	 set mapInfo(color-$objID) $colorspec
 	unset coords
     }
    set mapInfo(modified) 1
    
     if {$config(aname)} {
	obj_set_aname $objID
    }
 }

proc rect_start {x y} {
    global startpoint
    set startpoint [list $x $y]
}   

proc rect_draw {win x y} {
    global mapInfo startpoint config
    set colorspec $mapInfo(colordefault)
    set color $config($colorspec-act)
    foreach {startx starty} $startpoint {}
    $win delete newobj
    $win create rectangle $startx $starty $x $y -outline $color -tags newobj
}

proc rect_set {win x y} {
    global mapInfo startpoint config
    
    set start $startpoint
    unset startpoint

    foreach {startx starty} $start {
	if {$x == $startx || $y == $starty} {
	    return
	}
    }

    set colorspec $mapInfo(colordefault)
    set color $config($colorspec-act)

    set objID  [$win find withtag newobj]
    lappend mapInfo(objects) $objID
    set mapInfo(color-$objID) $colorspec

    $win addtag mapobj withtag newobj
    $win itemconfigure newobj -fill $color -stipple gray25
    $win dtag newobj

    set mapInfo(modified) 1

    if {$config(aname)} {
	obj_set_aname $objID
    }
}
