#include <qapp.h>
#include <qpushbt.h>
#include <qfiledlg.h>
#include <qimage.h>
#include <qslider.h> 
#include <kiconloader.h>
#include "webtree.h"

#include "view/viewport.h"
#include "view/treelist.h"
#include "view/wtview.h"

#include "dlg/dlg_appear.h"	// for appearance dialog
#include "dlg/dlg_pref.h"	// for preferences dlg
#include "dlg/dlg_about.h"	// about dialog

#include "jpeg.h"

#include <unistd.h>		// for mktemp

// initialize global variables

QList <WebTree> WebTree::windowList;
Pref defaultPrefs;
Appearance appear;
Document *context;
QPixmap *loadAnim[LOAD_ANIM_FRAMES], *noLoadAnim, *treeHtmlDoc, *treePicDoc,
        *treeCompDoc, *treeOtherDoc, *aboutPixmap = NULL, *dndPixmap;
bool displayGUI;
QString kdndHtmlPix;
QString tempFile= "/tmp/kwimageXXXXXX";

int debug_level = DBG_LEVEL;

void usage( int , char **argv )
{
  printf( APPNAME " Version " MAJOR_VER "." MINOR_VER "\n\n"
	  "Usage: %s [options] [files...] \n"
	  "Valid options are:\n"
	  "   -h        Usage\n"
	  "   -d #      debuglevel\n"

	  , argv[0] );
}

// main...
int main( int argc, char **argv )
{
  int argi=1;
  int rc;
  displayGUI = TRUE;	// display the gui? if invoked by cmndline
				// this is not wanted
  WebTree *wt;

  if(argc > 1)
  {
    if(argv[argi][0]=='-')
      {
	switch(argv[argi][1])
	  {
	  case 'd':
	    argi++;
	    debug_level = atoi( argv[argi] );
	    fprintf( stderr, "debuglevel %d\n", debug_level );
	    argi++;
	    break;
	  case 'h':
	    usage(argc,argv);
	    exit(0);
	  }
	
      }

  }

  http_glob_init();	/* initialize network backend and fork second process
			 (gethostname cache) */
  KApplication a( argc, argv );

  // TODO better solution for pixmap dir (not only here)
  QString pixdir = kapp->kde_toolbardir();

  /*  mktemp( (char *)tempFile.data() );*/

  WebTree::initAppear( &appear );

  WebTree::windowList.setAutoDelete( TRUE );
  WebTree::readAppPrefs();

  // del start
  defaultPrefs.viewfontsizedefault = 12;
  defaultPrefs.viewfontsizemin = 10;
  // del end
  
  if( displayGUI )
  {
    QImageIO::defineIOHandler("JFIF","^\377\330\377\340..JFIF", 0,
			      read_jpeg_jfif, NULL);

    noLoadAnim  = new QPixmap( pixdir + "noload.xpm" );
    loadAnim[0] = new QPixmap( pixdir + "load0.xpm" );
    loadAnim[1] = new QPixmap( pixdir + "load1.xpm" );
    loadAnim[2] = new QPixmap( pixdir + "load2.xpm" );
    loadAnim[3] = new QPixmap( pixdir + "load3.xpm" );

    treeHtmlDoc = new QPixmap( pixdir + "ti_doc.xpm" );
    treePicDoc = new QPixmap( pixdir + "ti_pic.xpm" );
    treeCompDoc = new QPixmap( pixdir + "ti_comp.xpm" );
    treeOtherDoc = new QPixmap( pixdir + "ti_other.xpm" );

    pixdir = kapp->kde_datadir();
    pixdir.detach();
    pixdir += "/kwebtree/";

    dndPixmap = new QPixmap( pixdir + "wtdnd.xpm" );

    if(argi >= argc) wt = WebTree::newWindow( TRUE );
    else
      while(argi<argc)
	{
	  wt = WebTree::newWindow( TRUE );
	  wt->loadFile( QString( argv[argi] ) );
	  argi++;
	}

    rc = a.exec();

    WebTree::endAppear( &appear );

    delete noLoadAnim;
    delete loadAnim[0];
    delete loadAnim[1];
    delete loadAnim[2];
    delete loadAnim[3];
    delete treeHtmlDoc;
    delete treePicDoc;
    delete treeCompDoc;
    delete treeOtherDoc;
    delete dndPixmap;
  }

  http_glob_finalize();
}

// ctor

WebTree::WebTree( Pref *copyPrefs, QWidget *, char *name )
  : KTopLevelWidget( name ),
    changed( FALSE ),
    empty( TRUE )
{
  Searchlist *sl1, *sl2, **psl2;
  Pref *cpyprf;

  timer = new QTimer( this );
  snappy = new QTimer( this );

  // initialize gui
  createMenubar();
  createToolbar();
  createStatusbar();

  viewport = new Viewport( this );
  CHECK_PTR( viewport );

  treelist = new TreeList( &doc, viewport );
  CHECK_PTR( treelist );

  panner = new KPanner( viewport, "panner", KPanner::O_HORIZONTAL | KPanner::U_PERCENT, 30 );

  view = new WtView( &doc, panner->child1() );
  CHECK_PTR( view );
  viewwin = view->getViewWin();

  overview = new WtView( &doc, panner->child0(), NULL, TRUE );
  CHECK_PTR( overview );
  overviewwin = overview->getViewWin();

  setView( viewport );

  dropzone = new KDNDDropZone( viewport, DndURL );
  CHECK_PTR( dropzone );

  // connect sigs & slots

  // triggered from the popupmenu
/*  connect( viewwin, SIGNAL( stopLoadingNode(TreeInfo *) ),
	   this, SLOT( stopLoadingNode(TreeInfo *) ) );

  connect( viewwin, SIGNAL( startLoadingNode(TreeInfo *) ),
	   this, SLOT( startLoadingNode(TreeInfo *) ) );*/

  connect( viewwin, SIGNAL( savePage(TreeInfo *) ),
	   this, SLOT( savePage(TreeInfo *) ) );

  connect( viewwin, SIGNAL( setLocation(const char *) ),
	   this, SLOT( startLoading(const char *) ) );
  // triggered if mouse is over ti. shows url
  connect( viewwin, SIGNAL( setStatusUrl(const char *) ),
	   this, SLOT( changeStatusUrl(const char *) ) );
  // timeout function for update
  connect( timer, SIGNAL( timeout() ),
	   this, SLOT( updateWebTree() ) );
  // this is needed for a snappy user interface...
  connect( snappy, SIGNAL( timeout() ),
	   this, SLOT( processWebTree() ) );
  // triggereg if someone drops anything on us
  connect( dropzone, SIGNAL( dropAction(KDNDDropZone *) ),
	   this, SLOT( dropEvent(KDNDDropZone *) ) );
  // triggered if the viewport resized to resize its children
  connect( viewport, SIGNAL( resizeViews() ),
	   this, SLOT( resizeViews() ) );
  // triggereg by timer to update the animation
  connect( this, SIGNAL( animationStep() ),
	   loadAnimW, SLOT( animationStep() ) );

  connect( panner, SIGNAL( positionChanged() ),
	   this, SLOT( resizePanner() ) );

  connect( overviewwin, SIGNAL( treeInfoSelected(TreeInfo *) ),
	   viewwin, SLOT( centerTreeInfo(TreeInfo *)) );

  connect( viewwin, SIGNAL( ensureLoading() ),
	   this, SLOT( ensureLoading() ));

  connect( this, SIGNAL( resetViewWin() ),
	   viewwin, SLOT( resetViewWin() ));

  cpyprf = ( copyPrefs != NULL ) ? copyPrefs : &defaultPrefs;
    
  // set document struct to zero
  memset( &doc, 0, sizeof(doc) );
  doc.view.window = this;
  doc.view.view   = viewwin;
  // copy default prefs to document
  memcpy( &(doc.pref), cpyprf, sizeof(Pref) );

  //locbar->haltAnim(); TODO

  // copy in/excludelists
  sl1 = cpyprf->searchincludelist;
  psl2 = &(doc.pref.searchincludelist);
  while( sl1 )
  {
    sl2 = (Searchlist*)malloc( sizeof(Searchlist) );
    memcpy( sl2, sl1, sizeof(Searchlist) );
    sl2->next = NULL;
    *psl2 = sl2;
    psl2 = &(sl2->next);
    sl1 = sl1->next;
  }
  *psl2 = NULL;

  sl1 = cpyprf->searchexcludelist;
  psl2 = &(doc.pref.searchexcludelist);
  while( sl1 )
  {
    sl2 = (Searchlist*)malloc( sizeof(Searchlist) );
    memcpy( sl2, sl1, sizeof(Searchlist) );
    sl2->next = NULL;
    *psl2 = sl2;
    psl2 = &(sl2->next);
    sl1 = sl1->next;
  }
  *psl2 = NULL;

  ::context = &doc;
  http_initialize();	// initialize http-engine
  html_initialize();	// initialize html-engine
  layout_init();	// initialize layout-engine

  viewwin->setSize( 1, 1 );
  overviewwin->setSize( 1, 1 );
  updateViewType();
  viewwin->setBackgroundColor( ::appear.back );
  overviewwin->setBackgroundColor( ::appear.back );
}

// dtor

WebTree::~WebTree()
{
  Searchlist *sl1, *sl2;

  ::context = &doc;
  html_free();		// free html memory
  if( !http_finalize() )
    dbg( DI, DBG_HTTP | 2, "http_finalize failed! (running connections)");

  // delete searchlists
  sl1 = doc.pref.searchincludelist;
  while( sl1 )
  {
    sl2 = sl1->next;
    free( sl1 );
    sl1 = sl2;
  }
  sl1 = doc.pref.searchexcludelist;
  while( sl1 )
  {
    sl2 = sl1->next;
    free( sl1 );
    sl1 = sl2;
  }

  delete toolbar1;
  delete toolbar2;
  delete dropzone;
  delete overview;
  delete view;
  delete panner;
  delete treelist;
  delete locbar;
  delete viewport;
  delete timer;
  delete snappy;
}

// is called before a window gets closed. if the document has changed 
// the user has the choice to save it or he can abort the close. quits the
// application if this was the last window
void WebTree::closeEvent( QCloseEvent * )
{
  bool accepted = TRUE;
  int rc;

  if( changed )
  {
    rc = KMsgBox::yesNoCancel(0, APPNAME, 
			      klocale->translate( "Query has changed.\n"
			      "Do you want to save it?"));
    if(rc == 1) // yes
      saveFile();
    if(rc == 3) // cancel
      accepted = FALSE;
  }

  if( accepted )
  {
    windowList.removeRef( this );		// remove ref & delete this

    if( windowList.count() == 0 )		// was i the last window ? then quit
      qApp->quit();
  }
}

// static function that constructs a new window
WebTree* WebTree::newWindow( bool first, Pref *copyPrefs  )
{
  WebTree *wt = new WebTree( copyPrefs );
  CHECK_PTR( wt );
  windowList.append( wt );

  dbg( DI, DBG_MAIN | 3 , "window created" );

  if( ::appear.width != 0 && ::appear.height != 0 )
    wt->resize( ::appear.width, ::appear.height );
  else
    wt->resize( 400, 300 );
  
  wt->setMinimumSize( 250, 250 );
  wt->setCaption( APPNAME " " MAJOR_VER "." MINOR_VER );

  // is there a way to move the show() after the following lines so that
  // the window doesn't move visible? if i do this the windowmanager overrides the
  // position on show() :-(
  wt->show();
  if( first && appear.width != 0 && appear.height != 0 )
    wt->move( appear.x, appear.y );

  return wt;
}

void WebTree::openFile()
{
  QString f = QFileDialog::getOpenFileName( 0, "*.wt", this );
  if( !f.isEmpty() )
  {
    if( !isEmpty() )
    {
      WebTree *wt = WebTree::newWindow();
      wt->loadFile( f );
    }
    else
      loadFile( f );
  }
}

void WebTree::loadFile( const QString &f )
{ 
  filename = f;

  QString str;
  KSimpleConfig conf( f );
  
  conf.setGroup( "WebTree" );
  
  str = conf.readEntry( "Version" );
  if( !str.isEmpty() )
    {
      readPrefs( &conf, &(doc.pref) );
      
      conf.setGroup( "Data" );
      location = conf.readEntry( "Location" );
      
      emit setLocation( location );
	
      startLoading( location );
      empty = FALSE;
      changed = FALSE;
    }
  else
    {
      dbg( DI, DBG_MAIN | 1, "Load error. Wrong file type" );
      KMsgBox::message( 0, "Load Error", "The File you selected is not a WebTree file" );
    }
  
}

void WebTree::saveFile( bool saveAs )
{
  QString f;

  if( saveAs || filename.isEmpty() )
    f = QFileDialog::getSaveFileName( 0, "*.wt", this );
  else
    f = filename;

  if ( !f.isEmpty() ) 
    {
      filename = f;

      KSimpleConfig conf( f );

      conf.setGroup( "WebTree" );
      conf.writeEntry( "Version", MAJOR_VER "." MINOR_VER );
      writePrefs( &conf, &(doc.pref) );
      conf.setGroup( "Data" );
      conf.writeEntry( "Location", location );
    
      empty = FALSE;
      changed = FALSE;

    }
  else
    {
      dbg( DI, DBG_MAIN | 1, "Write error. Could not open file for writing" );
      KMsgBox::message( 0, "Write Error", "Could not open file for writing" );
    }
}

void WebTree::quit()
{
  WebTree *wt;
  bool doQuit = TRUE;
  int rc;

  QListIterator<WebTree> it( windowList );

  for( it.toFirst(); it.current(); ++it )
  {
    wt = it.current();
    
    if( wt->isChanged() )
    {
      rc = KMsgBox::yesNoCancel(0, APPNAME, 
				klocale->translate("Query has changed.\n"
				"Do you want to save it?"));
      if( rc == 1 ) // yes
	wt->saveFile();
      if( rc == 3 ) // cancel
      {
	doQuit = FALSE;
	break;
      }
    }
  }

  if( doQuit )
  {
    for( it.toFirst(); it.current(); )
      windowList.removeRef( it.current() );
    qApp->quit();
  }
}

void WebTree::preferences( Pref *p )
{
  QTabDialog prefs( this, klocale->translate("Preferences"), TRUE );

  DlgHttpTab httpprefs( p, &prefs );
  DlgHtmlTab htmlprefs( p, &prefs );
  DlgViewTab viewprefs( p, &prefs );
  DlgInexTab inexprefs( p, &prefs );

  prefs.addTab( &httpprefs, klocale->translate("Http") );
  prefs.addTab( &htmlprefs, klocale->translate("Mirror") );
  prefs.addTab( &inexprefs, klocale->translate("Scope") );
  prefs.addTab( &viewprefs, klocale->translate("View") );

  prefs.setApplyButton(klocale->translate("Apply"));
  prefs.setCancelButton(klocale->translate("Cancel"));
  prefs.setOKButton(klocale->translate("OK"));

  connect( &prefs, SIGNAL( aboutToShow() ), &htmlprefs, SLOT( setForm() ) );
  connect( &prefs, SIGNAL( aboutToShow() ), &httpprefs, SLOT( setForm() ) );
  connect( &prefs, SIGNAL( aboutToShow() ), &viewprefs, SLOT( setForm() ) );
  connect( &prefs, SIGNAL( aboutToShow() ), &inexprefs, SLOT( setForm() ) );

  connect( &prefs, SIGNAL( applyButtonPressed() ), &htmlprefs, SLOT( readForm() ) );
  connect( &prefs, SIGNAL( applyButtonPressed() ), &httpprefs, SLOT( readForm() ) );
  connect( &prefs, SIGNAL( applyButtonPressed() ), &viewprefs, SLOT( readForm() ) );
  connect( &prefs, SIGNAL( applyButtonPressed() ), &inexprefs, SLOT( readForm() ) );

  prefs.resize( 410,272 );
  prefs.setFixedSize( 410, 272 );

  prefs.exec();

  updateViewType();

  if( p == &defaultPrefs )
    WebTree::writeAppPrefs();
}


void WebTree::aboutBox()
{
  About ab( this );

  ab.exec();
}

void WebTree::itemSelected( int tag )
{
  switch( tag )
  {

    // menubar or toolbar 1

    case ID_FILE_NEW:
      newWindow();
      break;
    case ID_FILE_OPEN:
      openFile();
      break;
    case ID_FILE_SAVE:
      saveFile();
      break;
    case ID_FILE_SAVE_AS:
      saveFile( TRUE );
      break;
    case ID_FILE_CLOSE:
      close();
      break;
    case ID_FILE_QUIT:
      quit();
      break;
    case ID_OPTIONS_DEFAULTPREFS:
      preferences( &defaultPrefs );
      break;
    case ID_OPTIONS_PREFS:
      preferences( &(doc.pref) );
      break;
    case ID_OPTIONS_APPEARANCE:
      setAppearance();
      break;
    case ID_OPTIONS_SAVE_WINPOS:
      saveWindowPos();
      break;
    case ID_HELP_ABOUT:
      aboutBox();
      break;
    case ID_STOP:
      stopLoading();
      break;
    case ID_PAUSE:
      pauseLoading( FALSE );
      break;
  case ID_CONT:
      pauseLoading( TRUE );
      break;
    case ID_HELP_CONTENTS:
      kapp->invokeHTMLHelp("kwebtree/kwebtree.html", NULL );
      break;
    case ID_TOGGLE_MLINKS:
      ::appear.mouseLinks = toolbar2->isButtonOn(ID_TOGGLE_MLINKS);
      updateAllViews( FALSE );
      break;

      // toolbar 2

      /*    case ID_VIEW_CENTER:
      doc.pref.viewtype = centralview;
      updateViewType();
      break;*/
    case ID_VIEW_TOPDOWN:
      if(toolbar2->isButtonOn(ID_VIEW_TOPDOWN))
	{
	  toolbar2->setButton(ID_VIEW_LEFTRIGHT,FALSE);
	  toolbar2->setButton(ID_VIEW_TREELIST,FALSE);
	}
      else toolbar2->setButton(ID_VIEW_TOPDOWN,TRUE);
      doc.pref.viewtype = topdownview;
      updateViewType();
      break;
    case ID_VIEW_LEFTRIGHT:
      if(toolbar2->isButtonOn(ID_VIEW_LEFTRIGHT))
	{
	  toolbar2->setButton(ID_VIEW_TOPDOWN,FALSE);
	  toolbar2->setButton(ID_VIEW_TREELIST,FALSE);
	}
      else toolbar2->setButton(ID_VIEW_LEFTRIGHT,TRUE);
      doc.pref.viewtype = leftrightview;
      updateViewType();
      break;
    case ID_VIEW_TREELIST:
      if(toolbar2->isButtonOn(ID_VIEW_TREELIST))
	{
	  toolbar2->setButton(ID_VIEW_LEFTRIGHT,FALSE);
	  toolbar2->setButton(ID_VIEW_TOPDOWN,FALSE);
	}
      else toolbar2->setButton(ID_VIEW_TREELIST,TRUE);
      doc.pref.viewtype = browserview;
      updateViewType();
      break;
    default:
      break;
  }
}

void WebTree::createMenubar()
{
  QPopupMenu *file = new QPopupMenu;
  CHECK_PTR( file );

  file->insertItem( klocale->translate("New"), ID_FILE_NEW );
  file->insertItem( klocale->translate("Open..."), ID_FILE_OPEN );
  file->insertItem( klocale->translate("Save"), ID_FILE_SAVE );
  file->insertItem( klocale->translate("Save as..."), ID_FILE_SAVE_AS );
  file->insertItem( klocale->translate("Close"), ID_FILE_CLOSE );
  file->insertSeparator();
  file->insertItem( klocale->translate("Quit"), ID_FILE_QUIT );

  QPopupMenu *opt = new QPopupMenu;
  CHECK_PTR( opt );

  opt->insertItem( klocale->translate("Preferences..."), ID_OPTIONS_PREFS );
  opt->insertSeparator();
  opt->insertItem( klocale->translate("Default Preferences..."), ID_OPTIONS_DEFAULTPREFS );
  opt->insertItem( klocale->translate("Appearance..."), ID_OPTIONS_APPEARANCE );
  opt->insertSeparator();
  opt->insertItem( klocale->translate("Save Window Position"), ID_OPTIONS_SAVE_WINPOS );

  QPopupMenu *help = new QPopupMenu;
  CHECK_PTR( help );

  help->insertItem( klocale->translate("About..."), ID_HELP_ABOUT );
  help->insertItem( klocale->translate("Help Contents"), ID_HELP_CONTENTS );

  connect(file, SIGNAL( activated(int) ), this, SLOT( itemSelected(int) ) );
  connect(opt, SIGNAL( activated(int) ), this, SLOT( itemSelected(int) ) );
  connect(help, SIGNAL( activated(int) ), this, SLOT( itemSelected(int) ) );

  menubar = new KMenuBar( this );
  CHECK_PTR(menubar);

  menubar->insertItem(klocale->translate("&File"), file );
  menubar->insertItem(klocale->translate("&Options"), opt );
  menubar->insertSeparator();
  menubar->insertItem(klocale->translate("&Help"), help );

  setMenu( menubar );
  menubar->show();
}

void WebTree::createToolbar()
{
  QPixmap pixmap;

  toolbar1 = new KToolBar( this );

  pixmap = kapp->getIconLoader()->loadIcon( "filenew.xpm" );
  toolbar1->insertButton(pixmap, ID_FILE_NEW, TRUE, 
			 klocale->translate("New window") );

  pixmap = kapp->getIconLoader()->loadIcon( "fileopen.xpm" );
  toolbar1->insertButton(pixmap, ID_FILE_OPEN , TRUE, 
			 klocale->translate("Open file") );

  pixmap = kapp->getIconLoader()->loadIcon( "filefloppy.xpm" );
  toolbar1->insertButton(pixmap, ID_FILE_SAVE_AS , FALSE, 
			 klocale->translate("Save file as...") );

  toolbar1->insertSeparator();
  pixmap = kapp->getIconLoader()->loadIcon("stop.xpm" );
  toolbar1->insertButton(pixmap, ID_STOP, FALSE,
			 klocale->translate( "Stop loading") );

  pixmap = kapp->getIconLoader()->loadIcon("pause.xpm" );
  toolbar1->insertButton(pixmap, ID_PAUSE, FALSE,
			 klocale->translate( "Pause loading") );

  pixmap = kapp->getIconLoader()->loadIcon("cont.xpm" );
  toolbar1->insertButton(pixmap, ID_CONT, FALSE,
			 klocale->translate( "Continue loading") );

  addToolBar( toolbar1 );
  toolbar1->setBarPos( KToolBar::Top );
  toolbar1->setFullWidth(FALSE);
  toolbar1->show();
  
  toolbar2 = new KToolBar( this );

  pixmap = kapp->getIconLoader()->loadIcon( "tick.xpm" );
  toolbar2->insertButton(pixmap, ID_OPTIONS_PREFS , TRUE,
			 klocale->translate( "Preferences...") );
  toolbar2->insertSeparator();
  pixmap = kapp->getIconLoader()->loadIcon( "mlink.xpm" );
  toolbar2->insertButton( pixmap, ID_TOGGLE_MLINKS, TRUE, 
	         klocale->translate("Show links only if mouse is over node") );
  toolbar2->setToggle(ID_TOGGLE_MLINKS);
  toolbar2->insertSeparator();

/*  TODO implement center view

  pixmap = kapp->getIconLoader()->loadIcon( "cv_view.xpm" );
  toolbar2->insertButton( pixmap, ID_VIEW_CENTER, TRUE, "Center view" );
  toolbar2->setToggle(ID_VIEW_CENTER); */
  pixmap = kapp->getIconLoader()->loadIcon( "td_view.xpm" );
  toolbar2->insertButton( pixmap, ID_VIEW_TOPDOWN, TRUE, 
			  klocale->translate("Top-down view") );
  toolbar2->setToggle(ID_VIEW_TOPDOWN);
  toolbar2->setButton(ID_VIEW_TOPDOWN,TRUE);
  pixmap = kapp->getIconLoader()->loadIcon( "lr_view.xpm" );
  toolbar2->insertButton( pixmap, ID_VIEW_LEFTRIGHT, TRUE, 
			  klocale->translate("Left-right view") );
  toolbar2->setToggle(ID_VIEW_LEFTRIGHT);
  pixmap = kapp->getIconLoader()->loadIcon( "tl_view.xpm" );
  toolbar2->insertButton( pixmap, ID_VIEW_TREELIST, TRUE, 
			  klocale->translate("Treelist view") );
  toolbar2->setToggle(ID_VIEW_TREELIST);

  addToolBar( toolbar2 );
  toolbar2->setBarPos( KToolBar::Top );
  toolbar2->setFullWidth(FALSE);
  toolbar2->show();

  connect( toolbar1, SIGNAL( clicked(int) ), this, SLOT( itemSelected(int) ) );
  connect( toolbar2, SIGNAL( clicked(int) ), this, SLOT( itemSelected(int) ) );

  locbar = new KToolBar( this );
  locbar->setBarPos( KToolBar::Top );
  locbar->setFullWidth(TRUE);

  locbar->insertFrame(ID_LOCATION_ANIM,24);
  loadAnimW = new AnimW(locbar->getFrame(ID_LOCATION_ANIM));
  loadAnimW->setGeometry(0,0,24,24);
  loadAnimW->show();

  QFontMetrics fm(font());
  int loclabelwidth = fm.width( (const char*)klocale->translate("Location:"));
  locbar->insertFrame(ID_LOCATION_FRAME,loclabelwidth);
  QLabel* locationLabel = new QLabel((const char*)
				     klocale->translate("Location:"), 
				     locbar->getFrame(ID_LOCATION_FRAME));
  locationLabel->setGeometry(0, 0, loclabelwidth, 24);
  locationLabel->show();

  locbar->insertLined("http://",ID_LOCATION_LINED,
		      SIGNAL( returnPressed() ),this,
		      SLOT( urlChanged() ),TRUE,NULL,
		      50);
  locbar->setItemAutoSized(ID_LOCATION_LINED);

  int zoomlwidth = fm.width( (const char*)klocale->translate("Zoom (400%):"));
  locbar->insertFrame(ID_ZOOM_FRAME,104 + zoomlwidth);
  locbar->alignItemRight(ID_ZOOM_FRAME,TRUE); // SHITT! Warum geht das nicht!
                                              // oft hast a pech ;-))
                                              // Ist das ein Source - chat ?!?
                                   // yeah - irc ist tot es lebe cvs chat ;-))
            // Genau! Sollten wir nicht einen anonymen CVS Zugang einrichten ?
  // Mit einem User, der nicht in der CVS-group drinnen ist, uns somit nicht
  // committen kann!  ===>>> Damit die Welt unseren CVS chat verfolgen kann!!
  QFrame* frame = locbar->getFrame(ID_ZOOM_FRAME);

  QSlider* zoom = new QSlider(QSlider::Horizontal, frame);
  zoom->setRange(10, 360);
  zoom->setSteps(18, 18);
  zoom->setValue(100);
  zoom->show();

  zoomlabel = new QLabel((const char*)klocale->translate("Zoom (400%):")
			 , frame);
  CHECK_PTR(zoomlabel);
  zoomlabel->show();

  zoomlabel->setGeometry(2, 0, zoomlwidth, 24);
  zoom->setGeometry(zoomlwidth+2 , 4, 100, 16);
  zoomlabel->setText( klocale->translate("Zoom (100%):") );

  connect(zoom, SIGNAL( valueChanged(int) ),
	  this, SLOT( zoomChanged(int)) );

  addToolBar( locbar );
  locbar->show();
}

void WebTree::urlChanged()
{
  startLoading(locbar->getLinedText(ID_LOCATION_LINED));
}

void WebTree::zoomChanged(int value)
{
  char buf[20];
  sprintf(buf, "Zoom (%d%%):", value);

  zoomlabel->setText(buf);
  zoomlabel->update();

  viewwin->zoomChanged(value);
}

void WebTree::createStatusbar()
{
  statusbar = new KStatusBar( this );
  statusbar->insertItem( "000", ID_STAT_THREADS );
  statusbar->insertItem( "0000", ID_STAT_POSTPONED );
  statusbar->insertItem( "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX", ID_STAT_URL );
  statusbar->insertItem( "", ID_STAT_TEXT );

  statusbar->changeItem( "", ID_STAT_THREADS );
  statusbar->changeItem( "", ID_STAT_POSTPONED );
  statusbar->changeItem( "", ID_STAT_URL );
  setStatusBar( statusbar );
  statusbar->show();
}

void WebTree::changeStatusText( const char *text )
{
  if( displayGUI )
  {
    dbg( DI, DBG_MAIN | 3, "%s", text );
    statusbar->changeItem( (char *)text, ID_STAT_TEXT );
  }
  else
    printf( text );
}

void WebTree::changeStatusUrl(const char *text)
{
  statusbar->changeItem( (char *)text, ID_STAT_URL );
}

void WebTree::readAppPrefs()
{
  KConfig *config = KApplication::getKApplication()->getConfig();

  readPrefs( config, &defaultPrefs );
  readAppear( config, &appear );
}

void WebTree::writeAppPrefs()
{
  KConfig *config = KApplication::getKApplication()->getConfig();

  writePrefs( config, &defaultPrefs );
  writeAppear( config, &appear );
}

void WebTree::readPrefs( KConfigBase *c, Pref *p )
{
  QString str;
  int v, i, j;
  char buf[40];
  Searchlist *sl, *slt, **psl;

  memset( p, 0, sizeof(Pref) );

  c->setGroup( "Preferences" );

  str = c->readEntry( "mirrorpath" );
  if( !str.isNull() )
    strcpy( p->mirrorpath, str.data() );
  else
    strcpy( p->mirrorpath, "/tmp" );

  p->mirrorsizemax = c->readNumEntry( "mirrorsizemax" );

  str = c->readEntry( "mirrorhttp" );
  p->mirrorhttp = ( !str.isNull() && str.find("on") == 0 ) ? TRUE : FALSE;

  str = c->readEntry( "mirrorpic" );
  p->mirrorpic = ( !str.isNull() && str.find("on") == 0 ) ? TRUE : FALSE;

  str = c->readEntry("mirrorother");
  p->mirrorother = (!str.isNull() && str.find("on") == 0) ? TRUE : FALSE;

  str = c->readEntry("http_proxy");
  if(!str.isNull())
    strcpy(p->http_proxy, str.data());
  
  v = c->readNumEntry("http_timeout");
  p->http_timeout = v ? v : 60; // 60 sekunden default

  v = c->readNumEntry("http_threadsmax");
  p->http_threadsmax = v ? v : 4; // default 4 stolen from Netscape 

  v = c->readNumEntry("searchdepth");
  p->searchdepth = v ? v : 10;

  str = c->readEntry("searchinclude");
  p->searchinclude = (!str.isNull() && str.find("on") == 0) ? TRUE : FALSE;

  i = c->readNumEntry("searchincludecount");

  sl = p->searchincludelist;
  psl = &(p->searchincludelist);
  for(j = 0; j < i; j++)
  {
    if(sl == NULL)
    {
      sl = (Searchlist*)malloc(sizeof(Searchlist));
      sl->next = NULL;
      *psl = sl;
    }
    sprintf(buf, "searchinclude_%d", j);
    str = c->readEntry(buf);

    strcpy(sl->domain, str.data());
      
    psl = &(sl->next);
    sl = sl->next;
  }
  *psl = NULL;
  while(sl)
  {
    slt = sl->next;
    free(sl);
    sl = slt;
  }

  i = c->readNumEntry("searchexcludecount");

  sl = p->searchexcludelist;
  psl = &(p->searchexcludelist);
  for(j = 0; j < i; j++)
  {
    if(sl == NULL)
    {
      sl = (Searchlist*)malloc(sizeof(Searchlist));
      sl->next = NULL;
      *psl = sl;
    }
    sprintf(buf, "searchexclude_%d", j);
    str = c->readEntry(buf);

    strcpy(sl->domain, str.data());
      
    psl = &(sl->next);
    sl = sl->next;
  }
  *psl = NULL;
  while(sl)
  {
    slt = sl->next;
    free(sl);
    sl = slt;
  }

  p->viewtype = (Viewtype)c->readNumEntry( "viewtype" );
  p->nodecontent = (Nodecontent)c->readNumEntry( "nodecontent" );
  p->viewshowordered = c->readNumEntry( "viewshowordered" );
/* moved to appearance
                
  v = c->readNumEntry("viewfontsizedefault");
  p->viewfontsizedefault = v ? v : 12;

  v = c->readNumEntry("viewfontsizemin");
  p->viewfontsizemin = v ? v : 8;*/
}

void WebTree::readAppear(KConfigBase *c, Appearance *a)
{
  QString str;
  int v;
  QColor col;
  QFont f, f2;
  
  c->setGroup("Appearance");

  col = c->readColorEntry("BackgroundColor");
  a->back = col.isValid() ? col : white;

  col = c->readColorEntry("TextColor");
  a->text = col.isValid() ? col : black;
  
  col = c->readColorEntry("LineColor");
  a->line = col.isValid() ? col : black;
  
  col = c->readColorEntry("HTMLColor");
  a->html = col.isValid() ? col : gray;

  col = c->readColorEntry("GraphicColor");
  a->graphic = col.isValid() ? col : blue;

  col = c->readColorEntry("PackedColor");
  a->packed = col.isValid() ? col : yellow;

  col = c->readColorEntry("OrderedColor");
  a->ordered = col.isValid() ? col : darkGray;

  col = c->readColorEntry("ErrorColor");
  a->error = col.isValid() ? col : red;

  col = c->readColorEntry("OtherColor");
  a->other = col.isValid() ? col : green;

  col = c->readColorEntry("CollapsedColor");
  a->collapsed = col.isValid() ? col : magenta;

  // a little hack...
  f2 = QFont::defaultFont();
  QFont::setDefaultFont( QFont( "Helvetica", 12 ) );
  f = c->readFontEntry( "BrowserFont" );
  a->font = f;
  QFont::setDefaultFont( f2 );

  str = c->readEntry( "MouseLinks" );
  a->mouseLinks = (!str.isNull() && str.find("on") == 0) ? TRUE : FALSE;

  str = c->readEntry( "LensEffect" );
  a->lens_effect = (!str.isNull() && str.find("on") == 0) ? TRUE : FALSE;

  str = c->readEntry( "DragMethod1" );
  a->drag_method1 = (!str.isNull() && str.find("on") == 0) ? TRUE : FALSE;

  str = c->readEntry( "LensExponent" );
  a->lens_exp = (!str.isNull()) ? atof( str ) : 2.0 ;

  if( a->lens_exp < 2.0 ) a->lens_exp = 2.0;
  else if( a->lens_exp > 5.0 ) a->lens_exp = 5.0;

  str = c->readEntry("PixmapRedraw");
  if(str.isEmpty())
    a->pixmapRedraw = TRUE;
  else
    a->pixmapRedraw = (!str.isNull() && str.find("on") == 0) ? TRUE : FALSE;

  v = c->readNumEntry( "FontMinsize" );
  a->fontminsize = v ? v : 8;

  v = c->readNumEntry( "FontMaxsize" );
  a->fontmaxsize = v ? v : 12;

  a->x = c->readNumEntry( "xpos" );
  a->y = c->readNumEntry( "ypos" );
  a->width = c->readNumEntry( "width" );
  a->height = c->readNumEntry( "height" );

  WebTree::recomputeAppear( a );
}

void WebTree::writePrefs(KConfigBase *c, Pref *p)
{
  QString str;
  int i;
  char buf[40];
  Searchlist *sl;

  c->setGroup("Preferences");

  c->writeEntry("mirrorpath", p->mirrorpath);
  c->writeEntry("mirrorsizemax", p->mirrorsizemax);
  c->writeEntry("mirrorhttp", p->mirrorhttp ? "on" : "off");
  c->writeEntry("mirrorpic", p->mirrorpic ? "on" : "off");
  c->writeEntry("mirrorother", p->mirrorother ? "on" : "off");
  c->writeEntry("http_proxy", p->http_proxy);
  c->writeEntry("http_timeout", p->http_timeout);
  c->writeEntry("http_threadsmax", p->http_threadsmax);
  c->writeEntry("searchdepth", p->searchdepth);
  c->writeEntry("searchinclude", p->searchinclude ? "on" : "off");

  sl = p->searchincludelist;
  i = 0;
  while(sl)
  {
    sprintf(buf, "searchinclude_%d", i);
    c->writeEntry(buf, sl->domain);
    sl = sl->next;
    i++;
  }
  c->writeEntry("searchincludecount", i);
  
  sl = p->searchexcludelist;
  i = 0;
  while(sl)
  {
    sprintf(buf, "searchexclude_%d", i);
    c->writeEntry(buf, sl->domain);
    sl = sl->next;
    i++;
  }
  c->writeEntry("searchexcludecount", i);

  c->writeEntry("viewtype", p->viewtype);
  c->writeEntry("nodecontent", p->nodecontent);
  c->writeEntry("viewshowordered", p->viewshowordered);
}

void WebTree::writeAppear(KConfigBase *c, Appearance *a)
{
  //  QColor col;
  QFont f;

  c->setGroup( "Appearance" );

  c->writeEntry( "BackgroundColor", a->back );
  c->writeEntry( "TextColor", a->text );
  c->writeEntry( "LineColor", a->line );
  c->writeEntry( "HTMLColor", a->html );
  c->writeEntry( "GraphicColor", a->graphic );
  c->writeEntry( "PackedColor", a->packed );
  c->writeEntry( "OrderedColor", a->ordered );
  c->writeEntry( "ErrorColor", a->error );
  c->writeEntry( "OtherColor", a->other );
  c->writeEntry( "CollapsedColor", a->collapsed );
  c->writeEntry( "BrowserFont", a->font );
  c->writeEntry( "MouseLinks", a->mouseLinks ? "on" : "off" );
  c->writeEntry( "PixmapRedraw", a->pixmapRedraw ? "on" : "off" );
  c->writeEntry( "MouseLinks", a->mouseLinks ? "on" : "off" );
  c->writeEntry( "LensEffect", a->lens_effect ? "on" : "off" );
  c->writeEntry( "DragMethod1", a->drag_method1 ? "on" : "off" );
  QString s;
  s.setNum( a->lens_exp );
  c->writeEntry( "LensExponent", s );
  c->writeEntry( "FontMinsize", a->fontminsize );
  c->writeEntry( "FontMaxsize", a->fontmaxsize );
  c->writeEntry( "xpos", a->x );
  c->writeEntry( "ypos", a->y );
  c->writeEntry( "width", a->width );
  c->writeEntry( "height", a->height );
}

void WebTree::saveWindowPos()
{
  appear.x = x();
  appear.y = y();
  appear.width = width();
  appear.height = height();

  writeAppPrefs();
}

void WebTree::setAppearance()
{
  int fma, fmi;
  QTabDialog a( this, klocale->translate("Appearance"), TRUE );

  DlgAppearFont tab1( this, &appear, &a );
  DlgAppearColor tab2( this, &appear, &a );
  DlgAppearOther tab3( this, &appear, &a );

  a.addTab( &tab1, klocale->translate("Fonts") );
  a.addTab( &tab2, klocale->translate("Colors") );
  a.addTab( &tab3, klocale->translate("Other") );

  a.setApplyButton(klocale->translate("Apply"));
  a.setCancelButton(klocale->translate("Cancel"));
  a.setOKButton(klocale->translate("OK"));

  a.resize( 380, 320 );
  a.setFixedSize( 380, 320 );

  connect( &a, SIGNAL( aboutToShow() ), &tab1, SLOT( setForm() ) );
  connect( &a, SIGNAL( aboutToShow() ), &tab2, SLOT( setForm() ) );
  connect( &a, SIGNAL( aboutToShow() ), &tab3, SLOT( setForm() ) );

  connect( &a, SIGNAL( applyButtonPressed() ), &tab1, SLOT( readForm() ) );
  connect( &a, SIGNAL( applyButtonPressed() ), &tab2, SLOT( readForm() ) );
  connect( &a, SIGNAL( applyButtonPressed() ), &tab3, SLOT( readForm() ) );

  fma = ::appear.fontminsize;
  fmi = ::appear.fontmaxsize;

  a.exec();

  WebTree::recomputeAppear( &appear );

  if( ::appear.fontminsize != fmi || ::appear.fontmaxsize != fma )
  {
    ::context = &doc;
    layout_recalcAll();    
  }

  view->resizeScrollBars();
  writeAppPrefs();
}

// start loading webpages. this is called after the user pressed enter in the
// location bar or loaded a webtree-document or selected "Set as new Location"
// in the popupmenu. it starts the timer end initializes all neccessary engines
void WebTree::startLoading( const char *loc )
{
  dbg( DI, DBG_MAIN | 3, "location changed to %s", loc);

  // delete items in treelist
  treelist->clear();

  if( strcmp( loc, location ) )
  {
    empty = FALSE;
    changed = TRUE;
    toolbar1->setItemEnabled( ID_FILE_SAVE_AS, TRUE );
  }

  location = loc;

  if( !location.isEmpty() )
  {
    // stolen from netscape...
    if( location.find("http://", 0, FALSE) == -1 &&
	location.find("ftp://", 0, FALSE) == -1 &&
	location.find("gopher://", 0, FALSE) == -1 &&
	location.find("file:/", 0, FALSE) == -1 )
      location.insert( 0, "http://" );

    viewwin->setSize( 1, 1 );
    overviewwin->setSize( 1, 1 );

    locbar->setLinedText(ID_LOCATION_LINED,location);
    //emit setLocation( location );

    emit resetViewWin();

    ::context = &doc;			// set context for c-functions
    if( html_startLoading( location.data()) == 0 )
    {
      toolbar1->setItemEnabled( ID_STOP, TRUE );
      toolbar1->setItemEnabled( ID_PAUSE, TRUE );

      timer->start( 300 );			// check every 200 ms
      processWebTree();
    }
  }

  viewwin->myUpdate();
  overviewwin->myUpdate();
}

void WebTree::stopLoading()
{
  dbg( DI, DBG_MAIN | 3, "stop loading" );

  //locbar->haltAnim(); TODO

  toolbar1->setItemEnabled( ID_STOP, FALSE );
  toolbar1->setItemEnabled( ID_PAUSE, FALSE );
  toolbar1->setItemEnabled( ID_CONT, FALSE );

  timer->stop();

  ::context = &doc;
  html_stopLoading();
  http_abortAll();

  updateThreadsPostponed();
}

// pause Loading
void WebTree::pauseLoading( bool cont )
{
  if( cont )
    {
      toolbar1->setItemEnabled( ID_PAUSE, TRUE );
      toolbar1->setItemEnabled( ID_CONT, FALSE );
      timer->start( 300 );
    }
  else
    {
      toolbar1->setItemEnabled( ID_PAUSE, FALSE );
      toolbar1->setItemEnabled( ID_CONT, TRUE );
      timer->stop();
    }
}

// this function is called by the timer. it checks for new arrived web-pages and
// inserts them into the webtree
void WebTree::updateWebTree()
{
  emit animationStep();
  processWebTree();
}

void WebTree::processWebTree()
{
  updateThreadsPostponed();

  ::context = &doc;
  if( html_buildWebTree() && timer->isActive() )
    snappy->start( 0, TRUE );	/* if we got data then call again... */
}

// the number of threads and portponed pages if the http module are shown in
// the statusline
void WebTree::updateThreadsPostponed()
{
  char buf[30];

  ::context = &doc;
  sprintf( buf, "%d", http_numberOfThreads() );
  statusbar->changeItem( buf, ID_STAT_THREADS );

  sprintf( buf, "%d", http_numberOfPostponed() );
  statusbar->changeItem( buf, ID_STAT_POSTPONED );
}

// sets the size of the view. this is called by the layout module after
// new nodes have been inserted into the tree to make room for them
void WebTree::setLayoutSpace( int width, int height )
{
  viewwin->setSize( width, height );
  overviewwin->setSize( width, height );
}

// updates the view after changes to the appearance
void WebTree::updateView( bool updov )
{
  if( viewwin->backgroundColor().rgb() != ::appear.back.rgb() )
    viewwin->setBackgroundColor( ::appear.back );
  if( overviewwin->backgroundColor().rgb() != ::appear.back.rgb() )
    overviewwin->setBackgroundColor( ::appear.back );

  viewwin->myUpdate();
  if( updov ) overviewwin->myUpdate();
}

// calls updateView for every top-level-widget
void WebTree::updateAllViews( bool updov )
{
  WebTree *wt;

  for( wt = windowList.first(); wt != NULL; wt = windowList.next() )
    wt->updateView( updov );
}

// starts loading at the selected node after the user stopped the the loading
// of this node or the entire tree
/*void WebTree::startLoadingNode( TreeInfo *ti )
{
  ensureLoading();

  ::context = &doc;
  html_startLoadingNode( ti );
}*/

// stops loading at the requested node. if there are any nodes below this point
// they are hidden (or deleted ?)
/*void WebTree::stopLoadingNode( TreeInfo *ti )
{
  ::context = &doc;
  html_stopLoadingNode( ti );
}*/

// saves the give page. calls the http module to order the page
void WebTree::savePage( TreeInfo *ti )
{
  char *fn = strrchr( ti->url, '/' );

  if( fn == NULL )
    fn = ti->url;

  QString fname = QFileDialog::getSaveFileName( 0, "*", this, fn );

  // TODO: check if file exists and warn user

  if(!fname.isEmpty())
  {
    ::context = &doc;
    view_orderPage( ti->url, GURL_SAVEPAGE, &doc, ti, fname.data(), NULL );
  }  
}

void WebTree::savePageArrived( const char *,  char * )
{
  // TODO implement
}

// is called when the user drops a file over the top-level-widget. starts loading
// of the given page
void WebTree::dropEvent( KDNDDropZone *dnd )
{
  QStrList ss = dnd->getURLList();
  QString s = ss.first();
  location = s;

  locbar->setLinedText(ID_LOCATION_LINED,location);
    //emit setLocation( s );
  emit resetViewWin();

  startLoading( s );
}

// updates the view between treelist-view and wtview (center/top-down & left-right
// -view)
void WebTree::updateViewType()
{
  if( doc.pref.viewtype == browserview )
  {
    treelist->show();
    panner->hide();
    treelist->setAutoUpdate( TRUE );
  }
  else
  {
    treelist->setAutoUpdate( FALSE );
    treelist->hide();
    panner->setOrientation( (doc.pref.viewtype == leftrightview) ?
			    KPanner::O_VERTICAL : KPanner::O_HORIZONTAL );
    layout_recalcAll();
    panner->show();
    resizePanner();
  }
}

// is called by the html module after a new node has been inserted into the
// tree
void WebTree::nodeAdded( TreeInfo *ti )
{
  treelist->addNode( ti );
}

// is called by the viewport to resize it's children
void WebTree::resizeViews()
{
  int w, h;
  
  w = viewport->width();
  h = viewport->height();

  treelist->setGeometry( 0, 0, w, h );
  // panner
  panner->setGeometry( 0, 0, w, h );
  resizePanner();
}

void WebTree::resizePanner()
{
  overview->setGeometry( 0, 0, panner->child0()->width(), panner->child0()->height() );
  view->setGeometry( 0, 0, panner->child1()->width(), panner->child1()->height() );
}

void WebTree::ensureLoading()
{
  if( !timer->isActive() )
  {
    ::context = &doc;
    toolbar1->setItemEnabled( ID_STOP, TRUE );
    toolbar1->setItemEnabled( ID_PAUSE, TRUE );
    toolbar1->setItemEnabled( ID_CONT, FALSE );

    timer->start( 300 );			// check every 300 ms
    processWebTree();
  }
}

void WebTree::initAppear( Appearance *a )
{
  a->cg_html = a->cg_graphic = a->cg_packed = a->cg_ordered = a->cg_error = a->cg_other = NULL;
  a->br_html = new QBrush( SolidPattern );
  a->br_graphic = new QBrush( SolidPattern );
  a->br_packed = new QBrush( SolidPattern );
  a->br_ordered = new QBrush( SolidPattern );
  a->br_error = new QBrush( SolidPattern );
  a->br_other = new QBrush( SolidPattern );
}

void WebTree::endAppear( Appearance *a )
{
  if( a->cg_html ) delete a->cg_html;
  if( a->cg_graphic ) delete a->cg_graphic;
  if( a->cg_packed ) delete a->cg_packed;
  if( a->cg_ordered ) delete a->cg_ordered;
  if( a->cg_error ) delete a->cg_error;
  if( a->cg_other ) delete a->cg_other;

  delete a->br_html;
  delete a->br_graphic;
  delete a->br_packed;
  delete a->br_ordered;
  delete a->br_error;
  delete a->br_other;
}

void WebTree::recomputeAppear( Appearance *a )
{
  if( a->cg_html ) delete a->cg_html;
  if( a->cg_graphic ) delete a->cg_graphic;
  if( a->cg_packed ) delete a->cg_packed;
  if( a->cg_ordered ) delete a->cg_ordered;
  if( a->cg_error ) delete a->cg_error;
  if( a->cg_other ) delete a->cg_other;

  a->cg_html = new QColorGroup( a->text, a->back, a->html.light(),
			     a->html.dark(), a->text, a->back, a->text );
  a->cg_graphic = new QColorGroup( a->text, a->back, a->graphic.light(),
				a->graphic.dark(), a->graphic, a->back, a->text );
  a->cg_packed = new QColorGroup( a->text, a->back, a->packed.light(),
			       a->packed.dark(), a->packed, a->back, a->text );
  a->cg_ordered = new QColorGroup( a->text, a->back, a->ordered.light(),
				a->ordered.dark(), a->ordered, a->back, a->text );
  a->cg_error = new QColorGroup( a->text, a->back, a->error.light(),
			      a->error.dark(), a->error, a->back, a->text );
  a->cg_other = new QColorGroup( a->text, a->back, a->other.light(),
			      a->other.dark(), a->other, a->back, a->text );

  a->br_html->setColor( a->html );
  a->br_graphic->setColor( a->graphic );
  a->br_packed->setColor( a->packed );
  a->br_ordered->setColor( a->ordered );
  a->br_error->setColor( a->error );
  a->br_other->setColor( a->other );
}
