#include "wtview.h"
#include "../webtree.h"
#include <qtooltip.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <math.h>

#include <values.h>	/* for MAX/MININT */

/* dynamic tooltips */

class WtTip : public QToolTip
{
public:
  WtTip( QWidget *parent );
  virtual ~WtTip() { ; }

protected:
  void maybeTip( const QPoint &pos );
};

WtTip::WtTip( QWidget *parent )
  : QToolTip( parent )
{
}

void WtTip::maybeTip( const QPoint &pos )
{
  WtViewWin *v = (WtViewWin*)parentWidget();
  TreeInfo *t = v->treeInfoUnder( pos );

  if( t )
  {
    QRect rc( -(v->offset_x) + (t->layout.x * v->zoom ) / 100 - 1,
	      -(v->offset_y) + (t->layout.y * v->zoom ) / 100 - 1,
	      (t->layout.width * v->zoom ) / 100 + 2,
	      (t->layout.height * v->zoom ) / 100 + 2);
    QString s;
    s.sprintf( "%s [%s]", t->url, t->title );
    tip( rc, s );
  }
}

/*
 *
 *
 */

WtView::WtView( Document *d, QWidget *parent, const char *name, bool over )
  : QWidget( parent, name )
{
  isOverview = over;

  /* create horizontal scrollbar */
  hscroll = new QScrollBar( QScrollBar::Horizontal, this );
  CHECK_PTR( hscroll );

  /* create vertical scrol bar*/
  vscroll = new QScrollBar( QScrollBar::Vertical, this );
  CHECK_PTR( vscroll );

  /* create viewwdget */
  viewwin = new WtViewWin( d, this, NULL, over );
  CHECK_PTR( viewwin );

  /* connect scrollbar events */
  connect( hscroll, SIGNAL( valueChanged( int ) ),
	   viewwin, SLOT( scrollHorizTo( int ) ) );
  connect( vscroll, SIGNAL( valueChanged( int ) ),
	   viewwin, SLOT( scrollVertTo( int ) ) );

  viewwin->move( 0, 0 );
  viewwin->show();
  show();
}

WtView::~WtView()
{
  delete viewwin;
}

/*
  after a resize or after the size of the window content has changed
  this function calculates if scrollbars are needed and sets their
  positions
*/

void WtView::resizeScrollBars()
{
  int w, h, ww, hh;
  bool hscr, vscr;
  int sw, sh;

  dbg( DI, DBG_VIEW | 3, "resizing scrollbars" );

  sw = viewwin->realSize_w;
  sh = viewwin->realSize_h;

  w = width();
  h = height();

  if( !isOverview && appear.lens_effect )
  {
    sw += w;
    sh += h;
  }

  /*
    do we need scrollbars ?
  */

  hscr = sw > w;
  vscr = sh > h;

  if( hscr && h - SC_WIDTH < sh )
    vscr = true;
  if( vscr && w - SC_WIDTH < sw )
    hscr = true;

  /*
    turn on/off horiz scrollbar
  */

  if( hscr )
  {
    hscroll->show();
    hh = h - SC_WIDTH;
  } 
  else
  {
    viewwin->scrollHorizTo( 0 );
    hscroll->hide();
    hh = h;
  }
  
  /*
    turn on/off vert scrollbar
  */

  if( vscr )
  {
    vscroll->show();
    ww = w - SC_WIDTH;
  }
  else
  {
    viewwin->scrollVertTo( 0 );
    vscroll->hide();
    ww = w;
  }

  if( vscr )
  {
    vscroll->setRange( 0, sh - hh );
    vscroll->setSteps( hh / 10, hh );
  }
  if( hscr )
  {
    hscroll->setRange( 0, sw - ww );
    hscroll->setSteps( ww / 10 , ww);
  }

  if( (ww != viewwin->width()) || (hh != viewwin->height()) )
  {
    dbg( DI, DBG_VIEW | 3, "resizing viewwin" );
    viewwin->resize( ww, hh );
  }

  if( hscr )
    hscroll->setGeometry( 0, h - SC_WIDTH, vscr ? w - SC_WIDTH : w, SC_WIDTH );
  if(vscr)
    vscroll->setGeometry( w - SC_WIDTH, 0, SC_WIDTH, hscr ? h - SC_WIDTH : h );
}

/*
  after resize check for scrollbars
*/

void WtView::resizeEvent( QResizeEvent * )
{
  if( isOverview )
    viewwin->overviewCheckSize();

  resizeScrollBars();
}

/* 
   ctor
   initializes the view and 
*/

WtViewWin::WtViewWin( Document *d, QWidget *parent, const char *name, bool over )
  : KDNDWidget( parent, name, WResizeNoErase ),
                         /* do not erase window contents on resize */
    pop_child_arr( 50 ), /* initialize the arrays with 50 elements */
    pop_father_arr( 50 )
{
  isOverview = over;
  doc = d;
  stickmlinks = FALSE;

  draggingEnabled = FALSE;
  setMouseTracking( TRUE );

  last_ti = NULL;
  selected_ti = NULL;

  zoom = 100;

  offset_x = 0;
  offset_y = 0;

  signal_choke = FALSE;
  pixBuffer = TRUE;

  infoList.setAutoDelete( TRUE );

  pressed = FALSE;

  if( isOverview )
    tooltip = new WtTip( this );
  else
    tooltip = NULL;

  /*
    create Popup Menu
  */

  popup = new KPopupMenu();
  pop_child = new QPopupMenu();
  pop_father = new QPopupMenu();

  popup->setTitle( klocale->translate("Selected Node:") );
  popup->insertItem( klocale->translate("Show Info"), ID_INFO );
  popup->insertItem( klocale->translate("Set as new location"), ID_NEW_LOCATION );
  popup->insertSeparator();
  popup->insertItem( klocale->translate("Save Page..."), ID_SAVE_PAGE );
  popup->insertSeparator();
  popup->insertItem( klocale->translate("Center child links"), pop_child, ID_POP_CHILD );
  popup->insertItem( klocale->translate("Center father links"), pop_father, ID_POP_FATHER );
  popup->insertSeparator();
  popup->insertItem( klocale->translate("Stop loading at Node"), ID_STOP_NODE );
  popup->insertItem( klocale->translate("Start loading at Node"), ID_START_NODE );
  popup->insertSeparator();
  popup->insertItem( klocale->translate("Collapse Node"), ID_COLLAPSE_NODE );
  popup->insertItem( klocale->translate("Expand Node"), ID_EXPAND_NODE );

  /* sigs and slots */
  connect( popup, SIGNAL( activated( int ) ),
	   this, SLOT( popupSelected( int ) ) );
  connect( pop_child, SIGNAL( activated( int ) ),
	   this, SLOT( childPopupSelected( int ) ) );
  connect( pop_father, SIGNAL( activated( int ) ),
	   this, SLOT( fatherPopupSelected( int ) ) );
}

WtViewWin::~WtViewWin()
{
  if( tooltip )
    delete tooltip;

  delete pop_child;
  delete pop_father;
  delete popup;
}

void WtViewWin::childPopupSelected( int item )
{
  TreeInfo *t;
  t = pop_child_arr[ item ];
  centerTreeInfo( t );
}

void WtViewWin::fatherPopupSelected( int item )
{
  TreeInfo *t;
  t = pop_father_arr[ item ];
  centerTreeInfo( t );
}

void WtViewWin::popupSelected( int item )
{
  TreeInfo *t;
  switch( item )
  {
    case ID_NEW_LOCATION:
      t = last_ti;
      resetViewWin();
      emit setLocation( t->url );
      break;

    case ID_START_NODE:
      ::context = doc;
      html_startLoadingNode( last_ti );
      emit ensureLoading();
      break;

    case ID_STOP_NODE:
      ::context = doc;
      html_stopLoadingNode( last_ti );
      break;

    case ID_COLLAPSE_NODE:
      ::context = doc;
      layout_collapseNode( last_ti );
      break;

    case ID_EXPAND_NODE:
      ::context = doc;
      layout_expandNode( last_ti );
      break;

    case ID_INFO:
      info( last_ti );
      break;

    case ID_SAVE_PAGE:
      emit savePage( last_ti );

    default:
      dbg( DI, DBG_VIEW | 1, "Haeaeaeae ??" );
      break;
  }
}

void WtViewWin::info( TreeInfo *ti )
{
  DlgInfo *i = new DlgInfo( ti );
  infoList.append( i );
  i->setForm();
  i->show();
}

void WtViewWin::fillPopupMenus()
{
  QString s;
  NextTreeInfo *nt;
  bool coll, nloaded, nactive;
  char *t;
  int i;

  pop_child->clear();
  pop_father->clear();

  if( last_ti == NULL )
    return;

  /* fill child list */
  for( nt = last_ti->nextTree; nt ; nt = nt->next )
  {
    t = nt->tree->layout.text;
    coll = nt->tree->layout.collapsed;
    nloaded = !nt->tree->loaded;
    nactive = !nt->tree->active;

    if( coll || nloaded || nactive )
    {
      QString s("(");
      if( coll ) s += "C";
      if( nloaded ) s += "L";
      if( nactive ) s += "A";
      s += ") ";
      s += t;
      i = pop_child->insertItem( s );
      pop_child->setItemEnabled( i, FALSE );
    }
    else
    {
      i = pop_child->insertItem( t );
      if( i >= (int)pop_child_arr.size() )
	pop_child_arr.resize( i + 10 );
      pop_child_arr[i] = nt->tree;
    }
  }

  /* fill father list */
  for( nt = last_ti->nextFather; nt ; nt = nt->next )
  {
    t = nt->tree->layout.text;
    coll = nt->tree->layout.collapsed;
    nloaded = !nt->tree->loaded;
    nactive = !nt->tree->active;

    if( coll || nloaded || nactive )
    {
      QString s("(");
      if( coll ) s += "C";
      if( nloaded ) s += "L";
      if( nactive ) s += "A";
      s += ") ";
      s += t;
      i = pop_father->insertItem( s );
      pop_father->setItemEnabled( i, FALSE );
    }
    else
    {
      i = pop_father->insertItem( t );
      if( i >= (int)pop_father_arr.size() )
	pop_father_arr.resize( i + 10 );
      pop_father_arr[i] = nt->tree;
    }
  }

}

void WtViewWin::picEnsureLoading()
{
  emit ensureLoading();
}

/*
  this function computes the lens effect
*/

int WtViewWin::lensTransform( int value, int maxval, double pot )
{
  double dmaxval = (double)maxval;
  int first_half = ( value < ( maxval / 2 ) );
  double maxval_exp = pow( dmaxval / 2, pot );
  double val_exp = pow( (double)( first_half ? value : maxval - value ), pot  ) * dmaxval /
    ( maxval_exp * 2 );
  return (int)( first_half ? val_exp : dmaxval - val_exp );
}

void WtViewWin::zoomChanged( int value )
{
  int mx, my, ox, oy, w, h, rw, rh;

  zoom = value;

  w = width();
  h = height();

  realSize_w = rw = (size_w * zoom) / 100;
  realSize_h = rh = (size_h * zoom) / 100;

  /* TODO fix broken algorithm. is not ok yet */

  mx = width() / 2;
  my = height() / 2;

  ox = ( offset_x * zoom ) / 100 + mx - ( mx * zoom ) / 100;
  oy = ( offset_y * zoom ) / 100 + mx - ( my * zoom ) / 100;

  offset_x = max( 0, min( rw - w, ox ) );
  offset_y = max( 0, min( rh - h, oy ) );

  /* end TODO */

  signal_choke = TRUE;

  ( (WtView*)parentWidget() )->resizeScrollBars();
  ( (WtView*)parentWidget() )->hscroll->setValue( offset_x );
  ( (WtView*)parentWidget() )->vscroll->setValue( offset_y );

  signal_choke = FALSE;

  myUpdate();
}

void WtViewWin::setSize( int w, int h )
{
  /*  QWidget *p;*/
  size_w = w;
  size_h = h;
  
  if( isOverview )
    overviewCheckSize();
  else
  {
    realSize_w = (size_w * zoom) / 100;
    realSize_h = (size_h * zoom) / 100;
  }

  signal_choke = TRUE;
  ( (WtView*)parentWidget() )->resizeScrollBars();
  signal_choke = FALSE;

  myUpdate();
}

void WtViewWin::paintEvent( QPaintEvent *e )
{
  int off_x = 0, off_y = 0, w, h, pntsz;
  int rx1, ry1, rx2, ry2;
  int sx1, sy1, sx2, sy2;
  int orig_bw, orig_bh;
  double f1, f2;
  TreeInfo *t;
  NextTreeInfo *nt = NULL;
  bool iscollapsed;
  bool isnotactive;
  QFont fnt;
  Appearance *a = &appear;
  QColorGroup *cg;
  QBrush *br;
  QRect clprc, boxrc, box2rc;
  QPainter p;
  QPixmap *pm = NULL;


  if( doc->html.root == NULL )
    return;

  w = width();
  h = height();
  clprc = e->rect();

  /*
    if pix buffer is set then draw the whole thing into a pixmap first and then
    blit it on screen
  */

  if( pixBuffer )
  {
    QRect trc;
    QSize s( clprc.size() );

    /*
      sometimes i get an invalid cliprect if a have called repaint directly 
      after myUpdate(). to avoid debug output from QPixmap & QPainter 
      the following hack is implemented 
      hack start
    */

    if( s.width() < 1 ) s.setWidth( 1 );
    if( s.height() < 1 ) s.setHeight( 1 );
    /* hack end */

    pm = new QPixmap( s );
    pm->fill( a->back );

    p.begin( pm );
    off_x = clprc.x();
    off_y = clprc.y();
    trc.setRect( 0, 0, clprc.width(), clprc.height() );
    clprc = trc;
  }

  fnt = a->font;
  
  /*
    draw links
  */
  
  for( t = doc->html.root; t != NULL; t = t->nextList )
    if( isOverview || !a->mouseLinks || last_ti == t )
      drawLinksFor( t, &p, off_x, off_y );

  p.setPen( QPen( a->line, 0 ) );
  
  /*
    draw the boxes
  */

  for( t = doc->html.root; t != NULL; t = t->nextList )
  {
    iscollapsed = isnotactive = FALSE;

    if ( !t->layout.draw )
      continue;

    rx1 = -offset_x + (t->layout.x * zoom ) / 100;
    ry1 = -offset_y + (t->layout.y * zoom ) / 100;

    /*
      add some extra space if we are in lens mode
    */

    if( !isOverview && a->lens_effect )
    {
      rx1 += w / 2;
      ry1 += h / 2;
    }

    rx2 = rx1 + (t->layout.width * zoom ) / 100;
    ry2 = ry1 + (t->layout.height * zoom ) / 100;

    if( !isOverview && a->lens_effect )
    {
      orig_bw = rx2 - rx1;
      orig_bh = ry2 - ry1;
      rx1 = lensTransform( rx1, w, a->lens_exp );
      ry1 = lensTransform( ry1, h, a->lens_exp );
      rx2 = lensTransform( rx2, w, a->lens_exp );
      ry2 = lensTransform( ry2, h, a->lens_exp );
      if( rx1 < 0 ) rx1 = 0;
      if( ry1 < 0 ) ry1 = 0;
      if( rx2 < rx1 + 2 ) continue; /*rx2 = rx1 + 4;*/
      if( ry2 < ry1 + 2 ) continue; /*ry2 = ry1 + 4;*/
    }

    rx1 -= off_x;
    rx2 -= off_x;
    ry1 -= off_y;
    ry2 -= off_y;

    /*
      drawing optimization: draw boxes and text only if they are in the
      clipping rectangle
    */
    
    if( rx2 <= clprc.left() || rx1 >= clprc.right() ||
	ry2 <= clprc.top() || ry1 >= clprc.bottom() )
      continue;


    sx1 = rx1 + BORDER_WIDTH;
    sy1 = ry1 + BORDER_WIDTH;
    sx2 = rx2 - BORDER_WIDTH * 2;
    sy2 = ry2 - BORDER_WIDTH * 2;

    /*
      set color of box
    */

    switch( t->doctype )
    {
      case TextFile:
      case HTMLDoc: 
	cg = a->cg_html;
	br = a->br_html;
	break;
      case Graphic:
	cg = a->cg_graphic;
	br = a->br_graphic;
	break;
      case PackedFile:
	cg = a->cg_packed;
	br = a->br_packed;
	break;
      case Error:
	cg = a->cg_error;
	br = a->br_error;
	break;
      case Unknown:
      default:
	if( t->loaded )
	{
	  cg = a->cg_other;
	  br = a->br_other;
	}
	else
	{
	  cg = a->cg_ordered;
	  br = a->br_ordered;
	}
    }

    /*
      check if this is a collapsed node or an inactive one
    */

    if( t->nextTree )
      for( nt = t->nextTree; nt ; nt = nt->next )
	if( nt->layout.type & DIRECT )
	{
	  if( nt->tree->layout.collapsed )
	    iscollapsed = TRUE;
	  if( nt->tree->active == 0 )
	    isnotactive = TRUE;
	  if( iscollapsed && isnotactive )
	    break;
	}

    /*
      draw shaded panel or plain box ( dependend on size of box and overview )
     */
 
    if( isOverview ||
	( ( rx2 - rx1 ) < ( BORDER_WIDTH + 4 ) * 2 ) ||
	( ( ry2 - ry1 ) < ( BORDER_WIDTH + 4 ) * 2 ) )
    {
      if( t == selected_ti )
      {
	QBrush b( a->line );
	qDrawPlainRect( &p, rx1, ry1, rx2 - rx1, ry2 - ry1, a->back, 1,  &b );
      }
      else
	qDrawPlainRect( &p, rx1, ry1, rx2 - rx1, ry2 - ry1, a->line, 1,  br );
    }
    else
      qDrawShadePanel( &p, rx1, ry1, rx2 - rx1, ry2 - ry1, *cg, FALSE, BORDER_WIDTH - 1, br );

    /*
      if this node is collapsed then draw border around it
    */
    
    if( iscollapsed || isnotactive )
    {
      uint coff = 1;
      if( zoom > 15 )
	coff++;

      p.setPen( QPen( a->collapsed, coff ) );
      p.setBrush( NoBrush );
      p.drawRect( rx1 - coff, ry1 - coff, rx2 - rx1 + 2 * coff, ry2 - ry1 + 2 * coff );
      if( isnotactive )
      {
	p.setPen( a->line );
	p.drawLine( rx1, ry1, rx2, ry2 );
	p.drawLine( rx2, ry1, rx1, ry2 );
      }
    }

    /*
      calc pointsize dependend on the real boxsize (if lens effect is applyed
    */

    pntsz = t->layout.fontsize * zoom;
    if( a->lens_effect )
    {
      f1 = ( (double)( rx2 - rx1 ) ) / ( (double)orig_bw );
      f2 = ( (double)( ry2 - ry1 ) ) / ( (double)orig_bh );
      pntsz = (int)( ( f1 < f2 ) ? ( (double)pntsz ) * f1  : ( (double)pntsz ) * f2 );
    }
    pntsz /= 100;
    
    /*
      draw box text
    */

    if( pntsz > 1 )
    {

      fnt.setPointSize( pntsz );
      p.setFont( fnt );

      if( selected_ti == t )
	p.setPen( a->back );
      else
	p.setPen( a->text );

      p.drawText( sx1, sy1, sx2 - sx1, sy2 - sy1, AlignCenter | WordBreak, (const char*)&t->layout.text );

      if( selected_ti == t ) p.setPen( a->line );
    }
  }

#ifdef DEBUG_LAYOUT
  /*BKJ start*/
  /* just a debug function */
  int gc; struct GridBox *g;

  for (gc=0; gc<MAXGRIDLINES; gc++)
    for (g=context->layout.grid[gc]; g; g = g->nextGrid)
      {
	boxrc.setRect( -offset_x - off_x + (g->pos * zoom ) / 100,
		       -offset_y - off_y + (gc*GRIDDY * zoom ) / 100,
		       ((g->size<20 ? 20 : g->size-20)* zoom ) / 100,
		       (GRIDDY/2 * zoom ) / 100);

	if( !clprc.intersects( boxrc ) )
	  continue;

	box2rc.setRect( boxrc.x() + BORDER_WIDTH, boxrc.y() + BORDER_WIDTH,
			boxrc.width() - 2*BORDER_WIDTH, boxrc.height() - 2*BORDER_WIDTH );
	
	qDrawPlainRect( &p, boxrc.x(), boxrc.y(), boxrc.width(), boxrc.height(),
			a->error, 1 );
      }
#endif /*BKJ end */

  p.end();

  if( pixBuffer )
  {
    bitBlt( this, off_x, off_y, pm );
    delete pm;
  }

}

// Calculate rectangle for text with pointsize fontsize
void WtViewWin::getSizeForText( int *width, int *height, char *text, int fontsize )
{
  QFont fnt( ::appear.font );
  fnt.setPointSize( fontsize );
  QFontMetrics fm( fnt );
  *height = fm.height();
  *width = fm.width( text );
}


void WtViewWin::mousePressEvent( QMouseEvent *m )
{
  TreeInfo *ti;

  dbg( DI, DBG_VIEW | 3, "mouse pressed");

  /*
    if the view is a overview then select node and emit message to
    center the main view for this node
  */

  if( isOverview )
  {
    selected_ti = treeInfoUnder( m->pos() );

    if( selected_ti)
    {
      repaint( -offset_x + (selected_ti->layout.x * zoom ) / 100,
	       -offset_y + (selected_ti->layout.y * zoom ) / 100,
	       (selected_ti->layout.width * zoom ) / 100,
	       (selected_ti->layout.height * zoom ) / 100, FALSE );
      emit treeInfoSelected( selected_ti );
    }
  }
  else
  {
    if( m->button() != MidButton )
    {
      ti = treeInfoUnder( m->pos() );

      if( ::appear.mouseLinks )
      {
	/* clear old links */
	if( last_ti )
	{
	  QRect rc = calcCliprectForLinks( last_ti );
	  last_ti = NULL;
	  if( rc.width() > 0 || rc.height() > 0 )
	    repaint( rc, FALSE );
	}
      
	if( ti )
	{
	  last_ti = ti;
	  QPainter p( this );
	  drawLinksFor( last_ti, &p );
	  stickmlinks = TRUE;
	}
	else
	{
	  stickmlinks = FALSE;
	}
      }
      else
	last_ti = ti;
    }
      
    if(last_ti != NULL && m->button() == RightButton)
    {
      fillPopupMenus();
      popup->popup(QCursor::pos());
      popup->exec();
    }
    else if(m->button() == MidButton)
    {
      draggingEnabled = TRUE;
      grabMouse( SizeAllCursor );
      dragContent();
    }
    else	// left button may be the start of a drag...
    {
      pressed = TRUE;
      press_x = m->pos().x();
      press_y = m->pos().y();
      dragUrl = ti;
    }
  }
}

void WtViewWin::mouseMoveEvent( QMouseEvent *m )
{
  TreeInfo *t;
  Appearance *a = &appear;

  if( isOverview )
    return;

  if( ( t = treeInfoUnder( m->pos() ) ) != last_ti )
  {
    if( t != NULL )
    {
      emit setStatusUrl( t->url );
      if( a->mouseLinks && !stickmlinks )
      {
	if( last_ti )
	{
	  QRect rc = calcCliprectForLinks( last_ti );
	  last_ti = NULL;
	  if( rc.width() > 0 || rc.height() > 0 )
	    repaint( rc, FALSE );
	}
	QPainter p( this );
	drawLinksFor( t, &p );
      }
    }
    else
    {
      emit setStatusUrl( "" );
      if( a->mouseLinks && !stickmlinks )
	if( last_ti )
	{
	  QRect rc = calcCliprectForLinks( last_ti );
	  last_ti = NULL;
	  if( rc.width() > 0 || rc.height() > 0 )
	    repaint( rc, FALSE );
	}
	else
	  repaint( FALSE ); // should not happen
    }
    if(!stickmlinks)
      last_ti = t;
  }
  if( draggingEnabled )
    dragContent();

  KDNDWidget::mouseMoveEvent( m );
}

void WtViewWin::mouseReleaseEvent( QMouseEvent *e )
{
  TreeInfo *so;
  if( isOverview )
  {
    if( selected_ti )
    {
      so = selected_ti;
      selected_ti = NULL;
      repaint( -offset_x + (so->layout.x * zoom ) / 100,
	       -offset_y + (so->layout.y * zoom ) / 100,
	       (so->layout.width * zoom ) / 100,
	       (so->layout.height * zoom ) / 100, FALSE );
    }
  }
  if( draggingEnabled )
    releaseMouse();
  draggingEnabled = FALSE;

  pressed = FALSE;

  KDNDWidget::mouseReleaseEvent( e );
}

void WtViewWin::dndMouseReleaseEvent( QMouseEvent * )
{
  pressed = FALSE;
}

void WtViewWin::dndMouseMoveEvent( QMouseEvent *m )
{
  int x, y;

  if( !pressed || dragUrl == NULL )
    return;

  x = m->pos().x();
  y = m->pos().y();
  if( abs( x - press_x ) > Dnd_X_Precision || abs( y - press_y ) > Dnd_Y_Precision )
  {
    QPoint p = mapToGlobal( m->pos() );
    QString data = dragUrl->url;
    int dx = -::dndPixmap->width() / 2;
    int dy = -::dndPixmap->height() / 2;

    startDrag( new KDNDIcon( *::dndPixmap, p.x() + dx, p.y() + dy ),
	       data.data(), data.length(), DndURL, dx, dy );
  }
}


// drags the window contents to a specific position. 
void WtViewWin::dragContent()
{
  int new_x, new_y, off_x, off_y;
  int w, h, rw, rh;
  QPoint /*gpos,*/ mpos;

  mpos = mapFromGlobal( QCursor::pos() );
  
  w = width();
  h = height();

  rw = realSize_w;
  rh = realSize_h;

  off_x = ( (w * w) / (2 * rw) );
  off_y = ( (h * h) / (2 * rh) );

  new_x = ( (mpos.x() - off_x) * rw / w );
  new_y = ( (mpos.y() - off_y) * rh / h );

  new_x = max( 0, min( rw - w, new_x ) );
  new_y = max( 0, min( rh - h, new_y ) );

  offset_x = new_x;
  offset_y = new_y;

  signal_choke = TRUE;	// avoid painting twice ...

  ( (WtView*)parentWidget() )->hscroll->setValue( new_x );
  ( (WtView*)parentWidget() )->vscroll->setValue( new_y );

  signal_choke = FALSE;

  myUpdate();
}

void WtViewWin::leaveEvent( QEvent * )
{
  emit setStatusUrl( "" );

  if( ::appear.mouseLinks && last_ti && !stickmlinks )
  {
    QRect rc = calcCliprectForLinks( last_ti );
    last_ti = NULL;
    if( rc.width() > 0 || rc.height() > 0 )
      repaint( rc, FALSE );
  }

  if( !stickmlinks )
    last_ti = NULL;
}

// returns the treeinfo under the given position ( in window coordinates )
TreeInfo* WtViewWin::treeInfoUnder( const QPoint& mpos )
{
  TreeInfo *t;
  QRect rc;
  int x1, y1, x2, y2;
  int w, h;
  Appearance *a = &appear;

  w = width();
  h = height();

  for(t = doc->html.root; t != NULL; t = t->nextList)
  {
    x1 = -offset_x + (t->layout.x * zoom ) / 100;
    y1 = -offset_y + (t->layout.y * zoom ) / 100;

    /*
      add some extra space if we are in lens mode
    */

    if( !isOverview && a->lens_effect )
    {
      x1 += w / 2;
      y1 += h / 2;
    }

    x2 = x1 + (t->layout.width * zoom ) / 100;
    y2 = y1 + (t->layout.height * zoom ) / 100;

    if( !isOverview && a->lens_effect )
    {
      x1 = lensTransform( x1, w, a->lens_exp );
      y1 = lensTransform( y1, h, a->lens_exp );
      x2 = lensTransform( x2, w, a->lens_exp );
      y2 = lensTransform( y2, h, a->lens_exp );
      if( x1 < 0 ) x1 = 0;
      if( y1 < 0 ) y1 = 0;
      if( x2 < x1 + 2 ) continue; /*rx2 = rx1 + 4;*/
      if( y2 < y1 + 2 ) continue; /*ry2 = ry1 + 4;*/
    }

    if( mpos.x() >= x1 && mpos.x() <= x2 && mpos.y() >= y1 && mpos.y() <= y2 )
      break;
  }

  return t;
}


/*
  TODO: drawing optimization
  there is a problem if the real size is bigger than 16bit because the
  COORDINATE system only understands 16bit signed ints but the layout-
  space can be bigger than that :-(
*/

void WtViewWin::drawLinksFor( TreeInfo *t, QPainter *p, int ox, int oy )
{
  NextTreeInfo *l;
  int x1, y1, x2, y2;
  int i, w, h;
  Appearance *a = &appear;
  QPointArray pa( 3 );
  int x11, y11;

  w = width();
  h = height();

  p->setPen( QPen( a->line, 0 ) );
  p->setBrush( QBrush( a->line ) );

  for ( l = t->nextTree; l != NULL; l = l->next )
    if ( l->layout.type & USED )
    {
#ifdef DEBUG_LAYOUT
      /*BKJ*/
      if (l->layout.type & DIRECT)
	p->setPen( QPen( a->error, 0 ) );
      else
	/*BKJ*/
#endif
      p->setPen( QPen( a->line, 0 ) );

      /*
	if we are a overview and the link is not direct goto next
      */

      if( isOverview && !(l->layout.type & DIRECT) )
	continue;

      /*
	draw the lines
      */
      
      for( i = 0; i < LINEMAX - 1; i++ )
      {
	x1 = -offset_x + l->layout.x[i] * zoom / 100;
	y1 = -offset_y + l->layout.y[i] * zoom / 100;
	x2 = -offset_x + l->layout.x[i+1] * zoom / 100;
	y2 = -offset_y + l->layout.y[i+1] * zoom / 100;
	
	if( !isOverview && a->lens_effect )
	{
	  x1 += w / 2;
	  y1 += h / 2;
	  x2 += w / 2;
	  y2 += h / 2;
	}

	x1 -= ox;
	x2 -= ox;
	y1 -= oy;
	y2 -= oy;

	if( !isOverview && appear.lens_effect )
	{
	  x1 = lensTransform( x1, w, a->lens_exp );
	  y1 = lensTransform( y1, h, a->lens_exp );
	  x2 = lensTransform( x2, w, a->lens_exp );
	  y2 = lensTransform( y2, h, a->lens_exp );

	  x11 = x2;
	  y11 = y2;

/*	  x1 = min( max( ) );
	  if( x1 < 0 ) x1 = 0;
	  if( y1 < 0 ) y1 = 0;
	  if( x2 > w ) x2 = w;
	  if( y2 < y1 + 2 );*/
	}

	/*
	  do clipping
	*/

	if( x1 == x2 )
	{
	  /*
	    vertical line
	  */
	  
	  if( x1 < 0 || x1 > w )
	    continue;

	  y1 = min( max( y1, 0 ), h);
	  y2 = min( max( y2, 0 ), h);
	  if( y1 == y2 )
	    continue;
	}
	else
	{
	  /*
	    horizintal line
	  */

	  if( y1 < 0 || y1 > h )
	    continue;

	  x1 = min( max( x1, 0 ), w );
	  x2 = min( max( x2, 0 ), w );
	  if( x1 == x2 )
	    continue;
	}

	p->drawLine( x1, y1, x2, y2 );
//	debug("Line %d %d %d %d drawn", x1, y1, x2, y2 );
	
      }

      x2 = x11;
      y2 = y11;

      if( zoom > 20 && !isOverview && 
	  x2 >= 0 && x2 <= w && y2 >= 0 && y2 <= h )
      {
	if( doc->pref.viewtype == leftrightview )
	  pa.setPoints( 3, x2, y2,
			(x2>x1)?x2 - ARROW_HEIGHT:x2 + ARROW_HEIGHT, y2 + ARROW_WIDTH,
			(x2>x1)?x2 - ARROW_HEIGHT:x2 + ARROW_HEIGHT, y2 - ARROW_WIDTH );
	else
	  pa.setPoints( 3, x2, y2,
			x2 + ARROW_WIDTH, (y2>y1)?y2 - ARROW_HEIGHT:y2 + ARROW_HEIGHT,
			x2 - ARROW_WIDTH, (y2>y1)?y2 - ARROW_HEIGHT:y2 + ARROW_HEIGHT );
	p->drawPolygon( pa );
      }
    }
}

// calculates the clipping rectangle for a given node
// TODO: wrong cliprects if big layoutspace and huge zoomlevel :-(
QRect WtViewWin::calcCliprectForLinks( TreeInfo *t )
{
  NextTreeInfo *l = NULL;;
  int x, y, x1, y1, x2, y2, i;
  bool rectChanged = TRUE;

  x1 = y1 = MAXINT;
  x2 = y2 = MININT;

  for (l=t->nextTree;l; l=l->next)
    if ( l->layout.type & USED )
      for (i = 0; i < LINEMAX; i++)
	{
	  x = -offset_x + l->layout.x[i] * zoom / 100;
	  y = -offset_y + l->layout.y[i] * zoom / 100;
	  
	  x1 = min(x1, x);
	  y1 = min(y1, y);
	  x2 = max(x2, x);
	  y2 = max(y2, y);
	}

  if(!rectChanged)
    x1 = y1 = x2 = y2 = 0;
  else
  {
    bool b = ( doc->pref.viewtype == leftrightview );
    // don't make rectangle bigger than the window
    x1 = max( 0, x1 - ( b ? ARROW_HEIGHT: ARROW_WIDTH ) );
    y1 = max( 0, y1 - ( b ? ARROW_WIDTH: ARROW_HEIGHT ) );
    x2 = min( width(), x2 + ( b ? ARROW_HEIGHT: ARROW_WIDTH ) );
    y2 = min( height(), y2 + ( b ? ARROW_WIDTH: ARROW_HEIGHT ) );
  }

  return QRect( x1, y1, x2 - x1 + 1, y2 - y1 + 1 );
}

void WtViewWin::resizeEvent( QResizeEvent * )
{
  myUpdate();
}

void WtViewWin::scrollHorizTo( int v )
{
  int oldoff, w, h;
  w = width();
  h = height();
  oldoff = offset_x;
  offset_x = v;

  if( signal_choke )
    return;

  dbg( DI, DBG_VIEW | 3, "hscroll to %d", v );

  if( &appear.lens_effect && !isOverview )
  {
    myUpdate();
    return;
  }

  bitBlt( this, oldoff - v, 0, this );

  if( oldoff > v )
    repaint( 0, 0, oldoff - v, h, FALSE );
  else if( oldoff < v )
    repaint( w - (v - oldoff), 0, v - oldoff, h, FALSE );
}

void WtViewWin::scrollVertTo( int v )
{
  int oldoff;

  oldoff = offset_y;
  offset_y = v;

  if( signal_choke )
    return;

  dbg( DI, DBG_VIEW | 3, "vscroll to %d", v );

  if( &appear.lens_effect && !isOverview )
  {
    myUpdate();
    return;
  }

  bitBlt( this, 0, oldoff - v, this );

  if( oldoff > v )
    repaint( 0, 0, width(), oldoff - v, FALSE );
  else if( oldoff < v )
    repaint( 0, height() - (v - oldoff), width(), v - oldoff, FALSE );
}

// sends Expose Events to the window
// is like the original except that it doesn't clear the background
void WtViewWin::myUpdate( int x, int y, int w, int h )
{
  XEvent e;

  if( !isVisible() )
    return;

  dbg( DI, DBG_VIEW | 3, "update rect: %d, %d -> %d, %d", x, y, w, h);

  e.type = Expose;
  e.xexpose.x = x;
  e.xexpose.y = y;
  e.xexpose.width = (w == 0) ? width() : w ;
  e.xexpose.height = (h == 0) ? height() : h;
  e.xexpose.window = winId();
  e.xexpose.display = x11Display();

  XSendEvent( x11Display(), winId(), FALSE, 0, &e );
}

// try to center the given TreeInfo into the window
void WtViewWin::centerTreeInfo( TreeInfo *t )
{
  int w, h, rw, rh, cx, cy, cw, ch, mx, my, ox, oy;

  w = width();
  h = height();

  rw = realSize_w;
  rh = realSize_h;

  if( !isOverview && appear.lens_effect )
  {
    rw + w;
    rh += h;
  }

  cx = (t->layout.x * zoom ) / 100;
  cy = (t->layout.y * zoom ) / 100;
  cw = (t->layout.width * zoom ) / 100;
  ch = (t->layout.height * zoom ) / 100;

  mx = w / 2;
  my = h / 2;

  ox = (cx + cw / 2) - mx;
  oy = (cy + ch / 2) - my;

  dbg( DI, DBG_VIEW | 3, "new offset: %d, %d", ox, oy );

  offset_x = max( 0, min( rw - w, ox ) );
  offset_y = max( 0, min( rh - h, oy ) );

  if( !isOverview && appear.lens_effect )
  {
    offset_x += w / 2;
    offset_y += h / 2;
  }

  signal_choke = TRUE;	// avoid painting twice ...

  ( (WtView*)parentWidget() )->hscroll->setValue( offset_x );
  ( (WtView*)parentWidget() )->vscroll->setValue( offset_y );

  signal_choke = FALSE;

  myUpdate();
}

/* if we are a overview then ensure that the zoomlevel is adjusted */
/* so that only the horiz/vert scrollbar is visible (if needed) */
void WtViewWin::overviewCheckSize()
{
  QWidget *p;
  int w, h;
  p = parentWidget();
    
  if( doc->pref.viewtype == leftrightview )
  {
    w = p->width();
    if( w <= SC_WIDTH )
      zoom = 1;
    else
    {
      if( (size_h * w / max( 1, size_w )) > p->height() )
	w -= SC_WIDTH;
      
      realSize_h = size_h * w / max( 1, size_w );
      realSize_w = w;
      zoom = 100 * w / max( 1, size_w );
    }
  }
  else
  {
    h = p->height();
    if( h <= SC_WIDTH )
      zoom = 1;
    else
    {
      if( (size_w * h / max( 1, size_h )) > p->width() )
	h -= SC_WIDTH;
      
      realSize_h = h;
      realSize_w = size_w * h / max( 1, size_h );
      zoom = 100 * h / max( 1, size_h );
    }
  }

  if( zoom < 9 )
  {
    zoom = 9;
    realSize_w = (size_w * zoom) / 100;
    realSize_h = (size_h * zoom) / 100;
  }

  dbg( DI, DBG_VIEW | 2, "Zoom: %d Realsize w: %d", zoom, realSize_w );
}

/*
  called bevore a new location is set. currently only the last_ti pointer is set to zero
*/

void WtViewWin::resetViewWin()
{
  last_ti = NULL;
  stickmlinks = FALSE;
}
