#include "../document.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define NO  0
#define YES (!NO)

#define FALSE 0
#define TRUE (!FALSE)

#define MIN(a,b) ( (a)<(b) ? (a) : (b))
#define MAX(a,b) ( (a)<(b) ? (b) : (a))


/* some substitutions for context->... */
#define CL (context->layout)
#define CH (context->html)
#define CP (context->pref)

/* view independent functions */
void layout_checkLinks(struct TreeInfo *node);
void layout_adjustContent(struct TreeInfo *node);
void layout_resetVisited();
void layout_resetMoved();
void layout_setLayoutSpace();
void layout_findAllNewLinks();

void layout_getPath(struct TreeInfo *node, int *path, int i);
int layout_isBefore(int *path1, int *path2);

void layout_initGrid();
void layout_resetGrid();

/* function for the topdown view */
int layout_topdownAddNode(struct TreeInfo *node);
void layout_topdownFindLinks(struct TreeInfo *node);
void layout_topdownShiftGridBoxes(struct GridBox *g);
void layout_topdownShiftGridBoxesBy(struct GridBox *g,int pos);
void layout_topdownShrinkTree();
void layout_topdownRestructureTree();

/* function for the leftright view */
int layout_leftrightAddNode(struct TreeInfo *node);
void layout_leftrightFindLinks(struct TreeInfo *node);
void layout_leftrightShiftGridBoxes(struct GridBox *g);
void layout_leftrightShiftGridBoxesBy(struct GridBox *g,int pos);
void layout_leftrightShrinkTree();
void layout_leftrightRestructureTree();

/*======================================================================================

   B A S I C   -   F U N C T I O N A L I T Y

======================================================================================*/

void layout_init()
  /* initializes module layout in current context                      */
  /* don`t forget to call it once for each context!!!                  */
{
  int i;

  CL.maxw=CL.maxh=0;
  for (i=0; i<MAXGRIDLINES; i++)
    CL.grid[i]=NULL;
}

void layout_free()
  /* free current context of module layout, call it if a context is 
     going to be discarded */
{
  layout_resetGrid();
}

void layout_recalcAll()
{
  struct TreeInfo *n;
  struct NextTreeInfo *l;

  if (!context)
    {
      dbg(DI,DBG_LAYOUT|3,"no context set");
      return;
    }
  
  /* reset gridlines, where the nodes are stored by there position */
  layout_resetGrid();

  if (!CH.root)
    {
      dbg(DI,DBG_LAYOUT|3,"no root node");
      return;
    }

  /* now reset all nodes */
  for (n=CH.root; n; n=n->nextList)
    {
      n->layout.directlinksout=0;
      n->layout.visiblelinksout=0;
      n->layout.linksout=0;
      n->layout.linksin=0;
      n->layout.draw=0;
      n->layout.x=n->layout.y=0;
      /* reset all links in the fatherlist */
      for (l=n->nextFather; l; l=l->next)
	l->layout.type=UNUSED;
      /* reset all links in the nextTree list and delete line entries */
      for (l=n->nextTree; l; l=l->next)
	l->layout.type=UNUSED;
    }

  /* use proper strategy to build tree */
  switch(CP.viewtype)
    {
      case leftrightview:
	for (n=CH.root; n; n=n->nextList)
	  {
	    layout_adjustContent(n);
	    layout_checkLinks(n);
	    if (n->active)
	      {
		if ((CP.viewshowordered || n->loaded) && !n->layout.collapsed)
		  layout_leftrightAddNode(n);
	      }
	  }
	layout_leftrightShrinkTree();
	layout_leftrightRestructureTree();
	layout_findAllNewLinks();
	break;

      case topdownview:
	for (n=CH.root; n; n=n->nextList)
	  {
	    layout_adjustContent(n);
	    layout_checkLinks(n);
	    if (n->active)
	      {
		if ((CP.viewshowordered || n->loaded) && !n->layout.collapsed) 
		  layout_topdownAddNode(n);
	      }
	  }
	layout_topdownShrinkTree();
	layout_topdownRestructureTree();
	layout_findAllNewLinks();
	break;
    }

  /* recalculate used layout space and set the view */
  layout_setLayoutSpace();
} 

void layout_checkLinks(struct TreeInfo *node)
{
  /* checks the links for this node - counts linksin, linksout, directlinksout;
     mark links as USED and/or DIRECT and resets the line structure */

  struct NextTreeInfo *l,*l2;
  int nr;

  dbg(DI,DBG_LAYOUT|3,"checkLinks");

  /* mark all children as not visited */
  for (l=node->nextTree; l; l=l->next)
      l->tree->layout.visited = FALSE;

  nr = 0;
  node->layout.directlinksout=0;
  node->layout.visiblelinksout=0;
  node->layout.visited=TRUE;

  /* run through all children, mark links as valid and/or direct */
  for (l=node->nextTree; l; l=l->next)
    if (/*l->tree->active && */ !l->tree->layout.visited)
    {
      /* mark visited node */
      l->tree->layout.visited=TRUE;
      l->layout.nr=++nr;
      if (!node->layout.collapsed && (CP.viewshowordered || l->tree->loaded) && !l->tree->layout.collapsed && l->tree->active && node->active)
	l->layout.type=USED;

      /* look in fatherlist of child for link to this node */
      for (l2=l->tree->nextFather; l2 && (l2->tree!=node); l2=l2->next);
      if (!l2)
	dbg(DI,DBG_LAYOUT|1,"BIG PROBLEM: missing entry in fatherlist - BUG in HTML - %s - %s", 
	    node->url, l->tree->url);
      else
      {
	if (l->tree->nextFather->tree==node && l->tree!=CH.root)
	{
	  /* it's a direct link downwards */
	  node->layout.directlinksout++;
	  if(l->layout.type&USED)
	    node->layout.visiblelinksout++;
	  l->layout.type|=DIRECT;
	  l2->layout.nr=nr;
	}
	l2->layout.type=l->layout.type;
      }
      l->tree->layout.linksin++;
      l->layout.x[0] = l->layout.y[0] = 0;
    }

  node->layout.linksout=nr;
}

void layout_nodeAdded(struct TreeInfo *node)
/* called by module html as soon as a new node arrives               */
/* this method calculates the new coordinates and eventually         */
/* repositions old ones                                              */
{
  dbg(DI,DBG_LAYOUT | 3, "new node %p",node);
 

  /* look if it isn't collapsed */
  if (node->layout.collapsed)
    return;

  if (!node->loaded)
    if (node->nextFather)
      if (node->nextFather->tree->layout.collapsed)
	{
	  node->layout.collapsed=1;
	  return;
	}

  /* now restructure tree */

  /* if (node->loaded) */
  layout_checkLinks(node);

  /* adjust the outlook of the node (width, height, content) */
  layout_adjustContent(node);

  switch(CP.viewtype)
    {
    case leftrightview:
      if(layout_leftrightAddNode(node))
	{
	  layout_leftrightShrinkTree();
	  layout_leftrightRestructureTree();
	}
      layout_setLayoutSpace();
      layout_findAllNewLinks();
      break;

    case topdownview:
      if(layout_topdownAddNode(node))
	{
	  layout_topdownShrinkTree();
	  layout_topdownRestructureTree();
	}
      layout_setLayoutSpace();
      layout_findAllNewLinks();
      break;
    }
}

void layout_adjustContent(struct TreeInfo *node)
{
  /* TODO: shorten this procedure: e.g. get array with width of each character */

  /* adjusts the content of the box representing the node
     chooses text to display and resizes the box */

  char *s,buffer[NAME_MAXLEN];
  int w,w2,h,p,p2;
  int fsmax, fsmin;

  /* choose title to display, see Preferences pref->nodecontent */

  if( CP.nodecontent == title && node->doctype == HTMLDoc )
  {
    strncpy(node->layout.text,node->title,NAME_MAXLEN-1);
    node->layout.text[NAME_MAXLEN-1]='\x0';
  }
  else
  {
      strncpy(node->layout.text,node->url,NAME_MAXLEN-1);
      node->layout.text[NAME_MAXLEN-1]='\x0';
  }

  /* TODO: where is filename */

  /* remove CR/LF, 'cause it confuses view_getSizeForText */
  for (s=node->layout.text; *s!='\x0'; s++)
    if (*s=='\n' || *s=='\r') *s=' ';

  /*  decrease fontsize according to treelevel, watch out for minimum */

  view_getFontSizes( &fsmax, &fsmin );	/* PHR */
  node->layout.fontsize = MAX( fsmin, fsmax - node->depth - 1 ); /* PHR */

  /* now find out how broad the box will be */
  strcpy(buffer,node->layout.text);
  p=strlen(buffer);
  /* shorten string until it fits into the box */
  view_getSizeForText(&w,&h,buffer,node->layout.fontsize);
  while ((w+6>MAXBOXWIDTH) && (p>0))
    {
      buffer[p-1]='\x0';
      p--;
      view_getSizeForText(&w,&h,buffer,node->layout.fontsize);
    }
  w2=w;

  /* if string is too long for one line, wrap the line and 
     adjust the second line to fit into the box */
  if (strlen(buffer)!=strlen(node->layout.text))
    {
      /* wrap the line between to words if possible */
      s=strrchr(buffer,' ');
      if (s)
	{
	  *(s+1)='\x0';
	  p=strlen(buffer);
	  view_getSizeForText(&w2,&h,buffer,node->layout.fontsize);
	}

      /* shorten the second line to fit into the box */
      strcpy(buffer,node->layout.text+p);
      p2=strlen(buffer);
      view_getSizeForText(&w,&h,buffer,node->layout.fontsize);
      while ((w+6>MAXBOXWIDTH) && (p>3))
	{
	  /* end the line with ... */
	  buffer[p2-1]='\x0';
	  buffer[p2-2]='.';
	  buffer[p2-3]='.';
	  buffer[p2-4]='.';
	  view_getSizeForText(&w,&h,buffer,node->layout.fontsize);
	  p2--;
	}
      /* adjust the complete text/cut it to calculated size */
      /* node->layout.text[p+p2]='\x0';
      node->layout.text[p+p2-1]=buffer[p2-1];
      node->layout.text[p+p2-2]=buffer[p2-2];
      node->layout.text[p+p2-3]=buffer[p2-3]; */
      node->layout.text[p]='\n';
      node->layout.text[p+1]='\r';
      strcpy(node->layout.text+p+2,buffer);
    }

  /* set nodesize: width is width of text, but broader than MINBOXWIDTH
     height is the size of two lines */
  node->layout.width=((w>w2) ? w : w2) + 15;
  if (node->layout.width<MINBOXWIDTH) 
    node->layout.width=MINBOXWIDTH;
  node->layout.height=2*h+15;

  node->layout.moved = TRUE; /* to recalc links */
}

void layout_resetVisited()
{
  /* resets the visited-flag in all nodes */
  struct TreeInfo *n;
  for (n=CH.root; n; n=n->nextList)
    n->layout.visited=FALSE;
}

void layout_resetMoved()
{
  /* resets the moved-flag in all nodes */
  struct TreeInfo *n;
  for (n=CH.root; n; n=n->nextList)
    n->layout.moved=FALSE;
}

void layout_setLayoutSpace()
{
  /* calculates the maximum space needed to display all nodes */
  struct TreeInfo *n;

  CL.maxw=CL.maxh=0;
  for (n=CH.root; n; n=n->nextList)
    if (n->layout.draw)
      {
	if (n->layout.x+n->layout.width>CL.maxw) 
	  CL.maxw=n->layout.x+n->layout.width;
	if (n->layout.y+n->layout.height>CL.maxh) 
	  CL.maxh=n->layout.y+n->layout.height;
      }
  view_setLayoutSpace(CL.maxw+DISTTOBORDER,CL.maxh+DISTTOBORDER);
}

void layout_findAllNewLinks()
{
  /* calculates the links for between all moved nodes */
  struct TreeInfo *n;

  layout_resetVisited();

  switch(CP.viewtype)
    {
      case leftrightview:
	for (n=CH.root; n; n=n->nextList)
	  if (n->active && (CP.viewshowordered || n->loaded))
	    layout_leftrightFindLinks(n);
	break;

      case topdownview:
	for (n=CH.root; n; n=n->nextList)
	  if (n->active && (CP.viewshowordered || n->loaded))
	    layout_topdownFindLinks(n);
	break;
    }

  /* reset the moved flag */
  layout_resetMoved();
}

void layout_getPath(struct TreeInfo *node, int *path, int i)
{
  /* returns the path from the leaf to the root 
     as a list of link number, so i can determine, if a node is left
     or right of another - see layout_isBefore(..) */

  if (node==CH.root)
    {
      /* root reached */
      *(path+i)=0;
      return;
    }

  if (i>=MAXPATH)
    {
      dbg(DI,DBG_LAYOUT|1,"BIG PROBLEM: increase MAXPATH");
      return;
    }

  *(path+i)=node->nextFather->layout.nr;
  layout_getPath(node->nextFather->tree,path,i+1);
}

int layout_isBefore(int *path1, int *path2)
{
  /* compares two paths in the tree - see layout_getPath(...) 
     and returns 1 if the first lies left from the other, else 0 */

  int i,j;
  for (i=0; *(path1+i); i++);
  for (j=0; *(path2+j); j++);
  while (i && j && (*(path1+i-1)==*(path2+j-1)) )
    i--,j--;

  if (*(path1+i-1)<*(path2+j-1))
    return 1;
  else
    return 0;
}

/*======================================================================================

   T O P D O W N   -   L A Y O U T

======================================================================================*/

int layout_topdownAddNode(struct TreeInfo *node)
{
  /* return 1 if you have to restructure the tree */

  /* called if a new node is to be inserted into the tree
     inserts the node and it's direct children in the gridlines and
     does the layout */

  int newy,j;
  struct NextTreeInfo *l;
  int p1[MAXPATH],p2[MAXPATH];
  struct GridBox *g,*gprev,*newg;

  dbg(DI,DBG_LAYOUT|1,"add node %s active %i loaded %i draw %i",node->url,node->active,node->loaded,node->layout.draw);

  if (node->layout.draw>1) return 0;

  if (node==CH.root)
    {
      /* special case - create grid entries for root */
      node->layout.x=DISTTOBORDER;
      node->layout.y=DISTTOBORDER;
      node->layout.draw++;  /* =1 */
      node->layout.moved=1;
      CL.grid[0]=(struct GridBox *)malloc(sizeof(struct GridBox));
      CL.grid[0]->parent=NULL;
      CL.grid[0]->pos=node->layout.x;
      CL.grid[0]->size=node->layout.width + MINHDIST;
      CL.grid[0]->nextGrid=NULL;
    }

  if (node->loaded && node->layout.draw==0)
    {
      layout_checkLinks(node->nextFather->tree);
      layout_checkLinks(node);
      for (g=CL.grid[node->layout.y/GRIDDY]; g->parent!=node->nextFather->tree; g=g->nextGrid);
      node->layout.x=g->pos+g->size;
      node->layout.moved=1;
      node->layout.draw++;
      g->size+=node->layout.width+MINHDIST;
      layout_topdownShiftGridBoxes(g);
    }

  if (node->layout.directlinksout==0)
    {
      if (node->loaded) 
	return 1;
      else
	return 0;
    }

  newy=node->layout.y + node->layout.height + 2*GRIDDY + 
       node->layout.directlinksout*GRIDDY/15;

  /* adjust to gridline */
  newy=(int)(newy/GRIDDY)*GRIDDY;

  /* get path for the first direct child */
  layout_getPath(node,p1,0);
  
  /* now look in gridline until the new child is right of the already entered
     and insert all children at that position */
  j=newy/GRIDDY;
  gprev=NULL;
  for (g=CL.grid[j]; g; g=(gprev=g)->nextGrid )
    if (g->parent)
      {
	layout_getPath(g->parent,p2,0);
	if (layout_isBefore(p1,p2)) break;
      }
  
  newg=(struct GridBox *)malloc(sizeof(struct GridBox));
  newg->parent=node;
  newg->size=0;

  if (gprev) 
    {
      /* it exists a gridbox before the actual one */
      newg->nextGrid=gprev->nextGrid;
      gprev->nextGrid=newg;
      newg->pos=gprev->pos + gprev->size;
    } 
  else
    {
      /* there's no gridbox more left than the actual one */
      newg->nextGrid=CL.grid[j];
      CL.grid[j]=newg;
      newg->pos=DISTTOBORDER;
    }
  

  if (node->layout.visiblelinksout)
    {
      for (l=node->nextTree; l; l=l->next)
	if (l->layout.type&DIRECT)
	  {
	    l->tree->layout.y=newy;
	    l->tree->layout.x=newg->pos+newg->size;
	    if (l->layout.type&USED)
	      {
		l->tree->layout.draw++; /* =1 */
		l->tree->layout.moved=1;
		newg->size+=l->tree->layout.width + MINHDIST;
	      }
	  }
    } else {
      for (l=node->nextTree; l; l=l->next)
	if (l->layout.type&DIRECT)
	  {
	    l->tree->layout.y=newy;
	    l->tree->layout.x=newg->pos;
	  }
    }

  layout_topdownShiftGridBoxes(newg);
  return 1;
}

void layout_topdownShiftGridBoxes(struct GridBox *g)
{
  struct NextTreeInfo *l;
  int shift;

  if (g->nextGrid)
    {
      shift=(g->pos+g->size) - g->nextGrid->pos;
      if (shift<=0)
	/* nothing to shift */
	return;
    }
  else
    /* nothing to shift */
    return;

  for (g=g->nextGrid; g; g=g->nextGrid)
    {
      for (l=g->parent->nextTree; l; l=l->next)
	if ((l->layout.type&(DIRECT|USED))==(DIRECT|USED))
	  {
	    l->tree->layout.x+=shift;
	    l->tree->layout.moved=1;
	  }
      g->pos+=shift;
    }
}


void layout_topdownShiftGridBoxesBy(struct GridBox *g, int pos)
{
  struct NextTreeInfo *l;
  int b=0,x;
  for (;g && !b;g=g->nextGrid)
    {
      if (g->pos<pos)
	{
	  x=pos;
	  g->pos=pos;
	  for (l=g->parent->nextTree; l; l=l->next)
	    if ((l->layout.type&(DIRECT|USED))==(DIRECT|USED))
	      {
		l->tree->layout.x=x;
		x+=l->tree->layout.width+MINHDIST;
		l->tree->layout.moved=1;
	      }
	  g->size=x-pos;
	  pos=x;
	} else
	  b=1;
    }
}

void layout_topdownShrinkTree()
{
  int gc,x,size;
  struct GridBox *g;
  struct NextTreeInfo *l;

  for (gc=1; gc<MAXGRIDLINES; gc++)
    {
      x=DISTTOBORDER;
      for (g=CL.grid[gc]; g; g = g->nextGrid)
	{
	  g->pos=x;
	  size=0;
	  for (l=g->parent->nextTree; l; l=l->next)
	  if ((l->layout.type&(DIRECT|USED))==(DIRECT|USED))
	      {
		l->tree->layout.x=x;
		l->tree->layout.moved=1;
		x+=l->tree->layout.width+MINHDIST;
		size+=l->tree->layout.width+MINHDIST;
	      }
	  g->size=size;
	}
    }

  CL.grid[0]->pos=DISTTOBORDER;
  CH.root->layout.x=DISTTOBORDER;
}


void layout_topdownRestructureTree()
{
  int gc,gc2;
  struct GridBox *g,*g2;
  int midx,dist;
  struct NextTreeInfo *l;
  int restart;
  int dlo,x;

  restart=1;
  while (restart)
    {
      restart=0;
  for (gc=MAXGRIDLINES-1; !restart && gc>=0; gc--)
    for (g=CL.grid[gc]; !restart && g; g=g->nextGrid)
      if (g->parent && g->parent->layout.visiblelinksout>0)
	{
	  midx=g->pos+((g->size==0) ? 0 : g->size - MINHDIST)/2;
	  dist = midx - (g->parent->layout.x + g->parent->layout.width/2);
	  if (dist > 0) 
	    {
	      /* shift parent-node */
	      gc2=g->parent->layout.y/GRIDDY;
	      if (gc2==0)
		{
		  /* alright, shift the root node */
		  CH.root->layout.x+=dist;
		  CH.root->layout.moved=1;
		  CL.grid[0]->pos+=dist;
		}
	      else
		{
		  /* find gridbox with parent entry in it */
		  for ( g2 = CL.grid[gc2]; g2->parent!=g->parent->nextFather->tree; g2=g2->nextGrid);

		  /* count number of visible direct links out of brothers before the (parent-)node to
		     shift */
		  dlo=0;
		  for ( l = g2->parent->nextTree; l->tree!=g->parent; l = l->next)
		    if ((l->layout.type&(DIRECT|USED))==(DIRECT|USED))
		      dlo+=l->tree->layout.directlinksout;

		  if (dlo==0)
		    {
		      /* no children are before the parentnode,
			 that means shift all nodes to the right */
		      for ( l = g2->parent->nextTree; l->tree!=g->parent; l=l->next)
			if ((l->layout.type&(DIRECT|USED))==(DIRECT|USED))
			  {
			    l->tree->layout.x+=dist;
			    l->tree->layout.moved=1;
			  }
		      l->tree->layout.x+=dist;
		      l->tree->layout.moved=1;

		      /* now adjust all nodes right of the parent node */
		      x = l->tree->layout.x + l->tree->layout.width + MINHDIST;
		      for ( l = l->next; l; l = l->next )
			if ((l->layout.type&(DIRECT|USED))==(DIRECT|USED))
			  {
			    /* if (l->tree->layout.x>=x)
			      {
				g2->size-=dist;
				break;
			      } */
			    /* l->tree->layout.x+=dist; */
			    l->tree->layout.x=x;
			    l->tree->layout.moved=1;
			    x+=l->tree->layout.width + MINHDIST;
			  }
		      g2->pos+=dist;
		      g2->size=x-g2->pos;
		    }
		  else
		    {
		      l->tree->layout.x+=dist;
		      l->tree->layout.moved=1;
		      x = l->tree->layout.x + l->tree->layout.width + MINHDIST;
		      for ( l = l->next; l; l = l->next )
			if ((l->layout.type&(DIRECT|USED))==(DIRECT|USED))
			  {
			    /* if (l->tree->layout.x>=x)
			      {
				g2->size-=dist;
				break;
			      }
			      */
			    l->tree->layout.x=x;
			    l->tree->layout.moved=1;
			    x+=l->tree->layout.width + MINHDIST;
			  }
		      g2->size=x-g2->pos;
		    }
		  layout_topdownShiftGridBoxesBy( g2->nextGrid, g2->pos+g2->size );
		}
	      restart=1;
	    }
	  else if (dist < 0)
	    {
	      /* shift children */
	      layout_topdownShiftGridBoxesBy( g, g->pos - dist );
	      restart=1;
	    }
	  /* else everythings ok */
	}
    }
}

void layout_topdownFindLinks(struct TreeInfo *node)
{
  struct NextTreeInfo *l;
  int x1,y1,x2,y2,ym,h;

  for (l=node->nextTree;l;l=l->next)
    /* link is in use - not a duplicate or invalid one */
    if ((l->layout.type&USED) && (l->tree->layout.moved || node->layout.moved))
      {
	if ((l->layout.type&(DIRECT|USED))==(DIRECT|USED))
	  /* direct link - goes from top to bottom at any rate */
	  {
	    x1=node->layout.x+node->layout.width/2;
	    y1=node->layout.y+node->layout.height;
	    x2=l->tree->layout.x+l->tree->layout.width/2;
	    y2=l->tree->layout.y;

	    l->layout.x[0] = x1;
	    l->layout.y[0] = y1;

	    l->layout.x[1] = x1;
	    l->layout.y[1] = y2 - MINVDIST;

	    l->layout.x[2] = x2;
	    l->layout.y[2] = y2 - MINVDIST;

	    l->layout.x[3] = x2;
	    l->layout.y[3] = y2;
	  }
	else
	  /* any other link*/
	  {
	    x1=node->layout.x+node->layout.width/2;
	    y1=node->layout.y+node->layout.height;
	    x2=l->tree->layout.x+l->tree->layout.width/2;
	    y2=l->tree->layout.y+l->tree->layout.height;

	    h=0;
	    ym=y1;
	    if (y2 < y1 - 3*MINVDIST)
	      ym=node->layout.y,h++;
	    if (y1 < y2 - 3*MINVDIST)
	      y2=l->tree->layout.y,h++;
	    y1=ym;

	    if (h==0) /* both links leave at the bottom side of the box */
	      ym=max(y1,y2)+1.5*MINVDIST;
	    else
	      ym=(y2>y1) ? y2-1.5*MINHDIST : y2+1.5*MINHDIST;

	    l->layout.x[0] = x1;
	    l->layout.y[0] = y1;

	    l->layout.x[1] = x1;
	    l->layout.y[1] = ym;

	    l->layout.x[2] = x2;
	    l->layout.y[2] = ym;

	    l->layout.x[3] = x2;
	    l->layout.y[3] = y2;
	  }
      }
}

/*======================================================================================

   L E F T - R I G H T   -   L A Y O U T

======================================================================================*/

int layout_leftrightAddNode(struct TreeInfo *node)
{
  /* return 1 if you have to restructure the tree */

  /* called if a new node is to be inserted into the tree
     inserts the node and it's direct children in the gridlines and
     does the layout */

  int newx,j;
  struct NextTreeInfo *l;
  int p1[MAXPATH],p2[MAXPATH];
  struct GridBox *g,*gprev,*newg;

  dbg(DI,DBG_LAYOUT|1,"add node %s active %i loaded %i",node->url,node->active,node->loaded);

  if (node->layout.draw>1) return 0;

  if (node==CH.root)
    {
      /* special case - create grid entries for root */
      node->layout.x=DISTTOBORDER;
      node->layout.y=DISTTOBORDER;
      node->layout.draw++;  /* =1 */
      node->layout.moved=1;
      CL.grid[0]=(struct GridBox *)malloc(sizeof(struct GridBox));
      CL.grid[0]->parent=NULL;
      CL.grid[0]->pos=node->layout.y;
      CL.grid[0]->size=node->layout.height + MINVDIST;
      CL.grid[0]->nextGrid=NULL;
    }

  if (node->loaded && node->layout.draw==0)
    {
      layout_checkLinks(node->nextFather->tree);
      layout_checkLinks(node);
      for (g=CL.grid[node->layout.x/GRIDDX]; g->parent!=node->nextFather->tree; g=g->nextGrid);
      node->layout.y=g->pos+g->size;
      node->layout.moved=1;
      node->layout.draw++;
      g->size+=node->layout.height+MINVDIST;
      layout_leftrightShiftGridBoxes(g);
    }

  if (node->layout.directlinksout==0)
    {
      if (node->loaded) 
	return 1;
      else
	return 0;
    }

  newx=node->layout.x + node->layout.width + 2*GRIDDX + 
       node->layout.directlinksout*GRIDDX/15;

  /* adjust to gridline */
  newx=(int)(newx/GRIDDX)*GRIDDX;

  /* get path for the first direct child */
  layout_getPath(node,p1,0);
  
  /* now look in gridline until the new child is right of the already entered
     and insert all children at that position */
  j=newx/GRIDDX;
  gprev=NULL;
  for (g=CL.grid[j]; g; g=(gprev=g)->nextGrid )
    if (g->parent)
      {
	layout_getPath(g->parent,p2,0);
	if (layout_isBefore(p1,p2)) break;
      }
  
  newg=(struct GridBox *)malloc(sizeof(struct GridBox));
  newg->parent=node;
  newg->size=0;

  if (gprev) 
    {
      /* it exists a gridbox before the actual one */
      newg->nextGrid=gprev->nextGrid;
      gprev->nextGrid=newg;
      newg->pos=gprev->pos + gprev->size;
    } 
  else
    {
      /* there's no gridbox more left than the actual one */
      newg->nextGrid=CL.grid[j];
      CL.grid[j]=newg;
      newg->pos=DISTTOBORDER;
    }
  

  if (node->layout.visiblelinksout)
    {
      for (l=node->nextTree; l; l=l->next)
	if (l->layout.type&DIRECT)
	  {
	    l->tree->layout.x=newx;
	    l->tree->layout.y=newg->pos+newg->size;
	    if (l->layout.type&USED)
	      {
		l->tree->layout.draw++; /* =1 */
		l->tree->layout.moved=1;
		newg->size+=l->tree->layout.height + MINVDIST;
	      }
	  }
    } else {
      for (l=node->nextTree; l; l=l->next)
	if (l->layout.type&DIRECT)
	  {
	    l->tree->layout.x=newx;
	    l->tree->layout.y=newg->pos;
	  }
    }

  layout_leftrightShiftGridBoxes(newg);
  return 1;
}

void layout_leftrightShiftGridBoxes(struct GridBox *g)
{
  struct NextTreeInfo *l;
  int shift;

  if (g->nextGrid)
    {
      shift=(g->pos+g->size) - g->nextGrid->pos;
      if (shift<=0)
	/* nothing to shift */
	return;
    }
  else
    /* nothing to shift */
    return;

  for (g=g->nextGrid; g; g=g->nextGrid)
    {
      for (l=g->parent->nextTree; l; l=l->next)
	if ((l->layout.type&(DIRECT|USED))==(DIRECT|USED))
	  {
	    l->tree->layout.y+=shift;
	    l->tree->layout.moved=1;
	  }
      g->pos+=shift;
    }
}


void layout_leftrightShiftGridBoxesBy(struct GridBox *g, int pos)
{
  struct NextTreeInfo *l;
  int b=0,y;
  for (;g && !b;g=g->nextGrid)
    {
      if (g->pos<pos)
	{
	  y=pos;
	  g->pos=pos;
	  for (l=g->parent->nextTree; l; l=l->next)
	    if ((l->layout.type&(DIRECT|USED))==(DIRECT|USED))
	      {
		l->tree->layout.y=y;
		y+=l->tree->layout.height+MINVDIST;
		l->tree->layout.moved=1;
	      }
	  g->size=y-pos;
	  pos=y;
	} else
	  b=1;
    }
}

void layout_leftrightShrinkTree()
{
  int gc,y,size;
  struct GridBox *g;
  struct NextTreeInfo *l;

  for (gc=1; gc<MAXGRIDLINES; gc++)
    {
      y=DISTTOBORDER;
      for (g=CL.grid[gc]; g; g = g->nextGrid)
	{
	  g->pos=y;
	  size=0;
	  for (l=g->parent->nextTree; l; l=l->next)
	    if ((l->layout.type&(DIRECT|USED))==(DIRECT|USED))
	      {
		l->tree->layout.y=y;
		l->tree->layout.moved=1;
		y+=l->tree->layout.height+MINVDIST;
		size+=l->tree->layout.height+MINVDIST;
	      }
	  g->size=size;
	}
    }

  CL.grid[0]->pos=DISTTOBORDER;
  CH.root->layout.y=DISTTOBORDER;
}


void layout_leftrightRestructureTree()
{
  int gc,gc2;
  struct GridBox *g,*g2;
  int midy,dist;
  struct NextTreeInfo *l;
  int restart;
  int dlo,y;

  restart=1;
  while (restart)
    {
      restart=0;
  for (gc=MAXGRIDLINES-1; !restart && gc>=0; gc--)
    for (g=CL.grid[gc]; !restart && g; g=g->nextGrid)
      if (g->parent && g->parent->layout.visiblelinksout>0)
	{
	  midy=g->pos+((g->size==0) ? 0 : g->size - MINHDIST)/2;
	  dist = midy - (g->parent->layout.y + g->parent->layout.height/2);
	  if (dist > 0) 
	    {
	      /* shift parent-node */
	      gc2=g->parent->layout.x/GRIDDX;
	      if (gc2==0)
		{
		  /* alright, shift the root node */
		  CH.root->layout.y+=dist;
		  CH.root->layout.moved=1;
		  CL.grid[0]->pos+=dist;
		}
	      else
		{
		  /* find gridbox with parent entry in it */
		  for ( g2 = CL.grid[gc2]; g2->parent!=g->parent->nextFather->tree; g2=g2->nextGrid);

		  /* count number of visible direct links out of brothers before the (parent-)node to
		     shift */
		  dlo=0;
		  for ( l = g2->parent->nextTree; l->tree!=g->parent; l = l->next)
		    if ((l->layout.type&(DIRECT|USED))==(DIRECT|USED))
		      dlo+=l->tree->layout.directlinksout;

		  if (dlo==0)
		    {
		      /* no children are before the parentnode,
			 that means shift all nodes to the right */
		      for ( l = g2->parent->nextTree; l->tree!=g->parent; l=l->next)
			if ((l->layout.type&(DIRECT|USED))==(DIRECT|USED))
			  {
			    l->tree->layout.y+=dist;
			    l->tree->layout.moved=1;
			  }
		      l->tree->layout.y+=dist;
		      l->tree->layout.moved=1;

		      /* now adjust all nodes right of the parent node */
		      y = l->tree->layout.y + l->tree->layout.height + MINVDIST;
		      for ( l = l->next; l; l = l->next )
			if ((l->layout.type&(DIRECT|USED))==(DIRECT|USED))
			  {
			    l->tree->layout.y=y;
			    l->tree->layout.moved=1;
			    y+=l->tree->layout.height + MINVDIST;
			  }
		      g2->pos+=dist;
		      g2->size=y-g2->pos;
		    }
		  else
		    {
		      l->tree->layout.y+=dist;
		      l->tree->layout.moved=1;
		      y = l->tree->layout.y + l->tree->layout.height + MINVDIST;
		      for ( l = l->next; l; l = l->next )
			if ((l->layout.type&(DIRECT|USED))==(DIRECT|USED))
			  {
			    l->tree->layout.y=y;
			    l->tree->layout.moved=1;
			    y+=l->tree->layout.height + MINVDIST;
			  }
		      g2->size=y-g2->pos;
		    }
		  layout_leftrightShiftGridBoxesBy( g2->nextGrid, g2->pos+g2->size );
		}
	      restart=1;
	    }
	  else if (dist < 0)
	    {
	      /* shift children */
	      layout_leftrightShiftGridBoxesBy( g, g->pos - dist );
	      restart=1;
	    }
	  /* else everythings ok */
	}
    }
}

void layout_leftrightFindLinks(struct TreeInfo *node)
{
  struct NextTreeInfo *l;
  int x1,y1,x2,y2,xm,w;

  for (l=node->nextTree;l;l=l->next)
    /* link is in use - not a duplicate or invalid one */
    if ((l->layout.type&USED) && (l->tree->layout.moved || node->layout.moved))
      {
	if ((l->layout.type&(DIRECT|USED))==(DIRECT|USED))
	  /* direct link - goes from top to bottom at any rate */
	  {
	    x1=node->layout.x+node->layout.width;
	    y1=node->layout.y+node->layout.height/2;
	    x2=l->tree->layout.x;
	    y2=l->tree->layout.y+l->tree->layout.height/2;

	    l->layout.x[0] = x1;
	    l->layout.y[0] = y1;

	    l->layout.x[1] = x2 - MINHDIST;
	    l->layout.y[1] = y1;

	    l->layout.x[2] = x2 - MINHDIST;
	    l->layout.y[2] = y2;

	    l->layout.x[3] = x2;
	    l->layout.y[3] = y2;
	  }
	else
	  /* any other link*/
	  {
	    x1=node->layout.x+node->layout.width;
	    y1=node->layout.y+node->layout.height/2;
	    x2=l->tree->layout.x+l->tree->layout.width;
	    y2=l->tree->layout.y+l->tree->layout.height/2;

	    w=0;
	    xm=node->layout.x+MAXBOXWIDTH;
	    if (x2 < x1 - 3*MINHDIST)
	      xm=node->layout.x,w++;
	    if (x1 < x2 - 3*MINHDIST)
	      x2=l->tree->layout.x,w++;
	    x1=xm;

	    if (w==0) /* both links leave at the bottom side of the box */
	      /* xm=max(x1,x2)+1.5*MINHDIST; */
	      xm=node->layout.x+MAXBOXWIDTH+1.5*MINHDIST;
	    else
	      /* xm=(x2>x1) ? x2-1.5*MINVDIST : x2+1.5*MINVDIST; */
	      xm=l->tree->layout.x+MAXBOXWIDTH + ((x2>x1) ? -1.5*MINHDIST : 1.5*MINHDIST);

	    l->layout.x[0] = x1;
	    l->layout.y[0] = y1;

	    l->layout.x[1] = xm;
	    l->layout.y[1] = y1;

	    l->layout.x[2] = xm;
	    l->layout.y[2] = y2;

	    l->layout.x[3] = x2;
	    l->layout.y[3] = y2;
	  }
      }
}


/*======================================================================================

   I N I T / F R E E   -   F U N C T I O N S

======================================================================================*/
void layout_initTreeInfo(struct TreeInfo *ti)
{
  struct layout_TreeInfo *lti=&(ti->layout);
  lti->x=lti->y=lti->width=lti->height=0;
  lti->fontsize=CP.viewfontsizedefault;
  lti->visited=0;
  lti->draw=0;
  lti->collapsed=0;
  lti->moved=0;
  lti->linksin=0;
  lti->linksout=0;
  lti->directlinksout=0;
  lti->visiblelinksout=0;
  lti->text[0]='\x0';
  /*  fprintf(stderr,"%p init\n",ti); */
}

void layout_freeTreeInfo(struct TreeInfo *ti)
{
  dbg(DI,DBG_LAYOUT|2,"free %p %s\n",ti,ti->url);
}

void layout_initNextTreeInfo(struct NextTreeInfo *nti)
{
  nti->layout.type=UNUSED;
  nti->layout.nr=0;
}

void layout_freeNextTreeInfo(struct NextTreeInfo *nti)
{
}

void layout_initGrid()
{
  /* initializes the grid */
  int i;
  for (i=0; i<MAXGRIDLINES; i++)
    CL.grid[i]=NULL;
}

void layout_resetGrid()
{
  /* reset grid: free grid boxes and set pointers to NULL */
  int i;
  struct GridBox *a,*b;
  for (i=0; i<MAXGRIDLINES; i++)
    {
      a=CL.grid[i];
      while (a)
	{
	  b=a->nextGrid;
	  free(a);
	  a=b;
	}
      CL.grid[i]=NULL;
    }
}

void layout_walkRecursive(struct TreeInfo *n, int collapsed)
{
  struct NextTreeInfo *l;
  for (l=n->nextTree; l; l = l->next)
    if (l->layout.type & DIRECT)
      {
	l->tree->layout.collapsed=collapsed;
	layout_walkRecursive(l->tree,collapsed);
      }
}

/* Collapses a subtree */
void layout_collapseNode(struct TreeInfo *n)
{
  layout_walkRecursive(n,1);
  layout_recalcAll();
}

/* Expandes a subtree */
void layout_expandNode(struct TreeInfo *n)
{
  layout_walkRecursive(n,0);
  layout_recalcAll();
}

/*=========== DEBUG-Functions ============*/
/* a debug function */
void pGrid(int i)
{
  int p[MAXPATH];
  struct GridBox *g;
  for (g=CL.grid[i]; g; g=g->nextGrid)
    {
      layout_getPath(g->parent,p,0);
      fprintf(stderr,"%p/%i/%i ",g->parent,g->pos,g->size);
      for (i=0; *(p+i); i++)
	fprintf(stderr,"%i,",*(p+i));
      fprintf(stderr,"\n");
    }
  fprintf(stderr,"\n");
}

void pNode(struct TreeInfo *node)
{
  struct NextTreeInfo *nti;
  struct NextTreeInfo *fti;
  int fatherok=FALSE;

  fprintf(stderr,"%s active %i loaded %i depth %in",node->url,node->active,node->loaded,
	  node->depth);

  for (nti=node->nextTree; nti; nti=nti->next)
    {
      fprintf(stderr,"child %s %s%s ",nti->tree->url,(nti->layout.type&DIRECT) ? "D":" ",
	      (nti->layout.type&USED) ? "U":" ");

      fatherok=FALSE;
      if (nti)
	for (fti=nti->tree->nextFather; fti; fti=fti->next)
	  if (fti->tree==node)
	    fatherok=TRUE;
      
      fprintf(stderr,"%s\n",(fatherok) ? "Father OK":"FATHER MISSING");
    }


}
